<?php

namespace Drupal\advanced_file_destination\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Plugin\Field\FieldWidget\FileWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Render\ElementInfoManagerInterface;

/**
 * Plugin implementation of the 'advanced_file_destination' widget.
 *
 * @FieldWidget(
 *   id = "advanced_file_destination",
 *   label = @Translation("File with destination selection"),
 *   field_types = {
 *     "file",
 *     "image"
 *   }
 * )
 */
class AdvancedFileDestinationWidget extends FileWidget {

  /**
   * The advanced file destination manager.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * Constructs an AdvancedFileDestinationWidget object.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, ElementInfoManagerInterface $element_info, AdvancedFileDestinationManager $destination_manager) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings, $element_info);
    $this->destinationManager = $destination_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
          $plugin_id,
          $plugin_definition,
          $configuration['field_definition'],
          $configuration['settings'],
          $configuration['third_party_settings'],
          $container->get('element_info'),
          $container->get('advanced_file_destination.manager')
      );
  }

  /**
   * Ajax callback for directory selection.
   */
  public function updateDirectoryCallback(array $form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement();
    $parents = array_slice($trigger['#array_parents'], 0, -1);

    // Get the directory wrapper element.
    $wrapper_element = NestedArray::getValue($form, $parents);

    // Get the selected directory.
    $directory = $trigger['#value'];

    // Special case handling for public directories.
    if ($directory == 'public://' || $directory == 'public:///'
          || $directory == 'public://public:/' || $directory == 'public://public://'
      ) {
      $directory = 'public://';
    }

    if ($directory) {
      // Get the manager service with dependency injection if available.
      $directory_service = $this->destinationManager ?? \Drupal::service('advanced_file_destination.manager');
      // Update service.
      $directory_service->setDestinationDirectory($form_state->getValue('directory'), $form_state->getValue('adf_instance_id'));
      $form_state->setValue('directory', $directory);
      $form_state->setValue(['advanced_file_destination', 'directory'], $directory);
      // Get the field element.
      $field_parents = array_slice($parents, 0, -1);
      $field_element = NestedArray::getValue($form, $field_parents);

      // Set the selected directory in the hidden field.
      if (isset($field_element['selected_directory'])) {
        $field_element['selected_directory']['#value'] = $directory;
      }

      // Update upload location.
      if (isset($field_element['#upload_location'])) {
        $field_element['#upload_location'] = $directory;
      }

      if (isset($field_element['upload']['#upload_location'])) {
        $field_element['upload']['#upload_location'] = $directory;
      }

      // Store in form state.
      $field_name = $this->fieldDefinition->getName();
      $delta = $field_element['#delta'];
      $form_state->set(['file_destination', $field_name, $delta], $directory);

      // Set the static value that Drupal uses.
      $settings = &drupal_static('file_upload_defaults', []);
      $settings['upload_location'] = $directory;
    }

    // Return only the wrapper element to prevent duplicating the entire field.
    return $wrapper_element;
  }

  /**
   * Validates the selected directory.
   */
  public function validateDirectory($element, FormStateInterface $form_state, $form) {
    $directory = $element['#value'];
    if ($directory) {

      // Fix the root directory cases only, preserving subdirectories.
      if ($directory === 'public:///' || $directory === 'public://public:/' || $directory === 'public:/' || $directory === 'public:/ ' || $directory === 'public://public://') {
        $directory = 'public://';
      }

      // Only update the form value if we modified the directory.
      if ($directory !== $element['#value']) {
        $form_state->setValueForElement($element, $directory);
      }

      $file_system = \Drupal::service('file_system');
      if (!$file_system->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
        $form_state->setError($element, $this->t('Selected directory is not writable.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validate($element, FormStateInterface $form_state, $form) {
    // Make sure the upload destination is properly set before validation.
    $upload_location = $element['selected_directory']['#value'] ?? NULL;
    if ($upload_location) {
      if (isset($element['#upload_location'])) {
        $element['#upload_location'] = $upload_location;
      }
      if (isset($element['upload']['#upload_location'])) {
        $element['upload']['#upload_location'] = $upload_location;
      }

      // Set the static value that Drupal uses.
      $settings = &drupal_static('file_upload_defaults', []);
      $settings['upload_location'] = $upload_location;
    }

    return parent::validate($element, $form_state, $form);
  }

  /**
   * Get the current directory for this field item.
   */
  protected function getCurrentDirectory(FieldItemListInterface $items, $delta) {
    // For existing files, get their current directory.
    if (!$items->isEmpty() && isset($items[$delta]->entity)) {
      $file = $items[$delta]->entity;
      if ($file) {
        $uri = $file->getFileUri();
        if ($uri) {
          return dirname($uri);
        }
      }
    }

    // Get directory from widget state.
    $form_state = $items->getEntity()->form_state;
    if ($form_state) {
      $widget_state = static::getWidgetState($form_state->getCompleteForm()['#parents'], $this->fieldDefinition->getName(), $form_state);
      if (isset($widget_state[$delta]['directory'])) {
        return $widget_state[$delta]['directory'];
      }
    }
  }

  /**
   * Validate callback for directory selection.
   */
  public static function validateDirectorySelection($element, FormStateInterface $form_state, $form) {
    $file_system = \Drupal::service('file_system');
    $selected_directory = $element['directory']['#value'];

    if ($selected_directory) {
      if (!$file_system->prepareDirectory($selected_directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
        $form_state->setError($element['directory'], \Drupal::translation()->translate('Selected directory is not writable.'));
      }
    }
  }


  /**
   * {@inheritdoc}
   */
  public function extractFormValues(FieldItemListInterface $items, array $form, FormStateInterface $form_state) {
    $field_name = $this->fieldDefinition->getName();
    $widget_state = static::getWidgetState($form['#parents'], $field_name, $form_state);

    // Store directory changes for each delta.
    foreach ($items as $delta => $item) {
      if (isset($widget_state[$delta]['directory']) && isset($item->entity)) {
        $form_state->set(
              [
                'advanced_file_destination',
                $field_name,
                $delta,
              ], [
                'file_id' => $item->entity->id(),
                'new_directory' => $widget_state[$delta]['directory'],
              ]
          );
      }
    }

    return parent::extractFormValues($items, $form, $form_state);
  }

}
