<?php

namespace Drupal\advanced_file_destination\Controller;

use Drupal\advanced_file_destination\Form\NewDirectoryModalForm;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Component\Utility\Html;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenDialogCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\State\StateInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for directory creation modal.
 */
class DirectoryModalController extends ControllerBase {

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The CSRF token generator.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator
   */
  protected $csrfToken;

  /**
   * The destination manager service.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The session service.
   *
   * @var \Symfony\Component\HttpFoundation\Session\SessionInterface
   */
  protected $session;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The current user account.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a new DirectoryModalController.
   *
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token
   *   The CSRF token generator.
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $destination_manager
   *   The destination manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Symfony\Component\HttpFoundation\Session\SessionInterface $session
   *   The session service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   */
  public function __construct(
    FormBuilderInterface $form_builder,
    CsrfTokenGenerator $csrf_token,
    AdvancedFileDestinationManager $destination_manager,
    ConfigFactoryInterface $config_factory,
    RendererInterface $renderer,
    StateInterface $state,
    SessionInterface $session,
    LoggerInterface $logger,
    RequestStack $request_stack,
    AccountProxyInterface $current_user
  ) {
    $this->formBuilder = $form_builder;
    $this->csrfToken = $csrf_token;
    $this->destinationManager = $destination_manager;
    $this->configFactory = $config_factory;
    $this->renderer = $renderer;
    $this->state = $state;
    $this->session = $session;
    $this->logger = $logger;
    $this->requestStack = $request_stack;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('form_builder'),
          $container->get('csrf_token'),
          $container->get('advanced_file_destination.manager'),
          $container->get('config.factory'),
          $container->get('renderer'),
          $container->get('state'),
          $container->get('session'),
          $container->get('logger.factory')->get('advanced_file_destination'),
          $container->get('request_stack'),
          $container->get('current_user')
      );
  }

  /**
   * Get the default directory from configuration.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected function getDefaultDirectory() {
    return $this->configFactory->get('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * Displays the modal form.
   */
  public function modal(Request $request, $adf_instance_id = NULL, $parent_directory = NULL) {

    if (!$parent_directory || empty($parent_directory) || !$adf_instance_id || empty($adf_instance_id)) {
      if (!$adf_instance_id || empty($adf_instance_id)) {
        // If no instance ID is provided, use the one from the request.
        $adf_instance_id = $request->get('adf_instance_id');
      }
      if (!$parent_directory || empty($parent_directory)) {
        // If no parent directory is provided, use the one from the request.
        $parent_directory = $request->get('parent_directory');
      }
      if (!$parent_directory || empty($parent_directory)) {
        // If no parent directory is provided, use the directory from the request.
        $parent_directory = Html::escape($this->requestStack->getCurrentRequest()->get('parent_directory'));
      }
      if (!$adf_instance_id || empty($adf_instance_id)) {
        // If no instance ID is provided, use the one from the request.
        $adf_instance_id = Html::escape($this->requestStack->getCurrentRequest()->get('adf_instance_id'));
      }
    }

    $response = new AjaxResponse();

    // Build the form.
    $modal_form = NewDirectoryModalForm::class;
    $form = $this->formBuilder->getForm($modal_form, $parent_directory, $adf_instance_id);
    if ($request->isMethod('POST') && $request->get('form_id') === 'new_directory_modal_form' && $request->get('_triggering_element_name') === 'save-directory') {
      // Handle form submission.
      return $response;
    }
    // Render the form.
    $options = [
      'width' => 'auto',
      'closeOnEscape' => TRUE,
      'title' => $this->t('Create a new directory'),
      'dialogClass' => 'advanced-file-destination-modal',
      'modal' => TRUE,
    ];

    // Add the form to the response.
    $response->addCommand(
      new OpenDialogCommand(
        '#new-directory-modal-form-wrapper',
        $this->t('Create a new directory'),
        $form,
        $options
      )
    );

    return $response;
  }

  /**
   * Media modal ajax callback.
   *
   * The ajax response.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The ajax response.
   */
  public function mediaModal() {

    // Get the form ID.
    $new_response = new AjaxResponse();
    $form_id = Html::escape($this->requestStack->getCurrentRequest()->get('adf_instance_id') ?
    $this->requestStack->getCurrentRequest()->get('adf_instance_id') : $this->requestStack->getCurrentRequest()->query->get('adf_instance_id'));
    $parent_directory = Html::escape($this->requestStack->getCurrentRequest()->get('parent_directory') ?
    $this->requestStack->getCurrentRequest()->get('parent_directory') : $this->requestStack->getCurrentRequest()->query->get('parent_directory'));
    $parent_directory = $this->destinationManager->normalizeDirectoryPath($parent_directory);

    // Set the directory.
    $current_user = $this->currentUser->id();
    $this->state->set('advanced_file_destination.directory.' . $current_user . '.' . $form_id, $parent_directory);

    return $new_response;
  }

}
