<?php

namespace Drupal\advanced_file_destination\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Component\Utility\Html;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Psr\Log\LoggerInterface;

/**
 * Controller for managing directory state.
 */
class DirectoryStateController extends ControllerBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The advanced file destination manager service.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The stream wrapper manager service.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface
   */
  protected $streamWrapperManager;

  /**
   * The current user account.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a new DirectoryStateController.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $destination_manager
   *   The destination manager service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface $stream_wrapper_manager
   *   The stream wrapper manager service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user account.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    AdvancedFileDestinationManager $destination_manager,
    LoggerInterface $logger,
    FileSystemInterface $file_system,
    StreamWrapperManagerInterface $stream_wrapper_manager,
    AccountProxyInterface $current_user
  ) {
    $this->configFactory = $config_factory;
    $this->destinationManager = $destination_manager;
    $this->logger = $logger;
    $this->fileSystem = $file_system;
    $this->streamWrapperManager = $stream_wrapper_manager;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('advanced_file_destination.manager'),
      $container->get('logger.factory')->get('advanced_file_destination'),
      $container->get('file_system'),
      $container->get('stream_wrapper_manager'),
      $container->get('current_user')
    );
  }

  /**
   * Gets the default directory from configuration.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected function getDefaultDirectory() {
    return $this->configFactory->get('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * Updates directory selection state.
   */
  public function updateState(Request $request) {

    $adf_instance_id = (string) Html::escape($request->get('adf_instance_id'));
    $directory = (string) Html::escape($request->get('directory'));
    $parent_directory = (string) Html::escape($request->get('parent_directory'));

    if($current_user = $this->currentUser->getAccount()) {
      if (
        !$current_user->hasPermission('administer advanced file destination') &&
        !$current_user->hasPermission('create advanced file destination') &&
        !$current_user->hasPermission('access advanced file destination')) {
        return new JsonResponse(
          [
            'status' => 'error',
            'message' => $this->t('Access denied. You do not have permission to execute this action.'),
            'error' => 'Access denied. You do not have permission to execute this action.',
          ],
          403
        );
      }
    }

    if (empty($adf_instance_id) || empty($directory)) {
      return new JsonResponse(
        [
          'status' => 'error',
          'message' => $this->t('Invalid form ID or directory.'),
          'error' => 'Invalid form ID or directory.',
        ],
        400
      );
    }

    $validated_directory = $this->validateDirectoryPath($directory, $parent_directory);
    if (!$validated_directory) {
      return new JsonResponse([
        'status' => 'error',
        'message' => $this->t('Invalid or unauthorized directory path.'),
        'error' => 'Invalid or unauthorized directory path.',
      ], 400);
    }

    try {
      $this->destinationManager->setDestinationDirectory($validated_directory, $adf_instance_id);

      return new JsonResponse(
        [
          'data' => [
            'status' => 'success',
            'directory' => $validated_directory,
          ],
          'message' => $this->t('Directory updated successfully.'),
          'directory' => $validated_directory,
          'adf_instance_id' => $adf_instance_id,
          'status' => 'success',
        ],
        200
      );
    }
    catch (\Exception $e) {
      return new JsonResponse(
        [
          'status' => 'error',
          'message' => $e->getMessage(),
        ],
        500
          );
    }
  }

  /**
   * Validates and normalizes a directory path to prevent directory traversal.
   *
   * @param string $directory
   *   The raw directory input from the user (e.g., "public://some/path").
   *
   * @param string $parent_directory
   *   The base directory to restrict access to (e.g., "public://advanced_file_destination").
   *
   * @return string|false
   *   Returns the normalized and validated path if valid, or FALSE if invalid.
   */
  protected function validateDirectoryPath(string $directory, string $parent_directory): string|FALSE {
    // Normalize the directory path.
    $normalized_directory = $directory;
    $normalized_parent_directory = $parent_directory;

    // Ensure the normalized paths are valid and the directory is within the parent directory.
    if (!$normalized_directory || !$normalized_parent_directory) {
      $this->logger->error('Directory is not within the parent directory or paths are invalid.');
      return FALSE;
    }
    if ($normalized_directory && strpos($normalized_directory, $normalized_parent_directory) !== 0) {
      $this->logger->error('Directory is not within the parent directory: @directory', ['@directory' => $normalized_directory]);
      return FALSE;
    }

    // Check if the directory exists and is writable.
    if (!$this->destinationManager->isDirectoryInList($normalized_directory, TRUE)) {
      $this->logger->error('Directory does not exist or is not writable: @directory', ['@directory' => $normalized_directory]);
      return FALSE;
    }

    if ($normalized_parent_directory && !$this->destinationManager->isDirectoryInList($normalized_parent_directory, TRUE)) {
      $this->logger->error('Parent directory does not exist or is not writable: @directory', ['@directory' => $normalized_parent_directory]);
      return FALSE;
    }

    // Ensure that user inputs are in the directories list.
    if ($parent_directory && !$this->destinationManager->isDirectoryInList($parent_directory, TRUE)) {
      $this->logger->error('Parent directory is not in the list of valid directories: @directory and @list', ['@directory' => $parent_directory]);
      return FALSE;
    }
    // Ensure the normalized paths are valid and the directory is within the parent directory.
    if (!$normalized_directory || !$normalized_parent_directory || strpos($normalized_directory, $normalized_parent_directory) !== 0) {
      $this->logger->error('Directory is not within the parent directory or paths are invalid.');
      return FALSE;
    }

    // Ensure the directory uses a valid stream wrapper.
    if ($this->destinationManager->containsNonPublicScheme($normalized_directory)) {
      $this->logger->error('Directory does not use a valid stream wrapper: @directory', ['@directory' => $normalized_directory]);
      return FALSE;
    }

    // Ensure the parent directory uses a valid stream wrapper.
    if ($this->destinationManager->containsNonPublicScheme($normalized_parent_directory)) {
      $this->logger->error('Directory does not use a valid stream wrapper: @directory', ['@directory' => $normalized_parent_directory]);
      return FALSE;
    }

    return $directory;
  }

}
