<?php

namespace Drupal\advanced_file_destination\Form;

use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Image\ImageFactory;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a form for confirming bulk operations on file destinations.
 */
class AdvancedFileDestinationBulkConfirmForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The file URL generator service.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected FileUrlGeneratorInterface $fileUrlGenerator;

  /**
   * The destination manager service.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected AdvancedFileDestinationManager $destinationManager;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The image factory service.
   *
   * @var \Drupal\Core\Image\ImageFactory
   */
  protected ImageFactory $imageFactory;

  /**
   * The current user account.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Constructs a new AdvancedFileDestinationBulkConfirmForm.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator.
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $destination_manager
   *   The destination manager.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\Image\ImageFactory $image_factory
   *   The image factory.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user account.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    FileUrlGeneratorInterface $file_url_generator,
    AdvancedFileDestinationManager $destination_manager,
    DateFormatterInterface $date_formatter,
    FileSystemInterface $file_system,
    ImageFactory $image_factory,
    RequestStack $request_stack,
    AccountProxyInterface $current_user
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->fileUrlGenerator = $file_url_generator;
    $this->destinationManager = $destination_manager;
    $this->dateFormatter = $date_formatter;
    $this->fileSystem = $file_system;
    $this->imageFactory = $image_factory;
    $this->requestStack = $request_stack;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('file_url_generator'),
      $container->get('advanced_file_destination.manager'),
      $container->get('date.formatter'),
      $container->get('file_system'),
      $container->get('image.factory'),
      $container->get('request_stack'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'advanced_file_destination_bulk_confirm_form';
  }

  /**
   * Builds the form.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $current_user = $this->currentUser;
    // Check if the user has permission to perform bulk actions.
    if (!$current_user->hasPermission('delete advanced file destination') &&
        !$current_user->hasPermission('enable advanced file destination') &&
        !$current_user->hasPermission('disable advanced file destination') &&
        !$current_user->hasPermission('administer advanced file destination')) {
      // If the user does not have permission, redirect to the collection page.
      $this->messenger()->addError($this->t('You do not have permission to perform this action.'));
      $form_state->setRedirect('entity.afd_directory.collection');
      return $form;
    }
    // Get values from URL parameters
    $request = $this->requestStack->getCurrentRequest();
    $action = $request->query->get('action') ?: $form_state->getValue('action');
    $selected_json = $request->query->get('selected') ?: $form_state->getValue('table', []);

    if (is_string($selected_json)) {
      $selected_ids = json_decode($selected_json, TRUE);
      $selected = array_combine($selected_ids, $selected_ids);
    } else {
      $selected = $selected_json;
    }

    // Filter the selected items
    $selected = array_filter($selected);

    if (empty($action) || empty($selected)) {
      // If no action or no directories selected, show error message
      $form_state->setRedirect('entity.afd_directory.collection');
      return $form;
    }


    // Display a confirmation dialog for bulk actions.
    $title = $this->t('Are you sure you want to @action the selected directories?', [
      '@action' => ucfirst($action),
    ]);

    if ($action === 'delete') {
      $message = '<p><strong>' . $this->t('This action cannot be undone.') . '</strong></p>';
    }
    else {
      $message = '<p>' . $this->t('This action can be reversed later.') . '</p>';
    }

    $message .= '<p>' . $this->t('Selected directories: @count', [
      '@count' => count($selected),
    ]) . '</p><ul>';

    foreach ($selected as $id => $value) {
      if ($value) {
        $entity = $this->entityTypeManager->getStorage('afd_directory')->load($id);
        if ($entity) {
          /**
           * @var \Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory $entity
           */
          $message .= '<li>' . $this->t('@name (@path)', [
            '@name' => $entity->getName(),
            '@path' => $entity->getPath(),
          ]);

          $message .= $entity->getFilesAndDirectories();

          // $message .= '</div>'; // This was an error from previous diffs, it should be <li>
          $message .= '</li>';
        }
      }
    }
    $message .= '</ul>';

    $form['#title'] = $title;

    $form['container'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['bulk-confirm-container']],
    ];

    $form['container']['confirmation'] = [
      '#type' => 'item',
      '#markup' => $message,
      '#allowed_tags' => ['p', 'ul', 'li', 'strong', 'div', 'span'],
    ];

    $form['container']['action_hidden'] = [
      '#type' => 'hidden',
      '#value' => $action,
    ];

    $form['container']['selected_hidden'] = [
      '#type' => 'hidden',
      '#value' => json_encode(array_keys(array_filter($selected))),
    ];

    // Add wrapper for AJAX
    $form['#prefix'] = '<div id="bulk-confirm-form-wrapper" class="bulk-confirm-form-wrapper confirmation">';
    $form['#suffix'] = '</div>';
    $form['#id'] = $this->getFormId();

    // Add a submit button to confirm the action.
    $form['container']['actions-bulk'] = [
      '#type' => 'actions',
      '#attributes' => ['class' => ['form-actions']],
      '#id' => 'edit-actions-modal',
      'submit' => [
        '#type' => 'submit',
        '#value' => $action === 'delete' ? $this->t('Delete') : $this->t('Confirm'),
        '#attributes' => ['class' => ['button', $action === 'delete' ? 'button--danger' : 'button--primary']],
        '#limit_validation_errors' => [
          ['container', 'action_hidden'],
          ['container', 'selected_hidden'],
        ],
        '#submit' => ['::executeBulkAction'], // This remains for non-JS fallback if needed, AJAX callback takes precedence.
        '#ajax' => [
          'callback' => '::executeBulkAction',
          'wrapper' => 'bulk-confirm-form-wrapper',
          'event' => 'click',
        ],
      ],
      'cancel' => [
        '#type' => 'link',
        '#title' => $this->t('Cancel'),
        '#url' => Url::fromRoute('entity.afd_directory.collection'),
        '#attributes' => ['class' => ['button', 'button--secondary']],
      ],
    ];

    return $form;
  }

  /**
   * Executes the selected bulk action.
   */
  public function executeBulkAction(array &$form, FormStateInterface $form_state) {
    $request = $this->requestStack->getCurrentRequest();
    $action = $request->query->get('action') ?: $form_state->getValue('action');
    $selected_json = $request->query->get('selected') ?: $form_state->getValue('table', []);
    if ($selected_json === null) {
      $selected_json = '[]'; // Default to an empty JSON array string
    }
    $selected = json_decode($selected_json, TRUE);
    $selected = array_filter($selected);
    $response = new AjaxResponse();
    $message = '';
    $redirect_url = Url::fromRoute('entity.afd_directory.collection');
    $redirect_command = new RedirectCommand($redirect_url->toString());

    if (empty($action) || empty($selected)) {
      // $this->messenger()->addError($this->t('No action selected or no directories selected.')); // Removed as per instruction
      $this->messenger()->addError($this->t('No action selected or no directories selected.'));
      $response->addCommand($redirect_command);
      return $response;
    }
    // Perform the action based on the selected action type.
    switch ($action) {
      case 'delete':
        $overall_errors = []; // To collect errors from all entities.

        if(!$this->currentUser->hasPermission('delete advanced file destination')) {
          $this->messenger()->addError($this->t('You do not have permission to perform this action.'));
          $response->addCommand($redirect_command);
          return $response;
        }

        foreach ($selected as $entity_id_to_delete) {
          $entity = $this->entityTypeManager->getStorage('afd_directory')->load($entity_id_to_delete);
          if ($entity) {
            /**
             * @var \Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory $entity
             */
            $path = $entity->getPath();
            // $errors array is now $overall_errors, or handle errors per entity
            try {
              // 1. First delete files in subdirectories recursively
              $files = $this->fileSystem->scanDirectory($path, '/.*/', ['recurse' => TRUE]);
              if (!empty($files)) {
                foreach ($files as $uri => $file) {
                  try {
                    // Delete file entity if it exists.
                    $file_entity = $this->destinationManager->loadFileByUri($uri);
                    if ($file_entity) {
                      $file_entity->delete();
                      $this->logger('advanced_file_destination')->notice(
                        'Deleted file entity: @file',
                        [
                          '@file' => $uri,
                        ]
                      );
                    }

                    // Delete physical file if it still exists.
                    if (file_exists($uri)) {
                      $this->fileSystem->delete($uri);
                      $this->logger('advanced_file_destination')->notice(
                        'Deleted file: @file',
                        [
                          '@file' => $uri,
                        ]
                      );
                    }
                  } catch (\Exception $e) {
                    $overall_errors[] = $this->t(
                      'Failed to delete file @file from directory @dir_path: @error',
                      [
                        '@file' => $uri,
                        '@dir_path' => $path,
                        '@error' => $e->getMessage(),
                      ]
                    );
                  }
                }
              }

              // 2. Delete subdirectory entities recursively
              $query = $this->entityTypeManager->getStorage('afd_directory')
                ->getQuery()
                ->condition('path', $path . '/%', 'STARTS_WITH')
                ->accessCheck(FALSE);
              $subdirectory_ids = $query->execute();

              if (!empty($subdirectory_ids)) {
                foreach ($subdirectory_ids as $sub_id) { // Changed $id to $sub_id to avoid conflict
                  try {
                    $subdirectory = $this->entityTypeManager->getStorage('afd_directory')->load($sub_id);
                    if ($subdirectory) {
                      /**
                       * @var \Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory $subdirectory
                       */
                      $subdirectory->delete();
                      $this->logger('advanced_file_destination')->notice(
                        'Deleted directory entity: @dir',
                        [
                          '@dir' => $subdirectory->getPath(),
                        ]
                      );
                    }
                  } catch (\Exception $e) {
                    $overall_errors[] = $this->t(
                      'Failed to delete subdirectory entity @id under @dir_path: @error',
                      [
                        '@id' => $sub_id,
                        '@dir_path' => $path,
                        '@error' => $e->getMessage(),
                      ]
                    );
                  }
                }
              }

              // 3. Delete the physical directory structure
              if (file_exists($path)) {
                if (!$this->fileSystem->deleteRecursive($path)) {
                  // Log this specific error, but also add to overall_errors for user feedback
                  $this->logger('advanced_file_destination')->error('Could not delete directory structure for @path', ['@path' => $path]);
                  $overall_errors[] = $this->t('Could not delete directory structure for @path.', ['@path' => $path]);
                } else {
                  $this->logger('advanced_file_destination')->notice(
                    'Deleted directory structure: @path',
                    [
                      '@path' => $path,
                    ]
                  );
                }
              }

              // 4. Finally delete the main directory entity
              $entity_label = $entity->label(); // Get label before deleting
              $entity->delete();
              // Removed individual success message: $this->messenger()->addStatus(...)
              $this->logger('advanced_file_destination')->notice('Successfully deleted directory entity: @label', ['@label' => $entity_label]);

            } catch (\Exception $e) {
              // This catch is for general errors for the current entity
              $overall_errors[] = $this->t(
                'Error processing directory @path: @error',
                [
                  '@path' => $path,
                  '@error' => $e->getMessage(),
                ]
              );
              $this->logger('advanced_file_destination')->error(
                'Overall deletion failed for directory @path: @error',
                [
                  '@path' => $path,
                  '@error' => $e->getMessage(),
                ]
              );
            }
          } else {
            $overall_errors[] = $this->t('Could not load directory with ID: @id for deletion.', ['@id' => $entity_id_to_delete]);
          }
        }

        // After iterating through all selected items:
        if (!empty($overall_errors)) {
          foreach ($overall_errors as $error_message) {
            $this->messenger()->addWarning($error_message);
          }
          // Add a summary message indicating partial success if some items were processed.
          // This depends on how you define "overall success". If any error means overall failure,
          // then perhaps a different message or only errors.
          // For now, let's assume we still try to redirect but show errors.
          $this->messenger()->addError($this->t('Some directories could not be deleted. Please check the messages for details.'));
        }
        else {
          $this->messenger()->addStatus($this->t('All selected directories have been successfully deleted.'));
        }

        $response->addCommand($redirect_command);
        return $response;

      case 'disable':
        // Implement disable logic here
        $response = new AjaxResponse();
        $overall_errors = []; // To collect errors from all entities.

        if (!$this->currentUser->hasPermission('disable advanced file destination')) {
          $this->messenger()->addError($this->t('You do not have permission to perform this action.'));
          $response->addCommand($redirect_command);
          return $response;
        }

        foreach ($selected as $entity_id_to_disable) {
          $entity = $this->entityTypeManager->getStorage('afd_directory')->load($entity_id_to_disable);
          if ($entity) {
            /**
             * @var \Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory $entity
             */
            try {
              $entity->set('status', 0); // Assuming 'status' is the field to disable the directory.
              $entity->save();
              $this->logger('advanced_file_destination')->notice(
                'Disabled directory entity: @label',
                ['@label' => $entity->label()]
              );
            } catch (\Exception $e) {
              $overall_errors[] = $this->t(
                'Error disabling directory @label: @error',
                [
                  '@label' => $entity->label(),
                  '@error' => $e->getMessage(),
                ]
              );
            }
          } else {
            $overall_errors[] = $this->t('Could not load directory with ID: @id for disabling.', ['@id' => $entity_id_to_disable]);
          }
        }
        // After iterating through all selected items:
        if (!empty($overall_errors)) {
          foreach ($overall_errors as $error_message) {
            $this->messenger()->addWarning($error_message);
          }
          // Add a summary message indicating partial success if some items were processed.
          // This depends on how you define "overall success". If any error means overall failure,
          // then perhaps a different message or only errors.
          // For now, let's assume we still try to redirect but show errors.
          $this->messenger()->addError($this->t('Some directories could not be disabled. Please check the messages for details.'));
        } else {
          $this->messenger()->addStatus($this->t('All selected directories have been successfully disabled.'));
        }

        $response->addCommand($redirect_command);
        return $response;

      case 'enable':
        // Implement enable logic here
        $overall_errors = []; // To collect errors from all entities.
        if (!$this->currentUser->hasPermission('enable advanced file destination')) {
          $this->messenger()->addError($this->t('You do not have permission to perform this action.'));
          $response->addCommand($redirect_command);
          return $response;
        }

        foreach ($selected as $entity_id_to_enable) {
          $entity = $this->entityTypeManager->getStorage('afd_directory')->load($entity_id_to_enable);
          if ($entity) {
            /**
             * @var \Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory $entity
             */
            try {
              $entity->set('status', 1); // Assuming 'status' is the field to enable the directory.
              $entity->save();
              $this->logger('advanced_file_destination')->notice(
                'Enabled directory entity: @label',
                ['@label' => $entity->label()]
              );
            } catch (\Exception $e) {
              $overall_errors[] = $this->t(
                'Error enabling directory @label: @error',
                [
                  '@label' => $entity->label(),
                  '@error' => $e->getMessage(),
                ]
              );
            }
          } else {
            $overall_errors[] = $this->t('Could not load directory with ID: @id for enabling.', ['@id' => $entity_id_to_enable]);
          }
        }

        // After iterating through all selected items:
        if (!empty($overall_errors)) {
          foreach ($overall_errors as $error_message) {
            $this->messenger()->addWarning($error_message);
          }
          // Add a summary message indicating partial success if some items were processed.
          // This depends on how you define "overall success". If any error means overall failure,
          // then perhaps a different message or only errors.
          // For now, let's assume we still try to redirect but show errors.
          $this->messenger()->addError($this->t('Some directories could not be enabled. Please check the messages for details.'));
        } else {
          $this->messenger()->addStatus($this->t('All selected directories have been successfully enabled.'));
        }

        $response->addCommand($redirect_command);
        return $response;

      default:
        // $this->messenger()->addError($this->t('Unknown action: @action', ['@action' => $action])); // Removed as per instruction
        // The $response object is initialized at the start of the method.
        // We add a message command to it and return it. No redirect.
        $this->messenger()->addError($this->t('Unknown action: @action', ['@action' => $action]));
        $response->addCommand($redirect_command);
        return $response;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // No specific validation needed for this form
    // The AJAX callback will handle the action execution and any errors
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // For AJAX submissions, the AJAX callback handles everything.
    // This method is called after executeBulkAction, so we just return an empty response.
    //return new AjaxResponse();
  }
}
