<?php

namespace Drupal\advanced_file_destination\Form;

use Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Core\Entity\EntityDeleteForm;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\file\FileUsage\FileUsageInterface;

/**
 * Provides a form for directory deletion with file usage information.
 */
class AdvancedFileDestinationDeleteForm extends EntityDeleteForm {

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The file usage service.
   *
   * @var \Drupal\file\FileUsage\FileUsageInterface
   */
  protected $fileUsage;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The advanced file destination manager service.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * The current user account.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The directory entity being deleted.
   *
   * @var \Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory
   */
  protected $entity;

  /**
   * Constructs a new AdvancedFileDestinationDeleteForm.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    FileSystemInterface $file_system,
    FileUsageInterface $file_usage,
    MessengerInterface $messenger,
    LoggerChannelFactoryInterface $logger_factory,
    AdvancedFileDestinationManager $destinationManager,
    AccountProxyInterface $current_user
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->fileSystem = $file_system;
    $this->fileUsage = $file_usage;
    $this->messenger = $messenger;
    $this->loggerFactory = $logger_factory;
    $this->destinationManager = $destinationManager;
    $this->currentUser = $current_user;
    // Ensure the entity is set from the parent class.
    if ($this->entity instanceof AdvancedFileDestinationDirectory) {
      $this->entity = $this->entity;
    }
    else {
      throw new \InvalidArgumentException('Entity must be an instance of AdvancedFileDestinationDirectory');
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('entity_type.manager'),
          $container->get('file_system'),
          $container->get('file.usage'),
          $container->get('messenger'),
          $container->get('logger.factory'),
          $container->get('advanced_file_destination.manager'),
          $container->get('current_user')
      );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    $current_user = $this->currentUser;

    // Check if the user has permission to perform bulk actions.
    if (
      !$current_user->hasPermission('delete advanced file destination') &&
      !$current_user->hasPermission('administer advanced file destination')
    ) {
      // If the user does not have permission, redirect to the collection page.
      $this->messenger()->addError($this->t('You do not have permission to perform this action.'));
      $form_state->setRedirect('entity.afd_directory.collection');
      return $form;
    }
    // Build description markup.
    $description = '<div class="directory-delete-confirmation">';
    $description .= '<p>' . $this->t(
          'Are you sure you want to delete the directory %name?', [
            '%name' => $this->entity->label(),
          ]
      ) . '</p>';

    $description .= $this->entity->getFilesAndDirectories();

    // Replace default description with our enhanced version.
    $form['description'] = [
      '#type' => 'item',
      '#markup' => $description,
      '#allowed_tags' => ['p', 'ul', 'li', 'strong', 'div', 'span'],
    ];

    // Add custom CSS.
    $form['#attached']['library'][] = 'advanced_file_destination/directory_selection';

    return $form;
  }

  /**
   * Formats a usage item for display.
   */
  protected function formatUsageItem($type, $id, $file) {
    try {
      $entity = $this->entityTypeManager->getStorage($type)->load($id);
      if ($entity) {
        $label = $entity->label() ?: $this->t('Untitled');
        $url = $entity->toUrl()->toString();
        return $this->t(
              '@type: <a href="@url">@label</a> (using @filename)', [
                '@type' => ucfirst($type),
                '@url' => $url,
                '@label' => $label,
                '@filename' => $file->getFilename(),
              ]
          );
      }
    }
    catch (\Exception $e) {
      // Log the error if the entity cannot be loaded.
      $this->loggerFactory->get('advanced_file_destination')->error('Error loading entity: @error', ['@error' => $e->getMessage()]);
    }
    return $this->t(
          '@type #@id (using @filename)', [
            '@type' => ucfirst($type),
            '@id' => $id,
            '@filename' => $file->getFilename(),
          ]
      );
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $path = $this->entity->getPath();
    $errors = [];

    try {
      // 1. First delete files in subdirectories recursively
      $files = $this->fileSystem->scanDirectory($path, '/.*/', ['recurse' => TRUE]);
      if (!empty($files)) {
        foreach ($files as $uri => $file) {
          try {
            // Delete file entity if it exists.
            $file_entity = $this->destinationManager->loadFileByUri($uri);
            if ($file_entity) {
              $file_entity->delete();
              $this->loggerFactory->get('advanced_file_destination')->notice(
                    'Deleted file entity: @file', [
                      '@file' => $uri,
                    ]
                );
            }

            // Delete physical file if it still exists.
            if (file_exists($uri)) {
              $this->fileSystem->delete($uri);
              $this->loggerFactory->get('advanced_file_destination')->notice(
                    'Deleted file: @file', [
                      '@file' => $uri,
                    ]
                );
            }
          }
          catch (\Exception $e) {
            $errors[] = $this->t(
                  'Failed to delete file @file: @error', [
                    '@file' => $uri,
                    '@error' => $e->getMessage(),
                  ]
              );
          }
        }
      }

      // 2. Delete subdirectory entities recursively
      $query = $this->entityTypeManager->getStorage('afd_directory')
        ->getQuery()
        ->condition('path', $path . '/%', 'STARTS_WITH')
        ->accessCheck(FALSE);
      $subdirectory_ids = $query->execute();

      if (!empty($subdirectory_ids)) {
        foreach ($subdirectory_ids as $id) {
          try {
            $subdirectory = $this->entityTypeManager->getStorage('afd_directory')->load($id);
            if ($subdirectory) {
              /**
               * @var \Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory $subdirectory
               */
              $subdirectory->delete();
              $this->loggerFactory->get('advanced_file_destination')->notice(
                    'Deleted directory entity: @dir', [
                      '@dir' => $subdirectory->getPath(),
                    ]
                );
            }
          }
          catch (\Exception $e) {
            $errors[] = $this->t(
                  'Failed to delete directory entity @id: @error', [
                    '@id' => $id,
                    '@error' => $e->getMessage(),
                  ]
              );
          }
        }
      }

      // 3. Delete the physical directory structure
      if (file_exists($path)) {
        if (!$this->fileSystem->deleteRecursive($path)) {
          throw new \Exception('Could not delete directory structure');
        }
        $this->loggerFactory->get('advanced_file_destination')->notice(
              'Deleted directory structure: @path', [
                '@path' => $path,
              ]
          );
      }

      // 4. Finally delete the main directory entity
      $this->entity->delete();
      $this->messenger->addStatus(
            $this->t(
                'The directory %label has been deleted.', [
                  '%label' => $this->entity->label(),
                ]
            )
        );

      // Show any errors that occurred.
      if (!empty($errors)) {
        foreach ($errors as $error) {
          $this->messenger->addWarning($error);
        }
      }

    }
    catch (\Exception $e) {
      $this->messenger->addError(
            $this->t(
                'Error deleting directory: @error', [
                  '@error' => $e->getMessage(),
                ]
            )
        );
      $this->loggerFactory->get('advanced_file_destination')->error(
            'Directory deletion failed: @error', [
              '@error' => $e->getMessage(),
            ]
        );
      return;
    }

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

  /**
   * Scans a directory recursively for files.
   *
   * @param string $path
   *   The directory path to scan.
   *
   * @return array
   *   Array of file objects with full path information.
   */
  protected function scanDirectory($path) {
    return $this->fileSystem->scanDirectory($path, '/.*/', ['recurse' => TRUE]);
  }

  /**
   * {@inheritdoc}
   */
  protected function copyFormValuesToEntity(EntityInterface $entity, array $form, FormStateInterface $form_state) {
    // Get all defined field names from the entity.
    $field_names = array_keys($entity->getFieldDefinitions());

    // Only copy values for fields that actually exist on the entity.
    foreach ($field_names as $field_name) {
      if ($form_state->hasValue($field_name)) {
        $entity->set($field_name, $form_state->getValue($field_name));
      }
    }
  }

}
