<?php

namespace Drupal\advanced_file_destination\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Form for listing directories.
 */

class AdvancedFileDestinationDirectoriesForm extends FormBase {
  /**
   * The advanced file destination manager.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * {@inheritdoc}
   */
  protected $limit = 25;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $storage;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new AdvancedFileDestinationDirectoriesForm object.
   *
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $destination_manager
   *   The destination manager service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(
    FormBuilderInterface $form_builder,
    AdvancedFileDestinationManager $destination_manager,
    RequestStack $request_stack,
    ConfigFactoryInterface $config_factory,
    DateFormatterInterface $date_formatter,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    $this->formBuilder = $form_builder;
    $this->destinationManager = $destination_manager;
    $this->requestStack = $request_stack;
    $this->configFactory = $config_factory;
    $this->dateFormatter = $date_formatter;
    $this->entityTypeManager = $entity_type_manager;
    $this->storage = $entity_type_manager->getStorage('afd_directory');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('form_builder'),
      $container->get('advanced_file_destination.manager'),
      $container->get('request_stack'),
      $container->get('config.factory'),
      $container->get('date.formatter'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId()
  {
    return 'directory_list_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Add the library.
    $form['#attached']['library'][] = 'advanced_file_destination/directory_list';
    $form['#attached']['library'][] = 'core/drupal.ajax';

    // Add filters.
    $form['filters'] = [
      '#type' => 'details',
      '#title' => $this->t('Filters'),
      '#open' => TRUE,
      '#attributes' => ['class' => ['directory-filters', 'clearfix']],
      'status' => [
        '#type' => 'select',
        '#title' => $this->t('Status'),
        '#options' => [
          '' => $this->t('- Any -'),
          '1' => $this->t('Active'),
          '0' => $this->t('Disabled'),
        ],
        '#default_value' => $this->requestStack->getCurrentRequest()->query->get('status'),
      ],

      'name' => [
        '#type' => 'search',
        '#title' => $this->t('Search by name'),
        '#default_value' => $this->requestStack->getCurrentRequest()->query->get('name'),
        '#size' => 30,
      ],
      'actions' => [
        '#type' => 'actions',
        'submit' => [
          '#type' => 'submit',
          '#value' => $this->t('Filter'),
          '#button_type' => 'primary',
        ],
        'reset' => [
          '#type' => 'link',
          '#title' => $this->t('Reset'),
          '#url' => Url::fromRoute('<current>'),
          '#attributes' => ['class' => ['button']],
        ],
      ],
    ];

    // Build table.
    $form['directories'] = [
      '#type' => 'details',
      '#title' => $this->t('Directories'),
      '#open' => TRUE,
      '#attributes' => ['class' => ['directory-listing']],
    ];
    $form['directories']['#prefix'] = '<div class="directory-listing-wrapper" id="directory-listing-wrapper">';
    $form['directories']['#suffix'] = '</div>';
    $form['directories']['table'] = [
      '#type' => 'table',
      '#header' => $this->buildHeader(),
      '#empty' => $this->t('No directories available.'),
      '#tableselect' => TRUE,
      '#attributes' => ['class' => ['directory-listing-table']],
    ];

    // Load entities with pagination.
    $entities = $this->load();
    // Sort entities by weight and then by name.
    usort($entities, function ($a, $b) {
      /**
       * @var \Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory $a
       * @var \Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory $b
       */
      // Compare by weight first.
      $weightA = method_exists($a, 'getWeight') ? $a->getWeight() : 0;
      $weightB = method_exists($b, 'getWeight') ? $b->getWeight() : 0;
      if ($weightA === $weightB) {
        // If weights are equal, compare by name.
        $nameA = method_exists($a, 'getName') ? $a->getName() : '';
        $nameB = method_exists($b, 'getName') ? $b->getName() : '';
        return strnatcasecmp($nameA, $nameB);
      }
      return $weightA <=> $weightB;
    });

    foreach ($entities as $entity) {
      $form['directories']['table'][$entity->id()] = $this->buildRow($entity);
    }

    // Add bulk operations with new structure.
    $form['directories']['operations'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['directory-bulk-operations'],
        'data-drupal-selector' => 'edit-bulk-actions',
        'role' => 'group',
      ],
      'status' => [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['directory-bulk-operations__item', 'directory-bulk-operations__item--status'],
        ],
        '#markup' => $this->t('No items selected'),
      ],
      'action' => [
        '#type' => 'select',
        '#title' => $this->t('Action'),
        '#title_display' => 'invisible',
        '#options' => [
          'enable' => $this->t('Enable'),
          'disable' => $this->t('Disable'),
          'delete' => $this->t('Delete'),
        ],
        '#empty_option' => $this->t('- Select action -'),
        '#wrapper_attributes' => [
          'class' => ['directory-bulk-operations__item', 'directory-bulk-operations__item--preceding-actions'],
        ],
      ],
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Apply to selected items'),
        '#button_type' => 'primary',
        '#attributes' => ['class' => ['button--small']],
        '#name' => 'operations-submit',
        '#ajax' => [
          'callback' => [$this, 'submitBulkForm'],
          'wrapper' => 'directory-listing-wrapper',
          'progress' => [
            'type' => 'throbber',
            'message' => $this->t('Processing...'),
          ],
        ],
      ],
    ];

    // Add pager.
    $form['pager'] = [
      '#type' => 'pager',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header = [
      'preview' => [
        'data' => $this->t('Preview'),
        'class' => ['directory-preview-header'],
      ],
    ];

    $header += [
      'name' => [
        'data' => $this->t('Name'),
        'field' => 'label',
        'specifier' => 'label',
        'sort' => 'asc',
      ],
      'path' => [
        'data' => $this->t('Path'),
        'field' => 'path',
        'specifier' => 'path',
      ],
      'status' => [
        'data' => $this->t('Status'),
        'field' => 'status',
        'specifier' => 'status',
      ],
      'created' => [
        'data' => $this->t('Created'),
        'field' => 'created',
        'specifier' => 'created',
        'sort' => 'desc',
      ],
      'changed' => [
        'data' => $this->t('Updated'),
        'field' => 'changed',
        'specifier' => 'changed',
        'sort' => 'desc',
      ],
      'operations' => [
        'data' => $this->t('Operations'),
      ],
    ];

    return $header;
  }

  /**
   * {@inheritdoc}
   */
  public function load() {
    $query = $this->getStorage()->getQuery()
      ->accessCheck(TRUE)
      ->pager($this->limit);

    // Add filters.
    if ($status = $this->requestStack->getCurrentRequest()->query->get('status')) {
      $query->condition('status', $status);
    }
    if ($name = $this->requestStack->getCurrentRequest()->query->get('name')) {
      $query->condition('name', '%' . $name . '%', 'LIKE');
    }

    // Get sort parameters from URL or use defaults.
    $header = $this->buildHeader();
    $order = $this->requestStack->getCurrentRequest()->query->get('order');
    $sort = $this->requestStack->getCurrentRequest()->query->get('sort');

    // If no sort specified in URL, use changed/DESC as default.
    if (empty($sort)) {
      $sort = 'changed';
      $order = 'DESC';
    }

    // Ensure we have a valid sort field.
    if (!isset($header[$sort]['field'])) {
      $sort = 'changed';
      $order = 'DESC';
    }

    // Apply sorting.
    $query->sort($header[$sort]['field'], $order);

    $ids = $query->execute();
    return $this->storage->loadMultiple($ids);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Check if it's a bulk operation submit.
    $triggering_element = $form_state->getTriggeringElement();
    if (isset($triggering_element['#name']) && $triggering_element['#name'] === 'operations-submit') {
      // For AJAX submissions, the AJAX callback handles the response
      // This method should not return anything for AJAX calls
      return;
    }

    // Handle other form submissions (like filters)
    $filters = [];
    foreach (['status', 'name'] as $filter) {
      if ($value = $form_state->getValue($filter)) {
        $filters[$filter] = $value;
      }
    }

    // Redirect with filters.
    $options = [];
    if (!empty($filters)) {
      $options['query'] = $filters;
    }
    $form_state->setRedirect('entity.advanced_file_destination_directory.collection', [], $options);
  }

  /**
   * Submit handler for bulk operations.
   */
  public function submitBulkForm(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    // Try different paths to get the action
    $action = $form_state->getValue(['directories', 'operations', 'action']);
    if (empty($action)) {
      // Try alternative path
      $action = $form_state->getValue(['action']);
    }
    if (empty($action)) {
      // Try getting from operations directly
      $operations = $form_state->getValue(['directories', 'operations']);
      if (is_array($operations) && isset($operations['action'])) {
        $action = $operations['action'];
      }
    }

    if (empty($action)) {
      $response->addCommand(new MessageCommand($this->t('Please select an action to perform.'), NULL, ['type' => 'error']));
      return $response;
    }

    // Get selected items from the table
    $selected = array_filter($form_state->getValue(['directories', 'table'], []));
    if (empty($selected)) {
      // Try alternative path
      $selected = array_filter($form_state->getValue(['table'], []));
    }
    if (empty($selected)) {
      $response->addCommand(new MessageCommand($this->t('No directories selected for the action.'), NULL, ['type' => 'error']));
      return $response;
    }

    // Create a dialog that opens the bulk confirm form
    $form_state->setValues([
      'action' => $action,
      'selected' =>  json_encode(array_keys($selected)),
    ]);

    // Build the dialog form.
    $redirect_url = Url::fromRoute('advanced_file_destination.bulk_confirm', $form_state->getValues());

    $response->addCommand(new RedirectCommand($redirect_url->toString()));
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
      // Validate filters.
      if ($name = $form_state->getValue('name')) {
        if (strlen($name) < 2) {
          $form_state->setErrorByName('name', $this->t('Search term must be at least 2 characters.'));
        }
      }
  }

  /**
   * Builds a row for the directory list.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return array
   *   A render array representing the row.
   */
  public function buildRow(EntityInterface $entity) {
    /**
     * @var \Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory $entity
     */
    $row = [];

    if ($this->containsNonPublicScheme($entity->getPath())) {
      return;
    }

    // Add preview column with folder icon.
    $row['preview'] = [
      'data' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['directory-preview-wrapper']],
        'icon' => [
          '#type' => 'html_tag',
          '#tag' => 'span',
          '#attributes' => [
            'class' => ['directory-icon'],
            'title' => $this->t('Directory'),
          ],
        ],
      ],
    ];

    // Check if path exists and is a directory.
    $path = $entity->getPath();
    $is_directory = file_exists($path) && is_dir($path);

    $row['name'] = [
      'data' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['directory-name-wrapper']],
        'icon' => [
          '#type' => 'html_tag',
          '#tag' => 'span',
          '#attributes' => [
            'class' => ['directory-icon'],
            'title' => $this->t('Directory'),
          ],
        ],
        'link' => [
          '#type' => 'link',
          '#title' => $entity->getName(),
          '#url' => $is_directory
            ? Url::fromRoute('entity.afd_directory.edit_form', ['afd_directory' => $entity->id()])
            : $entity->toUrl('edit-form'),
        ],
      ],
    ];

    $row['path'] = [
      'data' => ['#markup' => $path],
    ];

    $row['status'] = [
      'data' => [
        '#markup' => $entity->isActive() ? $this->t('Active') : $this->t('Disabled'),
        '#wrapper_attributes' => [
          'class' => [
            'directory-status',
            $entity->isActive() ?
              'directory-status--active' :
              'directory-status--disabled',
          ],
        ],
      ],
    ];

    $row['created'] = [
      'data' => [
        '#markup' => $this->dateFormatter->format($entity->getCreatedTime()),
      ],
    ];

    $row['changed'] = [
      'data' => [
        '#markup' => $entity->getChangedTime() ? $this->dateFormatter->format($entity->getChangedTime()) : $this->dateFormatter->format($entity->getCreatedTime()),
      ],
    ];

    $row['operations'] = [
      'data' => $this->buildOperations($entity),
    ];

    return $row;
  }

  /**
   * Gets the entity storage.
   *
   * @return \Drupal\Core\Entity\EntityStorageInterface
   *   The entity storage.
   */
  protected function getStorage() {
    return $this->storage;
  }

  /**
   * Build operations for an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return array
   *   The operations array.
   */
  protected function buildOperations(EntityInterface $entity) {
    $operations = [];

    if ($entity->access('update')) {
      $operations['edit'] = [
        'title' => $this->t('Edit'),
        'weight' => 10,
        'url' => $entity->toUrl('edit-form'),
      ];
    }

    if ($entity->access('delete')) {
      $operations['delete'] = [
        'title' => $this->t('Delete'),
        'weight' => 100,
        'url' => $entity->toUrl('delete-form'),
      ];
    }

    return [
      '#type' => 'operations',
      '#links' => $operations,
    ];
  }

  /**
   * Checks if the path contains a non-public scheme.
   *
   * @param string $path
   *   The path to check.
   *
   * @return bool
   *   TRUE if the path contains a non-public scheme, FALSE otherwise.
   */
  protected function containsNonPublicScheme($path) {
    return $this->destinationManager->containsNonPublicScheme($path);
  }
}
