<?php

namespace Drupal\advanced_file_destination\Form;

use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\File\FileExists;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Image\ImageFactory;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\Core\Routing\RouteMatchInterface;

/**
 * Form handler for directory add/edit forms.
 */
class AdvancedFileDestinationDirectoryForm extends EntityForm {

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The image factory.
   *
   * @var \Drupal\Core\Image\ImageFactory
   */
  protected $imageFactory;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The advanced file destination manager.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * Constructs a new AdvancedFileDestinationDirectoryForm.
   *
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Image\ImageFactory $image_factory
   *   The image factory service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user service.
   * @param \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager $destination_manager
   *   The advanced file destination manager service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   */
  public function __construct(
    FileSystemInterface $file_system,
    MessengerInterface $messenger,
    FileUrlGeneratorInterface $file_url_generator,
    DateFormatterInterface $date_formatter,
    ImageFactory $image_factory,
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    AccountProxyInterface $current_user,
    AdvancedFileDestinationManager $destination_manager,
    RouteMatchInterface $route_match,
  ) {
    $this->fileSystem = $file_system;
    $this->messenger = $messenger;
    $this->fileUrlGenerator = $file_url_generator;
    $this->dateFormatter = $date_formatter;
    $this->imageFactory = $image_factory;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
    $this->destinationManager = $destination_manager;
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('file_system'),
          $container->get('messenger'),
          $container->get('file_url_generator'),
          $container->get('date.formatter'),
          $container->get('image.factory'),
          $container->get('config.factory'),
          $container->get('entity_type.manager'),
          $container->get('current_user'),
          $container->get('advanced_file_destination.manager'),
          $container->get('current_route_match')
      );
  }

  /**
   * Gets the default directory from configuration.
   *
   * @return string
   *   The configured default directory or 'public://' as fallback.
   */
  protected function getDefaultDirectory() {
    return $this->configFactory->get('advanced_file_destination.settings')->get('default_directory') ?: 'public://';
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    $directory = $this->entity;
    /**
     * @var \Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory $directory
     */
    $route_name = $this->routeMatch->getRouteName();
    if (in_array($route_name, ['entity.afd_directory.edit_form'])) {
      $form['#title'] = $this->t('Edit directory %label', ['%label' => $directory->label()]);
    }
    else {
      $form['#title'] = $this->t('Add directory');
    }
    // Set theme hook early and explicitly to ensure it's applied.
    $form['#theme'] = 'afd_directory_form';

    // Always attach required libraries.
    $form['#attached']['library'][] = 'advanced_file_destination/directory_form';
    $form['#attached']['library'][] = 'advanced_file_destination/directory_list';
    $form['#attached']['library'][] = 'advanced_file_destination/directory_selection';

    // Add meta information.
    $form['meta'] = [
      '#type' => 'details',
      '#title' => $this->t('Directory information'),
      '#open' => TRUE,
      '#weight' => -10,
      '#attributes' => ['class' => ['entity-meta__header']],
    ];

    if ($directory->getCreatedTime()) {
      $form['meta']['created'] = [
        '#type' => 'item',
        '#title' => $this->t('Created'),
        '#markup' => $this->dateFormatter->format($directory->getCreatedTime(), 'medium'),
        '#wrapper_attributes' => ['class' => ['entity-meta__last-saved']],
        '#weight' => 0,
      ];
    }

    if ($directory->getChangedTime() && !$directory->isNew()) {
      $form['meta']['changed'] = [
        '#type' => 'item',
        '#title' => $this->t('Last saved'),
        '#markup' => $this->dateFormatter->format($directory->getChangedTime(), 'medium'),
        '#wrapper_attributes' => ['class' => ['entity-meta__last-saved']],
        '#weight' => 1,
      ];
    }

    // Add revision information group.
    $form['revision_information'] = [
      '#type' => 'details',
      '#title' => $this->t('Revision information'),
      '#weight' => 20,
      '#open' => TRUE,
      '#attributes' => ['class' => ['entity-meta__revision']],
    ];

    $form['revision_information']['revision'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Create new revision'),
      '#default_value' => $directory->isNew() || $directory->isNewRevision(),
      '#weight' => -1,
    ];

    $form['revision_information']['revision_log'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Revision log message'),
      '#description' => $this->t('Briefly describe the changes you have made.'),
      '#default_value' => $directory->getRevisionLogMessage(),
      '#rows' => 4,
      '#states' => [
        'visible' => [
          ':input[name="revision"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Name'),
      '#maxlength' => 255,
      '#default_value' => $directory->label(),
      '#description' => $this->t('The human-readable name of the directory.'),
      '#required' => TRUE,
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => !$directory->isNew() ? $directory->status() : TRUE,
      '#description' => $this->t('Whether the directory is active and available for use.'),
    ];

    $form['path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Directory path'),
      '#default_value' => $directory->getPath(),
      '#description' => $this->t(
          'The directory path (e.g., custom-folder). The directory will be created in the default stream @default_dir', [
            '@default_dir' => $this->getDefaultDirectory(),
          ]
      ),
      '#required' => TRUE,
    ];
    if (in_array($route_name, ['entity.afd_directory.edit_form'])) {
      $form['path']['#disabled'] = TRUE;
      $form['path']['#description'] = $this->t(
          'The directory @path is not editable. To change the path, please create a new directory.', [
            '@path' => $directory->getPath(),
          ]
      );
    }
    $form['weight'] = [
      '#type' => 'weight',
      '#title' => $this->t('Weight'),
      '#default_value' => $directory->getWeight(),
      '#description' => $this->t('Optional. Directories are displayed in ascending order by weight.'),
    ];

    // TO DO: Add roles to directory.
    $form['files'] = [
      '#type' => 'details',
      '#title' => $this->t('Files in directory'),
      '#open' => TRUE,
      '#weight' => 100,
    ];

    // Get files from directory.
    $path = $directory->getPath();
    $full_path = $this->destinationManager->normalizeDirectoryPath($path);

    if ($path && file_exists($path)) {
      $files = scandir($path);
      $rows = [];

      foreach ($files as $filename) {
        if ($filename === '.' || $filename === '..') {
          continue;
        }
        $file_path = $path . $filename;
        $is_dir = is_dir($file_path);
        $name = $filename;
        // Build preview with proper icon styling.
        $preview = [
          '#type' => 'container',
          '#attributes' => ['class' => ['directory-preview-wrapper']],
          'icon' => [
            '#type' => 'html_tag',
            '#tag' => 'span',
            '#attributes' => [
              'class' => [$is_dir ? 'directory-icon' : 'file-icon'],
              'title' => $is_dir ? $this->t('Directory') : $this->t('File'),
            ],
          ],
        ];

        if (!$is_dir && $this->isImage($file_path)) {
          $preview = [
            '#theme' => 'image_style',
            '#style_name' => 'thumbnail',
            '#uri' => $file_path,
          ];
        }

        // Generate proper URL.
        try {
          $absolute_url = $this->fileUrlGenerator->generateAbsoluteString($file_path);
          $url = Url::fromUri($absolute_url);
        }
        catch (\Exception $e) {
          $this->logger('advanced_file_destination')->error(
                'Error generating URL for @uri: @error',
                ['@uri' => $file_path, '@error' => $e->getMessage()]
            );
          continue;
        }

        // Build operations.
        $operations = [];

        // Add edit operation if it's a directory.
        if (is_dir($full_path) && $is_dir) {
          // Get entity bulk operations.
          $entity = $this->entityTypeManager
            ->getStorage('afd_directory')
            ->loadByProperties(['path' => $full_path]);
          $entity = reset($entity);
          if (!$entity) {
            continue;
          }
          $entityListBuilder = new EntityListBuilder(
            $entity->getEntityType(),
            $this->entityTypeManager->getStorage($entity->getEntityTypeId())
          );
          $operations['data'] = [
            '#type' => 'operations',
            '#links' => $entityListBuilder->getOperations($entity),
          ];
          $name = $entity->label();
        }
        // Add edit operation for media files.
        else {
          $file_entity = $this->getFileEntity($file_path);
          if ($file_entity) {
            $entityListBuilder = new EntityListBuilder(
                $file_entity->getEntityType(),
                $this->entityTypeManager->getStorage($file_entity->getEntityTypeId())
              );
            $operations['data'] = [
              '#type' => 'operations',
              '#links' => $entityListBuilder->getOperations($file_entity),
            ];
            $name = $file_entity->label();
          }
        }

        $rows[] = [
          'preview' => [
            'data' => $preview,
          ],
          'name' => [
            'data' => [
              '#type' => 'link',
              '#title' => $name,
              '#url' => $url,
            ],
          ],
          'modified' => $this->dateFormatter->format(filemtime($full_path), 'short'),
          'operations' => [
            'data' => $operations,
          ],
        ];
      }

      $form['files'] = [
        '#type' => 'table',
        '#header' => [
          'preview' => $this->t('Preview'),
          'name' => $this->t('Name'),
          'modified' => $this->t('Modified'),
          'operations' => $this->t('Operations'),
        ],
        '#rows' => $rows,
        '#empty' => $this->t('No files found.'),
      ];
    }
    else {
      $form['files']['error'] = [
        '#markup' => $this->t('Directory does not exist.'),
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $values = $form_state->getValues();

    // Replace path if present private, temporary or asset in default directory.
    if (isset($values['path'])) {
      $path = $values['path'];

      $path = $this->destinationManager->normalizeDirectoryPath($path);

      $values['path'] = $path;
    }

    // Validate the directory path.
    if (empty($values['path'])) {
      $form_state->setErrorByName('path', $this->t('The directory path cannot be empty.'));
    }
    elseif (preg_match('/^[a-zA-Z0-9\-_]+$/', $values['path'])) {
      $form_state->setErrorByName('path', $this->t('The directory path can only contain letters, numbers, underscores, dashes, and slashes.'));
    }
    elseif (file_exists($values['path'])) {
      $form_state->setErrorByName('path', $this->t('The directory path already exists.'));
    }

    // Verify if directory exist and if can be created.
    if (!is_dir($values['path'])) {
      $result = $this->fileSystem->prepareDirectory($path, FileSystemInterface::CREATE_DIRECTORY || FileSystemInterface::MODIFY_PERMISSIONS || FileExists::Replace);
      if (!$result) {
        // Accurate error message.
        $form_state->setErrorByName('path', $this->t('The directory could not be created. Please check the permissions.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $directory = $this->entity;
    /**
     * @var \Drupal\advanced_file_destination\Entity\AdvancedFileDestinationDirectory $directory
     */

    // Set the values from the form to the entity.
    $values = $form_state->getValues();

    // Replace path if present private, temporary or asset in default directory.
    if (isset($values['path'])) {
      $values['path'] = $this->normalizeWrapper($values['path']);
    }
    if (isset($values['name'])) {
      $values['name'] = ucfirst($values['name']);
    }
    // Check if the entity type supports revisions first.
    if ($directory->getEntityType()->hasKey('revision') && isset($values['revision'])) {
      $directory->setNewRevision($values['revision']);
      if ($values['revision'] && isset($values['revision_log'])) {
        $directory->setRevisionLogMessage($values['revision_log']);
        $directory->setRevisionUserId($this->currentUser->id());
      }
    }

    // Process roles (filter out unchecked roles).
    if (isset($values['roles']) && is_array($values['roles'])) {
      $roles = array_filter(
            $values['roles'], function ($item) {
                return !empty($item);
            }
        );
      $directory->setRoles(array_keys($roles));
    }

    // Ensure status is properly set.
    $directory->setStatus($values['status']);

    // Save the entity.
    $status = $directory->save();

    // Set success message and redirect.
    $message = $status === SAVED_NEW
        ? $this->t('Created directory %label.', ['%label' => $directory->label()])
        : $this->t('Updated directory %label.', ['%label' => $directory->label()]);

    $this->messenger->addStatus($message);

    // Redirect to the directory list.
    $form_state->setRedirect('entity.afd_directory.collection');
    return $status;
  }

  /**
   * Determines if a file is an image.
   *
   * @param string $uri
   *   The URI of the file.
   *
   * @return bool
   *   TRUE if the file is an image, FALSE otherwise.
   */
  protected function isImage($uri) {
    $image = $this->imageFactory->get($uri);
    return $image->isValid();
  }

  /**
   * Helper function to get file entity by URI.
   *
   * @param string $uri
   *   The URI of the file.
   *
   * @return \Drupal\file\FileInterface|false
   *   A file entity or FALSE if not found.
   */
  protected function getFileEntity($uri) {
    try {
      $files = $this->entityTypeManager
        ->getStorage('file')
        ->loadByProperties(['uri' => $uri]);

      return reset($files);
    }
    catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    $actions = parent::actions($form, $form_state);

    // Ensure the submit button has the right properties.
    $actions['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#submit' => ['::submitForm', '::save'],
      '#button_type' => 'primary',
      '#weight' => 5,
    ];

    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Call parent submitForm which builds the entity from form values.
    parent::submitForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  protected function copyFormValuesToEntity(EntityInterface $entity, array $form, FormStateInterface $form_state) {
    // Get all values from form state.
    $values = $form_state->getValues();

    // Extract and remove it before parent method processes values.
    if (isset($values['revision'])) {
      $create_revision = $values['revision'];
      unset($values['revision']);

      // Handle revision flag separately using proper methods.
      if ($entity->getEntityType()->hasKey('revision')) {
        // Check if entity supports the revision methods before calling them.
        if (method_exists($entity, 'setNewRevision')) {
          $entity->setNewRevision($create_revision);

          if ($create_revision && isset($values['revision_log'])) {
            if (method_exists($entity, 'setRevisionLogMessage')) {
              $entity->setRevisionLogMessage($values['revision_log']);
            }

            if (method_exists($entity, 'setRevisionUserId')) {
              $entity->setRevisionUserId($this->currentUser->id());
            }
          }
        }
      }
    }

    // Replace path if present private, temporary or asset in default directory.
    if (isset($values['path'])) {
      $values['path'] = $this->normalizeWrapper($values['path']);
    }

    // Filter values to only include actual field names.
    $field_names = array_keys($entity->getFieldDefinitions());
    $filtered_values = array_intersect_key($values, array_flip($field_names));

    // Let parent handle the entity fields that actually exist.
    foreach ($filtered_values as $key => $value) {
      $entity->set($key, $value);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function normalizeWrapper($path) {
    return $this->destinationManager->normalizeDirectoryPath($path);
  }

}
