<?php

namespace Drupal\advanced_file_destination\Plugin\Field\FieldWidget;

use Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager;
use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\ElementInfoManagerInterface;
use Drupal\file\Plugin\Field\FieldWidget\FileWidget;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'advanced_file_destination' widget.
 *
 * @FieldWidget(
 *   id = "advanced_file_destination",
 *   label = @Translation("File with destination selection"),
 *   field_types = {
 *     "file",
 *     "image"
 *   }
 * )
 */
class AdvancedFileDestinationWidget extends FileWidget {

  /**
   * The advanced file destination manager.
   *
   * @var \Drupal\advanced_file_destination\Service\AdvancedFileDestinationManager
   */
  protected $destinationManager;

  /**
   * The element info manager.
   *
   * @var \Drupal\Core\Render\ElementInfoManagerInterface
   */
  protected $elementInfoManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * Constructs an AdvancedFileDestinationWidget object.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, ElementInfoManagerInterface $element_info, AdvancedFileDestinationManager $destination_manager, FileSystemInterface $file_system) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings, $element_info);
    $this->destinationManager = $destination_manager;
    $this->elementInfoManager = $element_info;
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
          $plugin_id,
          $plugin_definition,
          $configuration['field_definition'],
          $configuration['settings'],
          $configuration['third_party_settings'],
          $container->get('element_info'),
          $container->get('advanced_file_destination.manager'),
          $container->get('file_system')
      );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    // First get the original element from the parent class.
    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    // Get current directory - check multiple sources in order.
    $current_directory = NULL;
    // Check if the form state has an adf_instance_id.
    $adf_instance_id = uniqid('adf_instance_id_' . $this->fieldDefinition->getName() . time() . '_', TRUE);
    if (!$form_state->getValue('adf_instance_id') && empty($form_state->getValue('adf_instance_id'))) {
      $form_state->setValue('adf_instance_id', $adf_instance_id);
      $form_state->setValue(['advanced_file_destination', 'adf_instance_id'], $adf_instance_id);
    }
    $form['adf_instance_id'] = [
      '#type' => 'hidden',
      '#attributes' => ['class' => ['adf_instance_id'], 'id' => 'adf-instance-id'],
      '#default_value' => $form_state->getValue('adf_instance_id') ?? $adf_instance_id,
    ];
    // 1. Check form state first.
    $current_directory = $form_state->get(['file_destination', $this->fieldDefinition->getName(), $delta]);

    // 2. Check service if not in form state.
    if (!$current_directory) {
      $current_directory = $this->destinationManager->getDestinationDirectory(Html::escape($form_state->getValue('adf_instance_id')));
    }

    // 3. Fallback to field settings if still not found.
    if (!$current_directory) {
      $field_settings = $this->fieldDefinition->getSettings();
      if (!empty($field_settings['file_directory'])) {
        $scheme = !empty($field_settings['uri_scheme']) ? $field_settings['uri_scheme'] : 'public';
        $current_directory = $scheme . '://' . trim($field_settings['file_directory'], '/');
      }
    }

    // Special case handling for public:// directory.
    if ($current_directory == 'public://' || $current_directory == 'public:///'
          || $current_directory == 'public://public:/' || $current_directory == 'public://public://'
      ) {
      $current_directory = 'public://';
    }

    // Generate a unique wrapper ID specifically for the directory dropdown.
    $directory_wrapper_id = Html::getUniqueId('directory-dropdown-' . $this->fieldDefinition->getName() . '-' . $delta);
    // Set a weight for the directory selection element to control its position.
    $positon = $this->destinationManager->getWidgetDirectoryPosition();
    if ($positon === 'before') {
      $weight = -100; // Place before the upload element.
    }
    else {
      $weight = -100; // Place after the upload element.
    }
    $element['#weight'] = $weight;
    // Add directory selection element.
    $element['directory_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => $directory_wrapper_id],
    ];

    $element['directory_wrapper']['directory'] = [
      '#type' => 'select',
      '#title' => $this->t('Destination folder'),
      '#options' => $this->destinationManager->getAvailableDirectories($current_directory),
      '#default_value' => $current_directory,
      '#ajax' => [
        'callback' => [$this, 'updateDirectoryCallback'],
        'wrapper' => $directory_wrapper_id,
        'event' => 'change',
        'progress' => ['type' => 'throbber'],
      ],
      '#element_validate' => [[$this, 'validateDirectory']],
    ];

    // Set upload location if we have a directory.
    if ($current_directory) {
      // Set upload location in all possible places.
      $element['#upload_location'] = $current_directory;

      // For Drupal 8/9+ compatibility, also set on upload elements.
      if (isset($element['upload'])) {
        $element['upload']['#upload_location'] = $current_directory;
      }

      // Store in form state for persistence between form rebuilds.
      $form_state->set(['file_destination', $items->getName(), $delta], $current_directory);

      // Set static variable that Drupal core uses for file uploads.
      $settings['upload_location'] = $current_directory;
    }

    // Add hidden field to store the directory value submitted with the form.
    $element['selected_directory'] = [
      '#type' => 'hidden',
      '#value' => $current_directory,
      '#attributes' => ['class' => ['file-destination-path']],
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    // Set the correct upload location before the parent class processes values.
    foreach ($values as $delta => $value) {
      if (!empty($value['selected_directory'])) {
        // Ensure the upload_location is set for each file.
        $values[$delta]['upload_location'] = $value['selected_directory'];

        // Store file module to use.
        $settings['upload_location'] = $value['selected_directory'];
      }
    }

    return parent::massageFormValues($values, $form, $form_state);
  }

  /**
   * Ajax callback for directory selection.
   */
  public function updateDirectoryCallback(array $form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement();
    $parents = array_slice($trigger['#array_parents'], 0, -1);

    // Get the directory wrapper element.
    $wrapper_element = NestedArray::getValue($form, $parents);

    // Get the selected directory.
    $directory = $trigger['#value'];

    // Special case handling for public directories.
    if ($directory == 'public://' || $directory == 'public:///'
          || $directory == 'public://public:/' || $directory == 'public://public://'
      ) {
      $directory = 'public://';
    }

    if ($directory) {
      // Normalize the directory path.
      $directory = $this->destinationManager->normalizeDirectoryPath($directory);
      // Update service.
      $this->destinationManager->setDestinationDirectory(Html::escape($directory), Html::escape($form_state->getValue('adf_instance_id')));

      // Get the field element.
      $field_parents = array_slice($parents, 0, -1);
      $field_element = NestedArray::getValue($form, $field_parents);

      // Set the selected directory in the hidden field.
      if (isset($field_element['selected_directory'])) {
        $field_element['selected_directory']['#value'] = $directory;
      }

      // Update upload location.
      if (isset($field_element['#upload_location'])) {
        $field_element['#upload_location'] = $directory;
      }

      if (isset($field_element['upload']['#upload_location'])) {
        $field_element['upload']['#upload_location'] = $directory;
      }

      // Store in form state.
      $field_name = $this->fieldDefinition->getName();
      $delta = $field_element['#delta'];
      $form_state->set(['file_destination', $field_name, $delta], $directory);

      // Set the static value that Drupal uses.
      $settings['upload_location'] = $directory;
    }

    // Return only the wrapper element to prevent duplicating the entire field.
    return $wrapper_element;
  }

  /**
   * Validates the selected directory.
   */
  public function validateDirectory($element, FormStateInterface $form_state, $form) {
    $directory = $element['#value'];
    if ($directory) {

      // Fix the root directory cases only, preserving subdirectories.
      if ($directory === 'public:///' || $directory === 'public://public:/' || $directory === 'public:/' || $directory === 'public:/ ' || $directory === 'public://public://') {
        $directory = 'public://';
      }

      // Only update the form value if we modified the directory.
      if ($directory !== $element['#value']) {
        $form_state->setValueForElement($element, $directory);
      }

      if (!$this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
        $form_state->setError($element, $this->t('Selected directory is not writable.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validate($element, FormStateInterface $form_state, $form) {
    // Make sure the upload destination is properly set before validation.
    $upload_location = $element['selected_directory']['#value'] ?? NULL;
    if ($upload_location) {
      if (isset($element['#upload_location'])) {
        $element['#upload_location'] = $upload_location;
      }
      if (isset($element['upload']['#upload_location'])) {
        $element['upload']['#upload_location'] = $upload_location;
      }

      // Set the static value that Drupal uses.
      $settings['upload_location'] = $upload_location;
    }
  }

  /**
   * Get the current directory for this field item.
   */
  protected function getCurrentDirectory(FieldItemListInterface $items, $delta) {
    // For existing files, get their current directory.
    if (!$items->isEmpty() && isset($items[$delta]->entity)) {
      $file = $items[$delta]->entity;
      if ($file) {
        $uri = $file->getFileUri();
        if ($uri) {
          return dirname($uri);
        }
      }
    }

    // Get directory from widget state.
    $form_state = $items->getEntity()->form_state;
    if ($form_state) {
      $widget_state = static::getWidgetState($form_state->getCompleteForm()['#parents'], $this->fieldDefinition->getName(), $form_state);
      if (isset($widget_state[$delta]['directory'])) {
        return $widget_state[$delta]['directory'];
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function extractFormValues(FieldItemListInterface $items, array $form, FormStateInterface $form_state) {
    $field_name = $this->fieldDefinition->getName();
    $widget_state = static::getWidgetState($form['#parents'], $field_name, $form_state);

    // Store directory changes for each delta.
    foreach ($items as $delta => $item) {
      if (isset($widget_state[$delta]['directory']) && isset($item->entity)) {
        $form_state->set(
              [
                'advanced_file_destination',
                $field_name,
                $delta,
              ], [
                'file_id' => $item->entity->id(),
                'new_directory' => $widget_state[$delta]['directory'],
              ]
          );
      }
    }

    return parent::extractFormValues($items, $form, $form_state);
  }

}
