<?php

namespace Drupal\Tests\advanced_file_destination\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Tests\TestFileCreationTrait;
use Drupal\Tests\field_ui\Traits\FieldUiTestTrait;
use Drupal\Tests\file\Functional\FileFieldCreationTrait;

/**
 * Tests the Advanced File Destination UI functionality.
 *
 * @group advanced_file_destination
 */
class AdvancedFileDestinationBrowserTest extends WebDriverTestBase {

  use TestFileCreationTrait;
  use FieldUiTestTrait;
  use FileFieldCreationTrait;

  /**
   * Modules to enable for this test.
   *
   * @var array
   */
  protected static $modules = [
    'node',
    'file',
    'field_ui',
    'media',
    'advanced_file_destination',
  ];

  /**
   * {@inheritdoc}
   */
  protected $strictConfigSchema = FALSE;

  /**
   * The theme to use for this test.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * Set up the test environment.
   *
   * Creates test with required permissions and content type with file field.
   */
  protected function setUp(): void {
    parent::setUp();

    // Create test content type.
    $this->drupalCreateContentType(['type' => 'article']);

    // Create test user.
    $admin_user = $this->drupalCreateUser(
          [
            'access content',
            'access administration pages',
            'administer site configuration',
            'administer content types',
            'administer node fields',
            'administer nodes',
          // This was causing the issue - needed to create content type first.
            'create article content',
            'access advanced file destination',
            'create advanced file destination directories',
          ]
      );
    $this->drupalLogin($admin_user);

    // Create file field.
    $this->createFileField('field_test_file', 'node', 'article');
  }

  /**
   * Tests the file upload process with custom directory selection.
   *
   * Verifies that files are properly uploaded to the selected destination
   * directory when using the Advanced File Destination module.
   */
  public function testFileUploadWithDirectorySelection() {
    $page = $this->getSession()->getPage();

    // Create test directories.
    $dir = 'public://';
    $this->container->get('file_system')->prepareDirectory($dir, FileSystemInterface::CREATE_DIRECTORY);

    // Navigate to node creation page.
    $this->drupalGet('node/add/article');

    // Select directory.
    $page->selectFieldOption('advanced_file_destination[directory]', 'public://');
    $this->waitForAjax();

    // Upload file.
    $file = current($this->getTestFiles('text'));
    $page->attachFileToField('files[field_test_file_0]', \Drupal::service('file_system')->realpath($file->uri));
    $this->waitForAjax();

    // Verify file location by waiting for element to appear.
    // Wait for the file element to become visible.
    $page->waitFor(
          10, function () use ($page) {
              return $page->find('css', '.file--mime-text-plain');
          }
      );

    $uploaded_file = $page->find('css', '.js-form-managed-file a');

    if (empty($uploaded_file)) {
      throw new \Exception('File upload element was not found after AJAX completion');
    }

    $href = $uploaded_file->getAttribute('href');
    $this->assertNotEmpty($href, 'File upload element href is not empty');
    $this->assertStringContainsString('sites', $href, 'File upload element href contains public://');
  }

  /**
   * Tests the directory selection process.
   *
   * Verifies that the directory selection process works as expected.
   * Verifies that the directory can not be changed from browser.
   */
  public function testDirectorySelection() {
    $page = $this->getSession()->getPage();

    // Create test directories.
    $dir = 'public://test-directory/';

    \Drupal::service('advanced_file_destination.manager')->createDirectory($dir, 'Test');

    // Navigate to node creation page.
    $this->drupalGet('node/add/article');

    $val_dir = 'public://test-directory/';

    // Select directory.
    $page->selectFieldOption('advanced_file_destination[directory]', 'Test');
    $this->waitForAjax();

    // Verify that the selected directory is set in the form.
    $selected_directory = $page->findField('advanced_file_destination[directory]')->getValue();
    $this->assertEquals($val_dir, $selected_directory, 'Selected directory is set correctly.');

    // Verify that the directory is in the list of existing directories.
    $directory_list = $page->findAll('css', '.js-advanced-file-destination-select option');
    $this->assertNotEmpty($directory_list, 'Directory list is not empty.');
    $directory_found = FALSE;
    foreach ($directory_list as $directory) {
      if (strpos($directory->getAttribute('value'), $val_dir) !== FALSE) {
        $directory_found = TRUE;
        break;
      }
    }
    $this->assertTrue($directory_found, 'Selected directory is in the list of existing directories.');

    // Verify that the directory cannot be changed from the browser.
    $this->waitForAjax();
    $new_selected_directory = $page->findField('advanced_file_destination[directory]')->getValue();
    $this->assertEquals($val_dir, $new_selected_directory, 'Selected directory is not changed from the browser.');

    // Verify that the directory is still in the list of existing directories.
    $directory_list = $page->findAll('css', '.js-advanced-file-destination-select option');
    $this->assertNotEmpty($directory_list, 'Directory list is not empty.');
    $directory_found = FALSE;
    foreach ($directory_list as $directory) {
      if (strpos($directory->getAttribute('value'), $val_dir) !== FALSE) {
        $directory_found = TRUE;
        break;
      }
    }
    $this->assertTrue($directory_found, 'Selected directory is still in the list of existing directories.');
    // Verify that the file is uploaded in the correct directory.
    $test_file = current($this->getTestFiles('text'));
    $page->attachFileToField('files[field_test_file_0]', \Drupal::service('file_system')->realpath($test_file->uri));
    $this->waitForAjax();
    // Wait for the file element to become visible.
    $page->waitFor(
          10, function () use ($page) {
              return $page->find('css', '.file--mime-text-plain');
          }
      );

    $uploaded_file = $page->find('css', '.js-form-managed-file a');
    if (empty($uploaded_file)) {
      throw new \Exception('File upload element was not found after AJAX completion');
    }
    $href = $uploaded_file->getAttribute('href');
    $this->assertNotEmpty($href, 'File upload element href is not empty');
    $this->assertStringContainsString('sites', $href, 'File upload element href contains public://');
    $this->assertStringContainsString(str_replace('public://', '', $dir), $href, 'File upload element href contains the selected directory');
    $this->assertStringNotContainsString('new_directory', $href, 'File upload element href does not contain the new directory');
  }

  /**
   * Tests security for directory injection via AJAX.
   *
   * Verifies that the module properly validates directories and prevents
   * uploading files to non-existing directories via AJAX request manipulation.
   */
  public function testDirectoryInjectionSecurity() {
    $page = $this->getSession()->getPage();

    // Create a valid directory.
    $valid_dir = 'public://test_directory/';
    \Drupal::service('advanced_file_destination.manager')->createDirectory($valid_dir, 'Test');

    // Create a non-existing directory path for injection attempt.
    $injected_dir = 'public://injected_directory/';

    // Make sure the injected directory does not exist.
    $file_system = \Drupal::service('file_system');
    if ($file_system->prepareDirectory($injected_dir, FileSystemInterface::CREATE_DIRECTORY)) {
      $file_system->deleteRecursive($injected_dir);
    }

    // Navigate to node creation page.
    $this->drupalGet('node/add/article');

    // Select the valid directory using the form.
    $page->selectFieldOption('advanced_file_destination[directory]', 'Test');
    $this->waitForAjax();

    // Get the instance ID from the form.
    $afd_instance_id = $page->find('css', '#adf-instance-id')->getValue();

    // Use executeScript to send a custom XMLHttpRequest to simulate the AJAX call.
    $script = <<<JS
      (function() {
        var xhr = new XMLHttpRequest();
        xhr.open('POST', '/advanced-file-destination/ajax/update-state', false);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
        xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');

        var params = 'adf_instance_id=' + encodeURIComponent('$afd_instance_id') +
                    '&directory=' + encodeURIComponent('$injected_dir') +
                    '&parent_directory=' + encodeURIComponent('public://');

        xhr.send(params);
        return xhr.status;
      })();
    JS;

    // Execute the JavaScript to send the injection attempt.
    $this->getSession()->executeScript($script);

    // Now try to upload a file.
    $this->drupalGet('node/add/article');
    $test_file = current($this->getTestFiles('text'));
    $page->attachFileToField('files[field_test_file_0]', $file_system->realpath($test_file->uri));
    $this->waitForAjax();

    // Wait for the file element to become visible.
    $page->waitFor(
      10, function () use ($page) {
        return $page->find('css', '.file--mime-text-plain');
      }
    );

    // Get the link to the uploaded file.
    $uploaded_file = $page->find('css', '.js-form-managed-file a');
    $this->assertNotEmpty($uploaded_file, 'File upload element was found after AJAX completion');

    $href = $uploaded_file->getAttribute('href');
    $this->assertNotEmpty($href, 'File upload element href is not empty');

    // Verify the file was NOT uploaded to the injected directory.
    $this->assertStringNotContainsString(
      str_replace('public://', '', $injected_dir),
      $href,
      'File was not uploaded to the injected directory path'
    );

    // Get current YYYY-MM.
    $current_date = date('Y-m');

    // Verify it was uploaded to a valid directory instead.
    // Either the default directory or the last valid selected directory.
    $this->assertStringContainsString(
      'files/' . $current_date,
      $href,
      'File was uploaded to a valid directory path'
    );

    // Finally, check that the injected directory was not created.
    $this->assertFalse(
      file_exists(\Drupal::service('file_system')->realpath($injected_dir)),
      'The injected directory was not created on the filesystem'
    );
  }

  /**
   * Helper method to wait for AJAX requests to complete.
   */
  protected function waitForAjax() {
    $condition = "(0 === jQuery.active)";
    $this->assertJsCondition($condition, 1000);
  }

}
