<?php

namespace Drupal\Tests\advanced_file_destination\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\file\Entity\File;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Tests\user\Traits\UserCreationTrait;

/**
 * Tests the Advanced File Destination functionality.
 *
 * @group advanced_file_destination
 */
class AdvancedFileDestinationTest extends KernelTestBase {

  use UserCreationTrait;
  /**
   * The modules to enable for this test.
   *
   * @var array
   */
  protected static $modules = [
    'system',
    'file',
    'user',
    'advanced_file_destination',
  ];

  /**
   * The destination manager service.
   *
   * @var \Drupal\advanced_file_destination\DestinationManagerInterface
   */
  protected $destinationManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);
    $this->installSchema('user', ['users_data']);
    $this->installEntitySchema('afd_directory');
    // Setting current user.
    $permissions = [
      'administer site configuration',
      'administer advanced file destination',
      'access advanced file destination',
      'create advanced file destination directories',
      'create advanced file destination',
    ];
    $this->setUpCurrentUser(['uid' => 1], $permissions);

    $this->destinationManager = $this->container->get('advanced_file_destination.manager');
    $this->fileSystem = $this->container->get('file_system');

    // Create test directories.
    $test_directories = [
      'public://test_directory',
      'public://images',
      'public://documents',
    ];

    // Create directories entities if they do not exist.
    foreach ($test_directories as $directory) {
      if (!$this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY)) {
        $this->fail('Failed to create directory: ' . $directory);
      }
      $this->destinationManager->createDirectory($directory);
    }
  }

  /**
   * Tests if files are moved to the configured destination directory.
   */
  public function testFileUploadLocation() {
    \Drupal::request()->query->set('adf_instance_id', 'instance_id');
    // Set the destination directory.
    $test_directory = 'public://test_directory/';
    $this->destinationManager->setDestinationDirectory($test_directory, 'instance_id');

    // Create a test file.
    $filename = $this->randomMachineName() . '.txt';
    $contents = $this->randomString();
    $original_uri = $test_directory . $filename;
    file_put_contents($original_uri, $contents);

    // Create file entity.
    $file = File::create(
          [
            'uri' => $original_uri,
            'filename' => $filename,
            'status' => 1,
          ]
      );
    $file->save();

    // Trigger pre-save operations.
    advanced_file_destination_file_presave($file);

    // Verify file location.
    $expected_uri = $test_directory . $filename;
    $this->assertEquals($expected_uri, $file->getFileUri(), 'File was not moved to the correct directory.');
    $this->assertTrue(file_exists($expected_uri), 'File does not exist in the target location.');
  }

  /**
   * Tests the directory selection functionality.
   */
  public function testDirectorySelection() {
    // Test initial directory selection.
    $test_directory = 'public://images';
    \Drupal::request()->query->set('adf_instance_id', 'instance_id');
    $this->destinationManager->setDestinationDirectory($test_directory, 'instance_id');
    // Normalize paths by ensuring trailing slashes are consistent for comparison.
    $expected = rtrim($test_directory, '/');
    $actual = rtrim($this->destinationManager->getDestinationDirectory('instance_id'));
    $this->assertEquals($expected, $actual);

    // Test directory change.
    $new_directory = 'public://documents';
    $this->destinationManager->setDestinationDirectory($new_directory, 'instance_id');
    $expected = rtrim($new_directory, '/');
    $actual = rtrim($this->destinationManager->getDestinationDirectory('instance_id'));
    $this->assertEquals($expected, $actual);
  }

}
