const { blocks, data, element, components, editor } = wp;
const { registerBlockType } = blocks;
const { dispatch, select } = data;
const { Fragment, useState, useEffect  } = element;
const { PanelBody, BaseControl, Icon, RangeControl, IconButton, Toolbar, SelectControl, CheckboxControl, Grid} = components;
const { InnerBlocks, RichText, InspectorControls, PanelColorSettings, MediaUpload, MediaUploadCheck, BlockControls } = editor;
const __ = Drupal.t;

const el = wp.element.createElement;

// Section Break H2
const sectionBreakH2 = {
  title: __('Section Break (H2)'),
  description: __('H2 is used as Section Heading'),
  icon: 'heading',
  attributes: {
      title: {
          type: 'string',
      },

      backgroundflag: {
        type: 'string',
      },
  },



  edit({ attributes, setAttributes, isSelected }) {
      const { title, backgroundflag } = attributes;
      return (
        <Fragment>

          <RichText
                identifier="title"
                tagName="h2"
                value={title}
                placeholder={__('Section Break (H2)')}
                onChange={title => {
                    setAttributes({
                        title: title,
                    });
                }}
                onSplit={() => null}
                unstableOnSplit={() => null}
            />

          <InspectorControls>
          <PanelBody title={ __('Block Settings') }>


            <SelectControl
            label="Background Color"
            value={ backgroundflag }
            options={ [
                { label: 'White', value: 'white_background' },
                { label: 'Grey', value: 'grey_background' },
                { label: 'Red', value: 'red_background' },
            ] }
            onChange={backgroundflag => {
                setAttributes({
                  backgroundflag: backgroundflag,
                });
            }}
          />
          </PanelBody>

        </InspectorControls>
        </Fragment>
      );
  },

  save({ attributes }) {
    const { title, backgroundflag} = attributes;

    return (
      <div className={`${(typeof backgroundflag!==undefined) ? backgroundflag : "white_background"}`}>
        <div className="wrapper">
          {title && (<h2 dangerouslySetInnerHTML={{__html: title}} />)}
        </div>
      </div>
    );
  },
};

// Custom Heading
const customHeading = {
  title: __('Custom Heading'),
  description: __('Custom heading block with background options'),
  icon: 'heading',
  attributes: {
      title: {
          type: 'string',
      },

      backgroundflag: {
        type: 'string',
      },
  },



  edit({ attributes, setAttributes, isSelected }) {
      const { title, backgroundflag } = attributes;
      return (
        <Fragment>

          <RichText
                identifier="title"
                tagName="h1"
                value={title}
                placeholder={__('Custom Heading H1')}
                onChange={title => {
                    setAttributes({
                        title: title,
                    });
                }}
                onSplit={() => null}
                unstableOnSplit={() => null}
            />

          <InspectorControls>
          <PanelBody title={ __('Block Settings') }>


            <SelectControl
            label="Background Color"
            value={ backgroundflag }
            options={ [
                { label: 'White', value: 'white_background' },
                { label: 'Grey', value: 'grey_background' },
                { label: 'Red', value: 'red_background' },
            ] }
            onChange={backgroundflag => {
                setAttributes({
                  backgroundflag: backgroundflag,
                });
            }}
          />
          </PanelBody>

        </InspectorControls>
        </Fragment>
      );
  },

  save({ attributes }) {
    const { title, backgroundflag} = attributes;

    return (
      <div className={`${(typeof backgroundflag!==undefined) ? backgroundflag : "white_background"}`}>
        <div className="wrapper">
          {title && (<h1 dangerouslySetInnerHTML={{__html: title}} />)}
        </div>
      </div>
    );
  },
};

// Highlighted text
const highlightedTexticon = el('svg', { width: 20, height: 20, viewBox:"0 0 512 512" },
  el('path', { d: "M451.4692,125.4041,380.5656,54.5016a13.1093,13.1093,0,0,0-17.7457-.7327L164.9985,221.3722,284.6018,340.9744l167.6-197.8225A13.11,13.11,0,0,0,451.4692,125.4041Z" } ),
  el('rect', { width: 172.7608, height: 41.0143, x:171.2274, y:227.8569, transform:"translate(-166.0416 227.6148) rotate(-45)"} ),
  el('path', { d: "M94.5521,363.7306,142.2413,411.42c18.5232-11.008,40.65-15.58,63.1619-17.3911l-93.46-93.46C110.1316,323.08,105.56,345.2073,94.5521,363.7306Z" } ),
  el('path', { d: "M58.6143,403.6279a6.5743,6.5743,0,0,0-1.2732,7.5013l22.3748,46.4866a6.5746,6.5746,0,0,0,10.5723,1.7965l31.4111-31.4111L77.9706,384.2726Z" } )
);
const highlightedText = {
  title: __('Highlighted text'),
  description: __(''),
  icon: highlightedTexticon,
  attributes: {
      title: {
          type: 'string',
      },
      textcopy: {
          type: 'string',
      },

      layoutoption: {
        type: 'string',
      },
  },



  edit({ attributes, setAttributes, isSelected }) {
      const { title, textcopy, layoutoption } = attributes;
      return (
        <Fragment>

          <RichText
                identifier="title"
                tagName="h3"
                value={title}
                placeholder={__('Highlighted Text Title')}
                onChange={title => {
                    setAttributes({
                        title: title,
                    });
                }}
                onSplit={() => null}
                unstableOnSplit={() => null}
            />
          <RichText
                identifier="Text Copy"
                tagName="p"
                value={textcopy}
                placeholder={__('Highlighted Text Copy')}
                onChange={textcopy => {
                    setAttributes({
                      textcopy: textcopy,
                    });
                }}
                onSplit={() => null}
                unstableOnSplit={() => null}
            />

          <InspectorControls>
          <PanelBody title={ __('Block Settings') }>


            <SelectControl
            label="layout Options"
            value={ layoutoption }
            options={ [
                { label: '⅓ title text + ⅔ body text side by side', value: 'large_text' },
                { label: '⅔ title text + ⅓ body text side by side', value: 'large_title' },
                { label: 'Stacked title and body text', value: 'stacked' },
            ] }
            onChange={layoutoption => {
                setAttributes({
                  layoutoption: layoutoption,
                });
            }}
          />
          </PanelBody>

        </InspectorControls>
        </Fragment>
      );
  },

  save({ attributes }) {
    const { title, textcopy, layoutoption} = attributes;

    return (
      <div className={`${(typeof layoutoption!==undefined) ? layoutoption : "large_text"}`}>
        <div className="wrapper">
          {title && (<div className="title h3" dangerouslySetInnerHTML={{__html: title}} />)}
          {textcopy && (<p dangerouslySetInnerHTML={{__html: textcopy}} />)}
        </div>
      </div>
    );
  },
};


// Image Grid
const imgGridicon = el('svg', { width: 20, height: 20 },
  el('path', { d: "M0,4 L4,4 L4,0 L0,0 L0,4 L0,4 Z M6,16 L10,16 L10,12 L6,12 L6,16 L6,16 Z M0,16 L4,16 L4,12 L0,12 L0,16 L0,16 Z M0,10 L4,10 L4,6 L0,6 L0,10 L0,10 Z M6,10 L10,10 L10,6 L6,6 L6,10 L6,10 Z M12,0 L12,4 L16,4 L16,0 L12,0 L12,0 Z M6,4 L10,4 L10,0 L6,0 L6,4 L6,4 Z M12,10 L16,10 L16,6 L12,6 L12,10 L12,10 Z M12,16 L16,16 L16,12 L12,12 L12,16 L12,16 Z" } )
);
const imageGrid_TEMPLATE = [
  [ 'core/columns', {}, [
    [ 'core/column', {}, [
        [ 'core/image' ],
    ] ],
    [ 'core/column', {}, [
      [ 'core/image' ],
    ] ],
    [ 'core/column', {}, [
      [ 'core/image' ],
    ] ],
  ] ],
  [ 'core/columns', {}, [
    [ 'core/column', {}, [
        [ 'core/image' ],
    ] ],
    [ 'core/column', {}, [
      [ 'core/image' ],
    ] ],
    [ 'core/column', {}, [
      [ 'core/image' ],
    ] ],
  ] ]
];
const imageGrid_ALLOWED_BLOCKS = [ 'core/image' ];
const imageGrid = {
  title: __('Image Grid'),
  description: __('Image Grid of six'),
  icon: imgGridicon,
  edit({ attributes, setAttributes, isSelected }) {

      return (
        <Fragment>
          <div className="image-grid-wrapper">
            <div className="image-grid">
            <InnerBlocks template={ imageGrid_TEMPLATE } templateLock="all" allowedBlocks={ imageGrid_ALLOWED_BLOCKS } />
            </div>
            </div>
        </Fragment>
      );
  },

  save({ attributes }) {
    return (
      <div className="image-grid-wrapper">
            <div className="image-grid">
      <InnerBlocks.Content />
      </div>
      </div>
    );
  },
};


// Custom Text Block
const CustomTextBlock = {
  title: __('Custom Text Block'),
  description: __(''),
  icon: 'menu',
  attributes: {
      title: {
          type: 'string',
      },

      backgroundflag: {
        type: 'string',
      },
  },
  edit({ attributes, setAttributes, isSelected }) {
    const { title, backgroundflag } = attributes;
      return (
        <Fragment>
            <RichText
                identifier="title"
                tagName="h3"
                value={title}
                placeholder={__('Text Block Title')}
                onChange={title => {
                    setAttributes({
                        title: title,
                    });
                }}
                onSplit={() => null}
                unstableOnSplit={() => null}
            />
            <InnerBlocks />

              <InspectorControls>
              <PanelBody title={ __('Block Settings') }>


                <SelectControl
                label="Background Color"
                value={ backgroundflag }
                options={ [
                    { label: 'White', value: 'white_background' },
                    { label: 'Grey', value: 'grey_background' },
                    { label: 'Red', value: 'red_background' },
                ] }
                onChange={backgroundflag => {
                    setAttributes({
                      backgroundflag: backgroundflag,
                    });
                }}
              />
              </PanelBody>

            </InspectorControls>


        </Fragment>
      );
  },

  save({ attributes }) {
    const { title, backgroundflag} = attributes;

    return (
      <div className={`${(typeof backgroundflag!==undefined) ? backgroundflag : "white_background"}`}>
        <div className="wrapper">
          {title && (<div className="title h3" dangerouslySetInnerHTML={{__html: title}} />)}
          <InnerBlocks.Content />
        </div>
      </div>
    );
  },
};


// Text Image Slide
const textImageSlidericon = el('svg', { width: 20, height: 20 },
  el('path', { d: "M 0 3 L 0 4 L 1 4 L 1 14 L 0 14 L 0 15 L 2 15 L 2 3 L 0 3 z M 3 3 L 3 15 L 17 15 L 17 3 L 3 3 z M 18 3 L 18 15 L 20 15 L 20 14 L 19 14 L 19 4 L 20 4 L 20 3 L 18 3 z M 4 4 L 16 4 L 16 14 L 4 14 L 4 4 z M 5 17 L 5 18 L 7 18 L 7 17 L 5 17 z M 9 17 L 9 18 L 11 18 L 11 17 L 9 17 z M 13 17 L 13 18 L 15 18 L 15 17 L 13 17 z" } )
);
const textImageSlide_ALLOWED_BLOCKS = [ 'core/heading', 'core/paragraph', 'core/button', 'core/image' ];
const textImageSlide_TEMPLATE = [
  [ 'core/columns', {}, [
    [ 'core/column', {}, [
        [ 'core/paragraph', {className:"slide-heading h3", fontSize: "Large", placeholder: "Slide Heading"} ],
        [ 'core/paragraph', {placeholder: "Slide Text"} ],
        [ 'core/button', {customBackgroundColor:"transparent"} ],
    ] ],
    [ 'core/column', {}, [
      [ 'core/image' ],
    ] ],
  ] ]
];
const TextImageSlide = {
  title: __('Text + image slide'),
  description: __('Single Slide of Text + Image'),
  icon: textImageSlidericon,
  parent: ['example/textimageslider-block'],
  edit({ attributes, setAttributes, isSelected }) {

      return (
        <Fragment>
            <InnerBlocks template={textImageSlide_TEMPLATE} allowedBlocks={textImageSlide_ALLOWED_BLOCKS} templateLock="all" />
        </Fragment>
      );
  },

  save({ attributes }) {
    return (
      <div>
        <div className="wrapper">
          <InnerBlocks.Content aria-roledescription="Text Image Slide" role="group" />
        </div>
      </div>
    );
  },
};


// Text Image Slider
const textImageSlider_ALLOWED_BLOCKS = [ 'example/textimageslide-block' ];
var textImageSlider_TEMPLATE = [
  [ 'example/textimageslide-block']
];
const TextImageSlider = {
  title: __('Text + image slider - Repeater'),
  description: __('Slider of Text + Image'),
  icon: textImageSlidericon,
  edit({ attributes, setAttributes, isSelected }) {
      return (
        <Fragment>
            <h2>Text + Image Slider</h2>
            <InnerBlocks allowedBlocks={textImageSlider_ALLOWED_BLOCKS} template={textImageSlider_TEMPLATE}  />
        </Fragment>
      );
  },

  save({ attributes }) {
    return (
      <div aria-roledescription="carousel" aria-label="Text Image Slider">
        <nav className="slider-navigation">
          <button className="prev" aria-controls="text-image-slider" data-slide="prev"> <span className="visually-hidden-focusable">Previous Slide</span> </button>
          <button className="next" aria-controls="text-image-slider" data-slide="next"> <span className="visually-hidden-focusable">Next Slide</span> </button>
        </nav>
        <div className="slider-wrapper" data-ride="carousel" id="text-image-slider">
          <InnerBlocks.Content aria-roledescription="Text Image Slide" role="group" />
        </div>
      </div>
    );
  },
};


// Featured media Block
const featuredMediaicon = el('svg', { width: 20, height: 20, fill:'#ffffff', stroke:"#000", },
  el('path', { d: "M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z" } ),
  el('circle', { cx:"12", cy:"13", r:"4" } ),
);
const featuredMedia_ALLOWED_BLOCKS = [ 'example/media-block' ];
const featuredMedia_TEMPLATE = [[ 'core/image', {align:"center"}]];
const CustomFeaturedMediaBlock = {
  title: __('Featured Media'),
  description: __(''),
  icon: featuredMediaicon,
  attributes: {
    widthflag: {
        type: 'string',
      },
  },
  edit({ attributes, setAttributes, isSelected }) {
    const {  widthflag } = attributes;
      return (
        <Fragment>
            <h2>Featured Media Block:</h2>
            <InnerBlocks allowedBlocks={featuredMedia_ALLOWED_BLOCKS} template={featuredMedia_TEMPLATE}  />

              <InspectorControls>
              <PanelBody title={ __('Block Settings') }>
                <SelectControl
                label="Width Option"
                value={ widthflag }
                options={ [
                    { label: 'Constrained', value: 'constrained' },
                    { label: 'Full Width', value: 'full_width' },
                ] }
                onChange={widthflag => {
                    setAttributes({
                      widthflag: widthflag,
                    });
                }}
              />
              </PanelBody>

            </InspectorControls>


        </Fragment>
      );
  },

  save({ attributes }) {
    const {  widthflag} = attributes;

    return (
      <div className={`${(typeof widthflag!==undefined) ? widthflag : "constrained"}`}>
        <div className="wrapper">
          <InnerBlocks.Content />
        </div>
      </div>
    );
  },
};

// Image Mosaic
/* const imageMosaic_icon = el('svg', { width: 20, height: 20, fill:'#ffffff', stroke:"#000", viewBox:"0 0 24 24" },
  el('path', { d: "M0 0h24v24H0z", fill:"none" } ),
  el('path', { d: "M16 21V10h5v10a1 1 0 0 1-1 1h-4zm-2 0H4a1 1 0 0 1-1-1V10h11v11zm7-13H3V4a1 1 0 0 1 1-1h16a1 1 0 0 1 1 1v4z" }),
); */

const featuredMedia_Sidebyside_TEMPLATE = [
  [ 'core/columns', {}, [
    [ 'core/column', {}, [
      [ 'core/image',{caption:" "} ],
    ] ],
    [ 'core/column', {}, [
      [ 'core/image',{caption:" "} ],
    ] ],
  ] ],
];
const featuredMedia_type1_TEMPLATE = [
  [ 'core/group', {}, [
    [ 'core/columns', {}, [
      [ 'core/column', {}, [
        [ 'core/image',{caption:" "} ],
      ] ],
      [ 'core/column', {}, [
        [ 'core/image',{caption:" "} ],
      ] ],
    ] ],
    [ 'core/image',{caption:" "} ],
  ] ],
];
const featuredMedia_type2_TEMPLATE = [
  [ 'core/group', {}, [
    [ 'core/image',{caption:" "} ],
    [ 'core/columns', {}, [
      [ 'core/column', {}, [
        [ 'core/image',{caption:" "} ],
      ] ],
      [ 'core/column', {}, [
        [ 'core/image',{caption:" "} ],
      ] ],
    ] ],
] ],
];

const featuredMedia_type3_TEMPLATE = [
  [ 'core/group', {}, [
    [ 'core/image',{caption:" "} ],
    [ 'core/columns', {}, [
      [ 'core/column', {}, [
        [ 'core/image',{caption:" "} ],
      ] ],
      [ 'core/column', {}, [
        [ 'core/image',{caption:" "} ],
      ] ],
    ] ],
    [ 'core/image',{caption:" "} ],
  ] ],
];

let key = 0;
const imageMosaicBlock = {
  title: __('Image Mosaic'),
  description: __(''),
  /* icon: imageMosaic_icon, */
  attributes: {
    layoutoption: {
        type: 'string',
    },
  },
  edit({ attributes, setAttributes, isSelected }) {
    const { layoutoption } = attributes;
    let initialLayout;
    //console.log("layoutoption: " + layoutoption);

    if(!layoutoption){
      setAttributes({
        layoutoption : "side_by_side",
      })
    }

    switch (layoutoption){
        case "side_by_side":
            initialLayout = featuredMedia_Sidebyside_TEMPLATE;
            break;
        case "type1":
            initialLayout = featuredMedia_type1_TEMPLATE;
            break;
        case "type2":
            initialLayout = featuredMedia_type2_TEMPLATE;
            break;
        case "type3":
            initialLayout = featuredMedia_type3_TEMPLATE;
            break;
        default:
            initialLayout = featuredMedia_Sidebyside_TEMPLATE;
      };
      //console.log("initial layout: " + initialLayout);
    const [ layout, setLayout ] = useState( initialLayout );

      return (
        <Fragment>
            <h2>Image Mosaic:</h2>
              <InnerBlocks template={layout} templateLock="all"  key={key} />
              <InspectorControls>
              <PanelBody title={ __('Block Settings') }>
                <SelectControl
                label="Layout Option"
                value={ layoutoption }
                options={ [
                    { label: 'Side by Side Images', value: 'side_by_side' },
                    { label: '⅓ + ⅔ images above full width image', value: 'type1' },
                    { label: 'Full width image above ⅔ + ⅓ images', value: 'type2' },
                    { label: 'Full width + two half width + full width', value: 'type3' },
                ] }
                onChange={layoutoption => {
                    setAttributes({
                      layoutoption: layoutoption,
                    });
                    key = ++key;
                    //console.log(key);
                    //console.log("layoutoption on change " + layout)
                    switch (layoutoption){
                        case "side_by_side":
                            setLayout(featuredMedia_Sidebyside_TEMPLATE);
                            break;
                        case "type1":
                            setLayout(featuredMedia_type1_TEMPLATE);
                            break;
                        case "type2":
                            setLayout(featuredMedia_type2_TEMPLATE);
                            break;
                        case "type3":
                            setLayout(featuredMedia_type3_TEMPLATE);
                            break;
                        default:
                            setLayout(featuredMedia_Sidebyside_TEMPLATE);
                    };
                }}
              />
              </PanelBody>
            </InspectorControls>
        </Fragment>
      );
  },

  save({ attributes }) {
    const { layoutoption} = attributes;
    let layoutclassname;
    if(layoutoption){
      layoutclassname = layoutoption;
    }
    else{
      layoutclassname = "side_by_side"
    }
    return (
      <div className={layoutclassname}>
        <div className="wrapper">
          <InnerBlocks.Content />
        </div>
      </div>
    );
  },
};



// Media repeater Grid


const mediarepeater_icon = el('svg', { width: 20, height: 20, viewBox:"0 0 32 32" },
  el('path', { d: "M6,0H2C0.895,0,0,0.896,0,2v4c0,1.105,0.895,2,2,2h4c1.105,0,2-0.895,2-2V2C8,0.896,7.105,0,6,0z M6,6H2V2   h4V6z" } ),
  el('path', { d: "M18,0h-4c-1.105,0-2,0.896-2,2v4c0,1.105,0.895,2,2,2h4c1.104,0,2-0.895,2-2V2C20,0.896,19.104,0,18,0z    M18,6h-4V2h4V6z" }),
  el('path', { d: "M30,0h-4c-1.105,0-2,0.896-2,2v4c0,1.105,0.895,2,2,2h4c1.104,0,2-0.895,2-2V2C32,0.896,31.104,0,30,0z    M30,6h-4V2h4V6z" }),
  el('path', { d: "M6,12H2c-1.105,0-2,0.895-2,2v4c0,1.105,0.895,2,2,2h4   c1.105,0,2-0.895,2-2v-4C8,12.896,7.105,12,6,12z M6,18H2v-4h4V18z" }),
  el('path', { d: "M18,12h-4c-1.105,0-2,0.895-2,2v4c0,1.105,0.895,2,2,2h4   c1.104,0,2-0.895,2-2v-4C20,12.896,19.104,12,18,12z M18,18h-4v-4h4V18z" }),
  el('path', { d: "M30,12h-4c-1.105,0-2,0.895-2,2v4c0,1.105,0.895,2,2,2h4   c1.104,0,2-0.895,2-2v-4C32,12.896,31.104,12,30,12z M30,18h-4v-4h4V18z" }),
  el('path', { d: "M6,24H2c-1.105,0-2,0.895-2,2v4c0,1.105,0.895,2,2,2h4   c1.105,0,2-0.895,2-2v-4C8,24.896,7.105,24,6,24z M6,30H2v-4h4V30z" }),
  el('path', { d: "M18,24h-4c-1.105,0-2,0.895-2,2v4c0,1.105,0.895,2,2,2h4   c1.104,0,2-0.895,2-2v-4C20,24.896,19.104,24,18,24z M18,30h-4v-4h4V30z" }),
  el('path', { d: "M30,24h-4c-1.105,0-2,0.895-2,2v4c0,1.105,0.895,2,2,2h4   c1.104,0,2-0.895,2-2v-4C32,24.896,31.104,24,30,24z M30,30h-4v-4h4V30z" }),
);

const media_ALLOWED_BLOCKS = [ 'core/image', 'core/embed', 'core/video'];
const media_TEMPLATE = [
  [ 'core/image', {} ]
];

const mediaBlock = {
  title: __( 'Media Item' ),
  description: __( '' ),
  attributes: {

  },
  edit({ attributes, setAttributes, isSelected }) {
      return (
        <Fragment>
            <h2>Media:</h2>
            <div>
              <InnerBlocks allowedBlocks={media_ALLOWED_BLOCKS}  />
            </div>
        </Fragment>
      );
  },

  save({ attributes }) {
    return (
      <div className="mediaItem">
          <InnerBlocks.Content />
      </div>
    );
  },

};

const teamMember_ALLOWED_BLOCKS = [ 'core/image', 'core/paragraph', 'core/embed', 'core/video'];
const teamMember_TEMPLATE = [
  [ 'example/media-block'],
  [ 'core/paragraph', { placeholder: 'Title', className: "title" } ],
  [ 'core/paragraph', { placeholder: 'Subhead', className: "subhead" } ],
];

const teamMemberBlock = {
  title: __( 'Media Repeater Item' ),
  parent: [ 'core/columns' ],
  icon: mediarepeater_icon,
  description: __( '' ),
  attributes: {

  },
  edit({ attributes, setAttributes, isSelected }) {
      return (
        <div>
            <InnerBlocks allowedBlocks={teamMember_ALLOWED_BLOCKS} template={teamMember_TEMPLATE}/>
        </div>
      );
  },

  save({ attributes }) {
    return (
      <div className="column">
          <InnerBlocks.Content />
      </div>
    );
  },

};
const mediaRepeaterBlock_ALLOWED_BLOCKS = [ 'example/teammember-block'];
const mediaRepeater_TEMPLATE = [
  [ 'example/teammember-block' ],
  [ 'example/teammember-block' ],
  [ 'example/teammember-block' ],
];
const mediaRepeaterBlock = {
  title: __('Media Repeater Grid'),
  description: __(''),
  icon: mediarepeater_icon,
  attributes: {

  },
  edit({ attributes, setAttributes, isSelected }) {
     // console.log("initial layout: " + initialLayout)
      return (
        <Fragment>
            <h2>Media Repeater Grid:</h2>
            <div>
              <InnerBlocks allowedBlocks={mediaRepeaterBlock_ALLOWED_BLOCKS} template={mediaRepeater_TEMPLATE} templateLock="insert" />
            </div>
              <InspectorControls>
              <PanelBody title={ __('Block Settings') }>
              </PanelBody>
            </InspectorControls>
        </Fragment>
      );
  },

  save({ attributes }) {
    return (
      <div>
        <div className="columns">
          <InnerBlocks.Content />
        </div>
      </div>
    );
  },
};


// Link Mosaic
const linkMosaic_icon = el('svg', { width: 20, height: 20, fill:'#ffffff', stroke:"#000", viewBox:"0 0 24 24" },
  el('path', { d: "M0 0h24v24H0z", fill:"none" } ),
  el('path', { d: "M16 21V10h5v10a1 1 0 0 1-1 1h-4zm-2 0H4a1 1 0 0 1-1-1V10h11v11zm7-13H3V4a1 1 0 0 1 1-1h16a1 1 0 0 1 1 1v4z" }),
);

const linkMosaic_TEMPLATE = [
  [ 'core/columns', {}, [
    [ 'core/column', {width: 33.33}, [
        [ 'core/image', {url:"https://demo.gutenberghub.com/templates/wp-content/uploads/sites/10/2020/09/d3bymnz0ank-1024x640.jpg"} ],
        [ 'core/image',{url:"https://demo.gutenberghub.com/templates/wp-content/uploads/sites/10/2020/09/tt-roxwj9na-1024x683.jpg"} ],
    ] ],
    [ 'core/column', {width: 66.66}, [
      [ 'core/columns', {}, [
          [ 'core/column', {width: 33.33}, [
              [ 'core/image', {url:"https://demo.gutenberghub.com/templates/wp-content/uploads/sites/10/2020/09/efx1kzhasvo-1024x683.jpg"} ],
          ] ],
          [ 'core/column', {width: 66.66}, [
            [ 'core/image', {url:"https://demo.gutenberghub.com/templates/wp-content/uploads/sites/10/2020/09/gwedi4swshq-1024x683.jpg"} ],
          ] ],
        ] ],
        [ 'core/columns', {}, [
          [ 'core/column', {width: 66.66}, [
              [ 'core/image', {url: "https://demo.gutenberghub.com/templates/wp-content/uploads/sites/10/2020/09/3q3tsj01nc-1024x683.jpg"} ],
          ] ],
          [ 'core/column', {width: 33.33}, [
            [ 'core/button', {customBackgroundColor:"#be1e2d", customTextColor:"#ffffff"} ],
          ] ],
        ] ],
    ] ],
  ] ],
];
const linkMosaicBlock = {
  title: __('Link Mosaic'),
  description: __(''),
  icon: linkMosaic_icon,
  attributes: {

  },
  edit({ attributes, setAttributes, isSelected }) {
      return (
        <Fragment>
            <h2>Link Mosaic:</h2>
            <div>
              <InnerBlocks template={linkMosaic_TEMPLATE} templateLock="all" />
            </div>
              <InspectorControls>
              <PanelBody title={ __('Block Settings') }>
              </PanelBody>
            </InspectorControls>
        </Fragment>
      );
  },

  save({ attributes }) {
    return (
      <div>
        <div className="columns">
          <InnerBlocks.Content />
        </div>
      </div>
    );
  },
};



// Featured Text
const featuredText_icon = el('svg', { width: 20, height: 20, viewBox:"0 0 384 512" },
  el('path', { d: "M381.5 435.7l-160-384C216.6 39.78 204.9 32.01 192 32.01S167.4 39.78 162.5 51.7l-160 384c-6.797 16.31 .9062 35.05 17.22 41.84c16.38 6.828 35.08-.9219 41.84-17.22l31.8-76.31h197.3l31.8 76.31c5.109 12.28 17.02 19.7 29.55 19.7c4.094 0 8.266-.7969 12.3-2.484C380.6 470.7 388.3 452 381.5 435.7zM119.1 320L192 147.2l72 172.8H119.1z" } ),
);
const featuredText = {
  title: __('Featured Text'),
  description: __(''),
  icon: featuredText_icon,
  attributes: {
      title: {
          type: 'string',
      },

      backgroundflag: {
        type: 'string',
      },
  },



  edit({ attributes, setAttributes, isSelected }) {
      const { title, backgroundflag } = attributes;
      return (
        <Fragment>

          <RichText
                identifier="title"
                tagName="h3"
                value={title}
                placeholder={__('Featured Text')}
                onChange={title => {
                    setAttributes({
                        title: title,
                    });
                }}
                onSplit={() => null}
                unstableOnSplit={() => null}
            />

          <InspectorControls>
          <PanelBody title={ __('Block Settings') }>


            <SelectControl
            label="Background Color"
            value={ backgroundflag }
            options={ [
                { label: 'White', value: 'white_background' },
                { label: 'Grey', value: 'grey_background' },
                { label: 'Red', value: 'red_background' },
            ] }
            onChange={backgroundflag => {
                setAttributes({
                  backgroundflag: backgroundflag,
                });
            }}
          />
          </PanelBody>

        </InspectorControls>
        </Fragment>
      );
  },

  save({ attributes }) {
    const { title, backgroundflag} = attributes;

    return (
      <div className={`${(typeof backgroundflag!==undefined) ? backgroundflag : "white_background"}`}>
        <div className="wrapper">
          {title && (<p dangerouslySetInnerHTML={{__html: title}} />)}
        </div>
      </div>
    );
  },
};

// Text + image Block
const textimage_icon = el('svg', { width: 20, height: 20, viewBox: "0 0 20.25 20.25" },
  el('path', { d: "M0,.92a.92.92,0,0,0,.92.92H3.13A.92.92,0,0,0,3.13,0H.92A.92.92,0,0,0,0,.92Z" }),
  el('path', { d: "M0,4.6a.92.92,0,0,0,.92.92H3.13a.92.92,0,0,0,0-1.84H.92A.92.92,0,0,0,0,4.6Z" }),
  el('path', { d: "M.92,7.36a.92.92,0,1,0,0,1.84H3.13a.92.92,0,1,0,0-1.84Z" }),
  el('path', { d: "M0,12a.92.92,0,0,0,.92.92H3.13a.92.92,0,0,0,0-1.84H.92A.92.92,0,0,0,0,12Z" }),
  el('path', { d: "M0,15.64a.92.92,0,0,0,.92.92H19.33a.92.92,0,1,0,0-1.84H.92A.92.92,0,0,0,0,15.64Z" }),
  el('path', { d: "M0,19.33a.92.92,0,0,0,.92.92H19.33a.93.93,0,0,0,0-1.85H.92A.93.93,0,0,0,0,19.33Z" }),
  el('path', { d: "M17.32,12.88A2.93,2.93,0,0,0,20.25,10v-7A2.93,2.93,0,0,0,17.32,0h-9A2.93,2.93,0,0,0,5.44,2.93v7a2.92,2.92,0,0,0,2.92,2.92ZM8.37,11a1.08,1.08,0,0,1-1-.74l.87-.86a.49.49,0,0,1,.31-.14.57.57,0,0,1,.34.14h0a2.27,2.27,0,0,0,3.22,0l3.66-3.65a.44.44,0,0,1,.32-.13.43.43,0,0,1,.31.13l2,2V10A1.08,1.08,0,0,1,17.32,11Zm9-9.2a1.1,1.1,0,0,1,1.09,1.09V5.21l-.73-.72a2.25,2.25,0,0,0-1.61-.67,2.29,2.29,0,0,0-1.62.67L10.79,8.13l-.31.14-.32-.14h0a2.33,2.33,0,0,0-1.62-.66,2.29,2.29,0,0,0-1.25.38V2.93A1.09,1.09,0,0,1,8.36,1.84Z" }),
  el('path', { d: "M9.21,4.47a1.27,1.27,0,1,1,1.27,1.26A1.26,1.26,0,0,1,9.21,4.47" }),
);
const textimage_TEMPLATE = [
  [ 'core/columns', {}, [
    [ 'core/column', {}, [
      [ 'core/image' ],
    ] ],
    [ 'core/column', {}, [
      [ 'core/heading', {level:3} ],
      [ 'core/paragraph' ],
      [ 'core/button',{} ],
    ] ],
  ] ]
];
const text2image_TEMPLATE = [
  [ 'core/columns', {}, [
    [ 'core/column', {}, [
      [ 'core/image' ],
      [ 'core/image' ],
    ] ],
    [ 'core/column', {}, [
      [ 'core/heading', {level:3} ],
      [ 'core/paragraph' ],
      [ 'core/button',{} ],
    ] ],
  ] ],
];

let templatekey = 0;
const textImage = {
  title: __('Text + Image'),
  description: __('Side by side text + image'),
  icon: textimage_icon,
  attributes: {
    backgroundflag: {
      type: 'string',
    },
    imagealignment: {
      type: 'string',
    },
    layoutoption: {
        type: 'string',
    },
  },
  edit({ attributes, setAttributes, isSelected }) {
    const { backgroundflag, imagealignment, layoutoption } = attributes;
    let initialLayout;
    //console.log("layoutoption: " + layoutoption);

    switch (layoutoption){
        case "default":
            initialLayout = textimage_TEMPLATE;
            break;
        case "type1":
            initialLayout = text2image_TEMPLATE;
            break;
        default:
            initialLayout = textimage_TEMPLATE;
      };
     // console.log("initial layout: " + initialLayout)
    const [ textImage_layout, set_textImage_Layout ] = useState( initialLayout );

      return (
        <Fragment>
            <h2>Text + Image:</h2>
              <InnerBlocks template={textImage_layout} templateLock="all"  key={templatekey} />
              <InspectorControls>
              <PanelBody title={ __('Block Settings') }>
              <SelectControl
                label="Image Alignment"
                value={ imagealignment }
                description={"Alignment change will reflect in front"}
                options={ [
                    { label: 'Left', value: 'left_alignment' },
                    { label: 'Right', value: 'right_alignment' },
                ] }
                onChange={imagealignment => {
                    setAttributes({
                      imagealignment: imagealignment,
                    });
                }}
              />

                <SelectControl
                label="Layout Option"
                value={ layoutoption }
                options={ [
                    { label: 'Text + Image', value: 'default' },
                    { label: 'Text + 2 Images', value: 'type1' },
                ] }
                onChange={layoutoption => {
                    setAttributes({
                      layoutoption: layoutoption,
                    });
                    templatekey = ++templatekey;
                    // console.log(templatekey);
                    switch (layoutoption){
                        case "default":
                            set_textImage_Layout(textimage_TEMPLATE);
                            break;
                        case "type1":
                            set_textImage_Layout(text2image_TEMPLATE);
                            break;
                        default:
                            set_textImage_Layout(textimage_TEMPLATE);
                    };
                }}
              />

            <SelectControl
              label="Background Color"
              value={ backgroundflag }
              options={ [
                  { label: 'White', value: 'white_background' },
                  { label: 'Grey', value: 'grey_background' },
                  { label: 'Red', value: 'red_background' },
              ] }
              onChange={backgroundflag => {
                  setAttributes({
                    backgroundflag: backgroundflag,
                  });
              }}
            />

              </PanelBody>
            </InspectorControls>
        </Fragment>
      );
  },

  save({ attributes }) {
    const {  backgroundflag, imagealignment, layoutoption} = attributes;

    return (
      <div className={`${(typeof backgroundflag!==undefined) ? backgroundflag : "white_background"} ${(typeof layoutoption!==undefined) ? layoutoption : "textimage"}`}>
        <div className="wrapper">
        <div className={`${(typeof backgroundflag!==undefined) ? imagealignment : "left_alignment"}`}>
        <InnerBlocks.Content />
          </div>
        </div>
      </div>
    );

  },
};


// Call to Action Block
const CTA_TEMPLATE = [
  [ 'core/button', {align:"center"} ],
];
const CTA2_TEMPLATE = [
  [ 'core/heading', {level:3} ],
  [ 'core/paragraph' ],
];

let cta_templatekey = 0;
const cta = {
  title: __('Call to Action'),
  description: __(''),
  attributes: {
    layoutoption: {
        type: 'string',
    },
  },
  edit({ attributes, setAttributes, isSelected }) {
    const { layoutoption } = attributes;
    let initialLayout;
    //console.log("layoutoption: " + layoutoption);

    switch (layoutoption){
        case "default":
            initialLayout = CTA_TEMPLATE;
            break;
        case "type1":
            initialLayout = CTA2_TEMPLATE;
            break;
        default:
            initialLayout = CTA_TEMPLATE;
      };
     // console.log("initial layout: " + initialLayout)
    const [ cta_layout, set_cta_Layout ] = useState( initialLayout );

      return (
        <Fragment>
            <h2>Call to Action:</h2>
              <InnerBlocks template={cta_layout} templateLock="all"  key={cta_templatekey} />
              <InspectorControls>
              <PanelBody title={ __('Block Settings') }>

                <SelectControl
                label="Layout Option"
                value={ layoutoption }
                options={ [
                    { label: 'Simple Linked Text', value: 'default' },
                    { label: 'Title + Body + Link', value: 'type1' },
                ] }
                onChange={layoutoption => {
                    setAttributes({
                      layoutoption: layoutoption,
                    });
                    cta_templatekey = ++cta_templatekey;
                    // console.log(templatekey);
                    switch (layoutoption){
                        case "default":
                            set_cta_Layout(CTA_TEMPLATE);
                            break;
                        case "type1":
                            set_cta_Layout(CTA2_TEMPLATE);
                            break;
                        default:
                            set_cta_Layout(CTA_TEMPLATE);
                    };
                }}
              />

              </PanelBody>
            </InspectorControls>
        </Fragment>
      );
  },

  save({ attributes }) {
    const { layoutoption} = attributes;

    return (
      <div className={`${(typeof layoutoption!==undefined) ? layoutoption : "default"}`}>
        <div className="wrapper">
        <InnerBlocks.Content />
        </div>
      </div>
    );

  },
};


// Link list Item
const linklist_icon = el('svg', { width: 20, height: 20, viewBox: "0 0 18.4 20.49" },
  el('path', { d: "M.77,13.33l.61-.69,1,1-.69.6a1.09,1.09,0,0,0,0,1.55l3,2.92a1,1,0,0,0,1.46,0l6.36-6.27a1.09,1.09,0,0,0,0-1.55L9.63,8l.95-1,3,2.93a2.55,2.55,0,0,1,0,3.52L7.22,19.78a2.55,2.55,0,0,1-3.52,0L.77,16.85a2.39,2.39,0,0,1,0-3.52Z" }),
  el('path', { d: "M4.81,7.05,11.18.77a2.39,2.39,0,0,1,3.52,0L17.63,3.7a2.39,2.39,0,0,1,0,3.52l-.95,1-1-1,1-1a1.17,1.17,0,0,0,0-1.46l-3-3a1.09,1.09,0,0,0-1.55,0L5.85,8.08a1.09,1.09,0,0,0,0,1.55l2.92,2.92-1,1-2.93-3a2.56,2.56,0,0,1,0-3.53Z" }),
);
const linkListItem = {
  title: __('Link list Item'),
  description: __(''),
  icon: linklist_icon,
  parent: ['example/linklist-block'],
  attributes: {
      link: {
          type: 'string',
      },
      textcopy: {
          type: 'string',
      },
  },



  edit({ attributes, setAttributes, isSelected }) {
      const { link, textcopy } = attributes;



      // Limitation of characters
      const linkcharsLimit = 34;
      const desccharsLimit = 27;

      const lengthwhithoutTags = str => {
          let tmp = document.createElement("DIV");
          tmp.innerHTML = str;
         // console.log(tmp.textContent.length);
        //  console.log(tmp.innerText.length);
          return tmp.textContent.length || tmp.innerText.length || 0;
      }



      return (
        <Fragment>
         <div style={{display: "flex"}}>
            <RichText
                  identifier="link"
                  tagName="p"
                  value={link}
                  description="character max of 34"
                  placeholder={__('Link Text')}
                  onChange={link => {
                    if (lengthwhithoutTags(link) <= linkcharsLimit) {
                      setAttributes({
                        link: link,
                      });
                    }
                    else{
                      alert("Link text maximum character limit is " + linkcharsLimit)
                    }
                  }}
                  onSplit={() => null}
                  unstableOnSplit={() => null}
                  style={{width: "50%", textAlign: "left"}}
              />
            <RichText
                  identifier="textcopy"
                  tagName="p"
                  value={textcopy}
                  description="character max of 27"
                  placeholder={__('Description Text')}
                  onChange={textcopy => {
                    if (lengthwhithoutTags(textcopy) <= desccharsLimit) {
                      //console.log("Textcopy: " + textcopy.length);
                      setAttributes({
                        textcopy: textcopy,
                      });
                    }
                    else{
                      alert("Description text maximum character limit is " + desccharsLimit)
                    }
                  }}
                  onSplit={() => null}
                  unstableOnSplit={() => null}
                  style={{width: "50%", textAlign: "right"}}
              />
              </div>

          <InspectorControls>
          <PanelBody title={ __('Block Settings') }>



          </PanelBody>

        </InspectorControls>
        </Fragment>
      );
  },

  save({ attributes }) {
    const { link, textcopy} = attributes;

    return (
      <div>
        <div className="wrapper">
          {link && (<div dangerouslySetInnerHTML={{__html: link}} />)}
          {textcopy && (<div dangerouslySetInnerHTML={{__html: textcopy}} />)}
        </div>
      </div>
    );
  },
};


// Link list
const linkList_ALLOWED_BLOCKS = [ 'example/linklistitem-block' ];

const linkList = {
  title: __('Link List'),
  description: __(''),
  attributes: {
    title: {
        type: 'string',
    },

  },
  edit({ attributes, setAttributes, isSelected }) {
    const { title } = attributes;
      return (
        <Fragment>
            <RichText
                  identifier="title"
                  tagName="h2"
                  value={title}
                  placeholder={__('Link List Title')}
                  onChange={title => {
                      setAttributes({
                        title: title,
                      });
                  }}
                  onSplit={() => null}
                  unstableOnSplit={() => null}
              />
            <InnerBlocks allowedBlocks={linkList_ALLOWED_BLOCKS} />
        </Fragment>
      );
  },

  save({ attributes }) {
    const { title} = attributes;
    return (
      <div>
        <div className="wrapper">
          {title && (<h2 dangerouslySetInnerHTML={{__html: title}} />)}
          <InnerBlocks.Content />
      </div>
      </div>
    );
  },
};


// Text mosaic
const testMosaic_icon = el('svg', { width: 20, height: 20, viewBox: "0 0 20.58 20.33" },
  el('path', { d: "M10.14,0a.49.49,0,0,0-.49.48.49.49,0,0,0,.49.49h2.58A.48.48,0,0,0,13.2.48.48.48,0,0,0,12.72,0ZM5,0a.49.49,0,0,0-.49.48A.49.49,0,0,0,5,1H7.56A.48.48,0,0,0,8,.48.48.48,0,0,0,7.56,0ZM2.43.17a.63.63,0,0,0-.19,0,3.41,3.41,0,0,0-2,2.15A.49.49,0,0,0,.58,3,.39.39,0,0,0,.73,3a.49.49,0,0,0,.46-.35,2.36,2.36,0,0,1,1.4-1.51A.5.5,0,0,0,2.87.47a.49.49,0,0,0-.44-.3ZM16.79.26a.62.62,0,0,0-.27,0,.56.56,0,0,0-.14.93l1.19,1.19L15.5,4.5a.61.61,0,0,0,0,.85.62.62,0,0,0,.86,0L18.44,3.3l1.17,1.17a.55.55,0,0,0,1-.4V.85A.58.58,0,0,0,20,.27ZM0,5.06V7.64a.48.48,0,0,0,.48.48A.48.48,0,0,0,1,7.64V5.06a.48.48,0,0,0-.49-.48A.48.48,0,0,0,0,5.06ZM7.23,6.13A2.39,2.39,0,0,0,5,8a.5.5,0,0,0,.37.59h.11a.47.47,0,0,0,.46-.38A1.4,1.4,0,0,1,7.23,7.09H9.09V13.9a2.3,2.3,0,0,0-1.59.95L7.37,15a.49.49,0,0,0,.76.61l.15-.18s0,0,0,0a1.34,1.34,0,0,1,1.14-.57h.1a1.34,1.34,0,0,1,1.13.57l.16.22a.49.49,0,0,0,.67.1.47.47,0,0,0,.11-.67l-.15-.21a2.2,2.2,0,0,0-1.41-.92s0,0,0,0V7.09h1.68a1.38,1.38,0,0,1,1.33,1.05A.48.48,0,1,0,14,7.91a2.31,2.31,0,0,0-2.26-1.78ZM19.83,7.5a.48.48,0,0,0-.48.48v2.58a.49.49,0,1,0,1,0V8a.48.48,0,0,0-.49-.48ZM0,10.22v2.59a.48.48,0,0,0,.48.48A.48.48,0,0,0,1,12.81V10.22a.48.48,0,0,0-.49-.48A.48.48,0,0,0,0,10.22Zm19.35,2.92v2.58a.49.49,0,0,0,1,0V13.14a.49.49,0,0,0-1,0ZM.48,14.9a.49.49,0,0,0-.48.49v2a2.65,2.65,0,0,0,.15.69.58.58,0,0,0,.55.36H.81a.45.45,0,0,0,.28-.58A1.83,1.83,0,0,1,1,17.39v-2a.49.49,0,0,0-.49-.49Zm19.4,2.91a.47.47,0,0,0-.52.31,1.86,1.86,0,0,1-1.52,1.22.49.49,0,0,0-.42.54.48.48,0,0,0,.48.42H18a2.85,2.85,0,0,0,2.31-1.85.47.47,0,0,0-.3-.61.29.29,0,0,0-.1,0ZM2.49,19.23a.48.48,0,0,0-.51.33.49.49,0,0,0,.32.62,3.5,3.5,0,0,0,1,.15H5a.49.49,0,0,0,0-1H3.32a2.6,2.6,0,0,1-.74-.11l-.09,0Zm10.26.13a.49.49,0,1,0,0,1h2.58a.49.49,0,0,0,0-1Zm-5.16,0a.49.49,0,0,0,0,1h2.58a.49.49,0,0,0,0-1Z" }),
);
const textMosaic_TEMPLATE = [
  [ 'core/columns', {}, [
    [ 'core/column', {width:33}, [
      [ 'core/group',{},[
        [ 'core/heading', {level:3,placeholder:"Title"} ],
        [ 'core/paragraph',{placeholder:"Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut tempus lectus libero, sit amet facilisis leo pharetra ac. "} ],
      ] ],
      [ 'core/group',{},[
        [ 'core/heading', {level:3,placeholder:"Title"} ],
        [ 'core/paragraph',{placeholder:"Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut tempus lectus libero, sit amet facilisis leo pharetra ac. "} ],
      ] ],
    ] ],
    [ 'core/column', {width:66}, [
      [ 'core/group',{},[
        [ 'core/heading', {level:3,placeholder:"Title"} ],
        [ 'core/paragraph',{placeholder:"Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut tempus lectus libero, sit amet facilisis leo pharetra ac. "} ],
      ] ],
      [ 'core/columns', {},[
        ['core/column',{width:66}, [
          [ 'core/group',{},[
            [ 'core/heading', {level:3,placeholder:"Title"} ],
          [ 'core/paragraph',{placeholder:"Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut tempus lectus libero, sit amet facilisis leo pharetra ac. "} ],
          ] ],
        ]],
        ['core/column',{width:33}, [
          [ 'core/group',{},[
            [ 'core/heading', {level:3,placeholder:"Title"} ],
          [ 'core/paragraph',{placeholder:"Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut tempus lectus libero, sit amet facilisis leo pharetra ac. "} ],
          ] ],
        ]],
      ] ],
    ] ],
  ] ],
  [ 'core/columns', {}, [
    [ 'core/column', {width:40}, [
      [ 'core/group',{},[
        [ 'core/heading', {level:3,placeholder:"Title"} ],
        [ 'core/paragraph',{placeholder:"Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut tempus lectus libero, sit amet facilisis leo pharetra ac. "} ],
      ] ],
    ] ],
    [ 'core/column', {width:60}, [
      [ 'core/group',{},[
        [ 'core/heading', {level:3,placeholder:"Title"} ],
        [ 'core/paragraph',{placeholder:"Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut tempus lectus libero, sit amet facilisis leo pharetra ac. "} ],
      ] ],
    ] ],
  ]],
];
const textMosaic = {
  title: __('Text Mosaic'),
  description: __(''),
  icon: testMosaic_icon,
  attributes: {
    backgroundflag: {
      type: 'string',
    },
  },

  edit({ attributes, setAttributes, isSelected }) {
    const { backgroundflag } = attributes;
      return (
        <Fragment>
          <h2>Text Mosaic</h2>
            <InnerBlocks templateLock="all" template={textMosaic_TEMPLATE} />
            <InspectorControls>
          <PanelBody title={ __('Block Settings') }>


            <SelectControl
            label="Background Color"
            value={ backgroundflag }
            options={ [
                { label: 'White', value: 'white_background' },
                { label: 'Grey', value: 'grey_background' },
                { label: 'Red', value: 'red_background' },
            ] }
            onChange={backgroundflag => {
                setAttributes({
                  backgroundflag: backgroundflag,
                });
            }}
          />
          </PanelBody>

        </InspectorControls>
        </Fragment>
      );
  },

  save({ attributes }) {
    const { backgroundflag } = attributes;
    return (
      <div className={`${(typeof backgroundflag!==undefined) ? backgroundflag : "white_background"}`}>
        <div className="wrapper">
          <InnerBlocks.Content />
      </div>
      </div>
    );
  },
};


// Full Width Image Slider
const fullwidthSlider_ALLOWED_BLOCKS = [ 'core/image' ];
const fullwidthSlider_icon = el('svg', { width: 20, height: 20, viewBox: "0 0 23.36 22.02" },
  el('path', { d: "M20.86,0H2.5A2.49,2.49,0,0,0,0,2.5V15a2.49,2.49,0,0,0,2.5,2.5H20.86a2.49,2.49,0,0,0,2.5-2.5V2.5A2.49,2.49,0,0,0,20.86,0Zm.83,15a.82.82,0,0,1-.83.83H2.5a.87.87,0,0,1-.59-.24A.87.87,0,0,1,1.67,15V2.5a.87.87,0,0,1,.24-.59.87.87,0,0,1,.59-.24H20.86a.87.87,0,0,1,.59.24.87.87,0,0,1,.24.59Z" }),
  el('path', { d: "M6.26,22A.83.83,0,0,0,7,20.77a.83.83,0,0,0-.72-.42H4.59a.83.83,0,0,0-.72.42A.83.83,0,0,0,4.59,22Z" }),
  el('path', { d: "M10.43,22a.83.83,0,0,0,.72-1.25.83.83,0,0,0-.72-.42H8.76a.83.83,0,0,0-.72.42A.83.83,0,0,0,8.76,22Z" }),
  el('path', { d: "M14.6,22a.83.83,0,0,0,.72-1.25.83.83,0,0,0-.72-.42H12.93a.83.83,0,0,0-.72.42A.83.83,0,0,0,12.93,22Z" }),
  el('path', { d: "M18.77,22a.83.83,0,0,0,.72-1.25.83.83,0,0,0-.72-.42H17.1a.83.83,0,0,0-.72.42A.83.83,0,0,0,17.1,22Z" }),
);
const fullwidthSlider = {
  title: __('Full Width Image Slider'),
  description: __(''),
  icon: fullwidthSlider_icon,
  attributes: {
    backgroundflag: {
      type: 'string',
    },
  },
  edit({ attributes, setAttributes, isSelected }) {
    const { backgroundflag } = attributes;
      return (
        <Fragment>
          <h2>Full Width Image Slider</h2>
            <InnerBlocks allowedBlocks={fullwidthSlider_ALLOWED_BLOCKS} />
            <InspectorControls>
          <PanelBody title={ __('Block Settings') }>


            <SelectControl
            label="Background Color"
            value={ backgroundflag }
            options={ [
                { label: 'White', value: 'white_background' },
                { label: 'Grey', value: 'grey_background' },
                { label: 'Red', value: 'red_background' },
            ] }
            onChange={backgroundflag => {
                setAttributes({
                  backgroundflag: backgroundflag,
                });
            }}
          />
          </PanelBody>

        </InspectorControls>
        </Fragment>

      );
  },

  save({ attributes }) {
    const { backgroundflag } = attributes;
    return (
      <div aria-roledescription="carousel" aria-label="Full Width Image Slider" className={`${(typeof backgroundflag!==undefined) ? backgroundflag : "white_background"}`}>
        <nav className="slider-navigation">
          <button className="prev" aria-controls="text-image-slider" data-slide="prev"> <span className="visually-hidden-focusable">Previous Slide</span> </button>
          <button className="next" aria-controls="text-image-slider" data-slide="next"> <span className="visually-hidden-focusable">Next Slide</span> </button>
        </nav>
        <div className="fullwidth-slider-wrapper" data-ride="carousel" id="full-width-slider">
          <InnerBlocks.Content aria-roledescription="Image Slide" role="group" />
        </div>
      </div>
    );
  },
};


// Spacer Block

const spacevlock_icon = el('svg', { width: 20, height: 20, viewBox: "0 0 21.06 21.06" },
  el('path', { d: "M18.16,2.9V13.06H8V2.9H18.16M20.69.38H5.45V15.61H20.69Z" }),
  el('path', { d: "M2.9,18.16V5.45H.38V20.69H15.61V18.16Z" }),
);
const spaceBlock = {
  title: __('Spacer Block'),
  description: __(''),
  icon: spacevlock_icon,
  attributes: {
    layoutoption: {
        type: 'string',
    },
    backgroundflag: {
      type: 'string',
    },
  },
  edit({ attributes, setAttributes, isSelected }) {
    const { layoutoption, backgroundflag } = attributes;

    let initialLayout;

    if (layoutoption){
      initialLayout = layoutoption;
    }
    else{
      initialLayout = "small";
    }
    //console.log("layoutoption: " + layoutoption);


    const [ layout, setLayout ] = useState( initialLayout );

      return (
        <Fragment>
            <h2>Spacer Block Size: {layout} </h2>

              <InspectorControls>
              <PanelBody title={ __('Block Settings') }>
                <SelectControl
                  label="Size Option"
                  value={ layoutoption }
                  options={ [
                      { label: 'Small', value: 'small' },
                      { label: 'Medium', value: 'medium' },
                      { label: 'Large', value: 'large' },
                  ] }
                  onChange={layoutoption => {
                      setAttributes({
                        layoutoption: layoutoption,
                      });
                      setLayout(layoutoption);
                  }}
                />

              <SelectControl
                label="Background Color"
                value={ backgroundflag }
                options={ [
                    { label: 'White', value: 'white_background' },
                    { label: 'Grey', value: 'grey_background' },
                    { label: 'Red', value: 'red_background' },
                ] }
                onChange={backgroundflag => {
                    setAttributes({
                      backgroundflag: backgroundflag,
                    });
                }}
              />

              </PanelBody>
            </InspectorControls>
        </Fragment>
      );
  },

  save({ attributes }) {
    const { backgroundflag, layoutoption} = attributes;

    return (
      <div className={`${(typeof backgroundflag!==undefined) ? backgroundflag : "white_background"}`}>
        <div className={`${(typeof layoutoption!==undefined) ? layoutoption : "small"}`}>
        </div>
      </div>
    );
  },
};


// FAQ Item

const faqaccodian_icon = el('svg', { width: 20, height: 20, viewBox: "0 0 29.13 13.29" },
  el('path', { d: "M2.34,13H26.79A2.1,2.1,0,0,0,28.88,11V2.34A2.1,2.1,0,0,0,26.79.25H2.34A2.1,2.1,0,0,0,.25,2.34V11A2.1,2.1,0,0,0,2.34,13ZM2.28,2.34a.06.06,0,0,1,.06-.06H26.79a.06.06,0,0,1,.06.06V11a.06.06,0,0,1-.06.06H2.34A.06.06,0,0,1,2.28,11Z" }),
  el('path', { d: "M20.82,8.25a1.24,1.24,0,0,0,.92.42,1.23,1.23,0,0,0,.92-.42l1.78-2.06A1,1,0,1,0,22.9,4.87L21.74,6.21,20.58,4.87a1,1,0,1,0-1.53,1.32Z" }),
);
function randomIntFromInterval(min, max) { // min and max included
  return Math.floor(Math.random() * (max - min + 1) + min);
}
let accordionGroupID = randomIntFromInterval(1, 10);
let accordionID = 0;
const faqItem = {
  title: __('FAQ Accodion Item'),
  description: __(''),
  icon: faqaccodian_icon,
  parent: ['example/faq-block'],
  attributes: {
      title: {
          type: 'string',
      },
      textcopy: {
          type: 'string',
      },
      sectid: {
        type: 'string',
      },
      id: {
        type: 'string',
      }
  },
  edit({ attributes, setAttributes, isSelected }) {
      const { title, textcopy } = attributes;


      accordionID = randomIntFromInterval(1, 100);

      //console.log("accordionID:" + accordionID);
      //console.log("accordionGroupID:" + accordionGroupID);

      return (
        <Fragment>
            <RichText
                  identifier="title"
                  tagName="h3"
                  value={title}
                  placeholder={__('FAQ Accordion Title')}
                  onChange={title => {
                    setAttributes({
                      title: title,
                      id : `accordion${accordionGroupID}${accordionID}id`,
                      sectid : `sect${accordionGroupID}${accordionID}`,
                    });
                  }}
                  onSplit={() => null}
                  unstableOnSplit={() => null}
              />
            <RichText
                  identifier="textcopy"
                  tagName="p"
                  value={textcopy}
                  placeholder={__('FAQ Accordion Body Copy')}
                  onChange={textcopy => {
                    setAttributes({
                      textcopy: textcopy,
                    });
                  }}
                  onSplit={() => null}
                  unstableOnSplit={() => null}
              />

          <InspectorControls>
          <PanelBody title={ __('Block Settings') }>



          </PanelBody>

        </InspectorControls>
        </Fragment>
      );
  },


  save({ attributes }) {
    const { title, textcopy, id, sectid } = attributes;
    return (
      <Fragment>
        <div class='faqitem'>
          <div class="faq-title h3">
            <button type="button" aria-expanded="false" class="accordion-trigger" aria-controls={sectid} id={id}>
              <span class="accordion-title">
                {title && (<span dangerouslySetInnerHTML={{__html: title}} />)}
                <span class="accordion-icon"></span>
              </span>
            </button>
          </div>
          <div id={sectid} role="region" aria-labelledby={id} class="accordion-panel" hidden>
              {textcopy && (<div dangerouslySetInnerHTML={{__html: textcopy}} />)}
          </div>
          </div>
      </Fragment>
    );
  },
};


// FAQ Accordion

const faq_ALLOWED_BLOCKS = [ 'example/faqitem-block' ];
const faqBlock = {
  title: __('FAQ Accordion'),
  description: __(''),
  attributes: {
    groupid: {
      type: 'string',
      value: accordionGroupID,
    }
},
  edit({ attributes, setAttributes, isSelected }) {

      return (
        <Fragment>
            <h2>FAQ Accordion</h2>
            <InnerBlocks allowedBlocks={faq_ALLOWED_BLOCKS} />
        </Fragment>
      );
  },

  save({ attributes }) {
    const { groupid } = attributes;

    return (
      <div>
        <div className="wrapper">
          <div id={groupid} class="accordion">
            <InnerBlocks.Content />
          </div>
        </div>
      </div>
    );
  },
};

//Registering Blocks
const category = {
  slug: 'example',
  title: __('Advanced Gutenberg Blocks'),
};

// Audio Block
const customaudio_icon = el('svg', { width: 20, height: 20, viewBox: "0 0 22.85 22.85" },
  el('path', { d: "M22.3,0H.55A.55.55,0,0,0,0,.55V22.3H1.1V1.1H21.75V21.75H4.4V8.39H5.75V14.3H5.32v1.1h.8l5.36,4a.55.55,0,0,0,.33.11.53.53,0,0,0,.24-.06.54.54,0,0,0,.31-.49V4a.57.57,0,0,0-.3-.49.56.56,0,0,0-.57,0L6.13,7.29H3.85a.55.55,0,0,0-.55.55V22.3a.55.55,0,0,0,.55.55H22.3a.54.54,0,0,0,.55-.55V.55A.55.55,0,0,0,22.3,0Zm-11,17.88L6.85,14.57V8.12L11.26,5Z" }),
);
const audio_TEMPLATE = [
  [ 'core/columns', {}, [
    [ 'core/column', {}, [
      [ 'core/group',{},[
        [ 'core/audio'],
      ] ],
    ] ],
    [ 'core/column', {}, [
      [ 'core/group',{},[
        [ 'core/heading', {level:3,placeholder:"Title"} ],
        [ 'core/paragraph',{placeholder:"Lorem ipsum dolor sit amet, consectetur adipiscing elit. "} ],
      ] ],
    ] ],
  ] ],
];
const audioBlock = {
  title: __('Custom Audio Block'),
  description: __(''),
  icon: customaudio_icon,
  attributes: {

  },

  edit({ attributes, setAttributes, isSelected }) {

      return (
        <Fragment>
          <h2>Audio Block</h2>
            <InnerBlocks templateLock="all" template={audio_TEMPLATE} />
            <InspectorControls>
        </InspectorControls>
        </Fragment>
      );
  },

  save({ attributes }) {

    return (
      <div>
        <div className="wrapper">
          <InnerBlocks.Content />
      </div>
      </div>
    );
  },
};


const currentCategories = select('core/blocks').getCategories().filter(item => item.slug !== category.slug);
dispatch('core/blocks').setCategories([ category, ...currentCategories ]);

registerBlockType(`${category.slug}/sectionbreak-block`, { category: category.slug, ...sectionBreakH2 });
registerBlockType(`${category.slug}/customheading-block`, { category: category.slug, ...customHeading });
registerBlockType(`${category.slug}/highlightedtext-block`, { category: category.slug, ...highlightedText });
registerBlockType(`${category.slug}/imagegrid-block`, { category: category.slug, ...imageGrid });
registerBlockType(`${category.slug}/customtext-block`, { category: category.slug, ...CustomTextBlock });
registerBlockType(`${category.slug}/textimageslide-block`, { category: category.slug, ...TextImageSlide });
registerBlockType(`${category.slug}/textimageslider-block`, { category: category.slug, ...TextImageSlider });
registerBlockType(`${category.slug}/customfeaturedmedia-block`, { category: category.slug, ...CustomFeaturedMediaBlock });
registerBlockType(`${category.slug}/customimagemosaic-block`, { category: category.slug, ...imageMosaicBlock });
//registerBlockType(`${category.slug}/teammember-block`, { category: category.slug, ...teamMemberBlock });
//registerBlockType(`${category.slug}/media-block`, { category: category.slug, ...mediaBlock });
//registerBlockType(`${category.slug}/mediarepeater-block`, { category: category.slug, ...mediaRepeaterBlock });
registerBlockType(`${category.slug}/linkmosaic-block`, { category: category.slug, ...linkMosaicBlock });
registerBlockType(`${category.slug}/featuredtext-block`, { category: category.slug, ...featuredText });
registerBlockType(`${category.slug}/textimage-block`, { category: category.slug, ...textImage });
registerBlockType(`${category.slug}/cta-block`, { category: category.slug, ...cta });
registerBlockType(`${category.slug}/linklistitem-block`, { category: category.slug, ...linkListItem });
registerBlockType(`${category.slug}/linklist-block`, { category: category.slug, ...linkList });
registerBlockType(`${category.slug}/textmosaic-block`, { category: category.slug, ...textMosaic });
registerBlockType(`${category.slug}/fullwidthslider-block`, { category: category.slug, ...fullwidthSlider });
registerBlockType(`${category.slug}/spacer-block`, { category: category.slug, ...spaceBlock });
registerBlockType(`${category.slug}/faqitem-block`, { category: category.slug, ...faqItem });
registerBlockType(`${category.slug}/faq-block`, { category: category.slug, ...faqBlock });
registerBlockType(`${category.slug}/audio-block`, { category: category.slug, ...audioBlock });
