<?php

namespace Drupal\advanced_language_negotiation\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure the URL language negotiation method for this site.
 *
 * @internal
 */
class AdvancedLanguageNegotiationForm extends ConfigFormBase {

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Constructs a new NegotiationUrlForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typedConfigManager, LanguageManagerInterface $language_manager) {
    parent::__construct($config_factory, $typedConfigManager);
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('language_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'language_negotiation_configure_url_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['language.negotiation'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('language.negotiation');

    $languages = $this->languageManager->getLanguages();
    $default_config = $config->get('domain_url');
    $form['domain_url']['#tree'] = TRUE;
    foreach ($languages as $langcode => $language) {
      $t_args = ['%language' => $language->getName(), '%langcode' => $language->getId()];
      $form['domain_url'][$langcode] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['container-inline']],
      ];

      $form['domain_url'][$langcode]['domain'] = [
        '#type' => 'textfield',
        '#title' => $this->t('%language (%langcode) domain', $t_args),
        '#maxlength' => 128,
        '#default_value' => $default_config[$langcode]['domain'] ?? '',
      ];
      $form['domain_url'][$langcode]['prefix'] = [
        '#type' => 'textfield',
        '#maxlength' => 10,
        '#default_value' => $default_config[$langcode]['prefix'] ?? '',
        '#field_prefix' => '/',
        '#size' => 10,
      ];
    }

    $form_state->setRedirect('language.negotiation');

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $languages = $this->languageManager->getLanguages();

    foreach ($languages as $langcode => $language) {
      $domain = $form_state->getValue(['domain_url', $langcode, 'domain']);
      $prefix = $form_state->getValue(['domain_url', $langcode, 'prefix']);
      if (str_contains($prefix, '/')) {
        // Throw a form error if the string contains a slash,
        // which would not work.
        $form_state->setErrorByName("domain_url][$langcode][prefix", $this->t('The prefix may not contain a slash.'));
      }
      if (empty($domain)) {
        $form_state->setErrorByName("domain_url][$langcode][domain", $this->t('The domain may not be left blank for %language.', ['%language' => $language->getName()]));
      }
      else {
        // Ensure we have exactly one protocol when checking the hostname.
        $host = 'http://' . str_replace(['http://', 'https://'], '', $domain);
        if (parse_url($host, PHP_URL_HOST) != $domain) {
          $form_state->setErrorByName("domain_url][$langcode][domain", $this->t('The domain for %language may only contain the domain name, not a trailing slash, protocol and/or port.', ['%language' => $language->getName()]));
        }
      }
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Save selected format (prefix or domain).
    $this->config('language.negotiation')
      ->set('domain_url', $form_state->getValue('domain_url'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
