<?php

namespace Drupal\advanced_language_negotiation\Plugin\LanguageNegotiation;

use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\PathProcessor\InboundPathProcessorInterface;
use Drupal\Core\PathProcessor\OutboundPathProcessorInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\language\Attribute\LanguageNegotiation;
use Drupal\language\LanguageNegotiationMethodBase;
use Drupal\language\LanguageSwitcherInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides a language negotiation method that combines domain and path prefix.
 */
#[LanguageNegotiation(
  id: AdvancedLanguageNegotiation::METHOD_ID,
  name: new TranslatableMarkup('URL (Domain and Path)'),
  types: [LanguageInterface::TYPE_INTERFACE,
    LanguageInterface::TYPE_CONTENT,
    LanguageInterface::TYPE_URL,
  ],
  weight: -8,
  description: new TranslatableMarkup("Allows language negotiation based on both domain and path prefix."),
  config_route_name: 'advanced_language_negotiation.config'
)]
class AdvancedLanguageNegotiation extends LanguageNegotiationMethodBase implements InboundPathProcessorInterface, OutboundPathProcessorInterface, LanguageSwitcherInterface {

  /**
   * The language negotiation method id.
   */
  const METHOD_ID = 'language-domain-url';

  /**
   * URL language negotiation: use the path prefix as URL language indicator.
   */
  const CONFIG_PATH_PREFIX = 'path_prefix';

  /**
   * URL language negotiation: use the domain as URL language indicator.
   */
  const CONFIG_DOMAIN = 'domain';

  /**
   * {@inheritdoc}
   */
  public function getLangcode(?Request $request = NULL) {
    $langcode = 'en';

    if ($request && $this->languageManager) {
      $languages = $this->languageManager->getLanguages();
      $config = $this->config->get('language.negotiation')->get('domain_url');
      $request_path = urldecode(trim($request->getPathInfo(), '/'));
      $path_args = explode('/', $request_path);
      $prefix = array_shift($path_args);
      $http_host = $request->getHost();
      foreach ($languages as $language) {
        // If there is no prefix, go with domain specific language.
        if (!empty($config[$language->getId()]['domain'])) {
          // Ensure that there is exactly one protocol in the URL when
          // checking the hostname.
          $host = 'http://' . str_replace(['http://', 'https://'], '', $config[$language->getId()]['domain']);
          $host = parse_url($host, PHP_URL_HOST);
          if ($http_host == $host) {
            if (!empty($config[$language->getId()]['prefix'])) {
              if ($config[$language->getId()]['prefix'] !== $prefix) {
                continue;
              }
              else {
                $langcode = $language->getId();
                break;
              }
            }
            else {
              $langcode = $language->getId();
            }
          }
        }
      }
    }
    return $langcode;
  }

  /**
   * {@inheritdoc}
   */
  public function processInbound($path, Request $request) {
    $config = $this->config->get('language.negotiation')->get('domain_url');
    $http_host = $request->getHost();
    $parts = explode('/', trim($path, '/'));
    $prefix = array_shift($parts);

    // Search prefix within added languages.
    foreach ($this->languageManager->getLanguages() as $language  ) {
      $config_domain = $config[$language->getId()]['domain'];
      $config_prefix = $config[$language->getId()]['prefix'];
      $host = 'http://' . str_replace(['http://', 'https://'], '', $config_domain);
      $host = parse_url($host, PHP_URL_HOST);
      if ($host == $http_host) {
        if (!empty($config_prefix) && $config_prefix == $prefix) {
          // Rebuild $path with the language removed.
          $path = '/' . implode('/', $parts);
          break;
        }
      }
    }
    return $path;
  }

  /**
   * {@inheritdoc}
   */
  public function processOutbound($path, &$options = [], ?Request $request = NULL, ?BubbleableMetadata $bubbleable_metadata = NULL) {
    $url_scheme = 'http';
    $port = 80;
    if ($request) {
      $url_scheme = $request->getScheme();
      $port = $request->getPort();
    }
    $languages = array_flip(array_keys($this->languageManager->getLanguages()));
    // Language can be passed as an option, or we go for current URL language.
    if (!isset($options['language']) || ($options['language'] instanceof LanguageInterface && $options['language']->getId() == LanguageInterface::LANGCODE_NOT_SPECIFIED)) {
      $language_url = $this->languageManager->getCurrentLanguage(LanguageInterface::TYPE_URL);
      $options['language'] = $language_url;
    }
    // We allow only added languages here.
    elseif (!is_object($options['language']) || !isset($languages[$options['language']->getId()])) {
      return $path;
    }

    $lancode = $options['language']->getId();
    $config = $this->config->get('language.negotiation')->get('domain_url');
    if (is_object($options['language']) && !empty($config[$lancode]['prefix'])) {
      $options['prefix'] = $config[$lancode]['prefix'] . '/';
      if ($bubbleable_metadata) {
        $bubbleable_metadata->addCacheContexts(['languages:' . LanguageInterface::TYPE_URL]);
      }
    }
    if (is_object($options['language']) && !empty($config[$lancode]['domain'])) {
      // Save the original base URL. If it contains a port, we need to
      // retain it below.
      if (!empty($options['base_url'])) {
        // The colon in the URL scheme messes up the port checking below.
        $normalized_base_url = str_replace(['https://', 'http://'], '', $options['base_url']);
      }

      // Ask for an absolute URL with our modified base URL.
      $options['absolute'] = TRUE;
      $options['base_url'] = $url_scheme . '://' . $config[$lancode]['domain'];

      // In case either the original base URL or the HTTP host contains a
      // port, retain it.
      if (isset($normalized_base_url) && str_contains($normalized_base_url, ':')) {
        [, $port] = explode(':', $normalized_base_url);
        $options['base_url'] .= ':' . $port;
      }
      elseif (($url_scheme == 'http' && $port != 80) || ($url_scheme == 'https' && $port != 443)) {
        $options['base_url'] .= ':' . $port;
      }

      if (isset($options['https'])) {
        if ($options['https'] === TRUE) {
          $options['base_url'] = str_replace('http://', 'https://', $options['base_url']);
        }
        elseif ($options['https'] === FALSE) {
          $options['base_url'] = str_replace('https://', 'http://', $options['base_url']);
        }
      }

      // Add Drupal's subfolder from the base_path if there is one.
      $options['base_url'] .= rtrim(base_path(), '/');
      if ($bubbleable_metadata) {
        $bubbleable_metadata->addCacheContexts(['languages:' . LanguageInterface::TYPE_URL, 'url.site']);
      }
    }
    return $path;
  }

  /**
   * {@inheritdoc}
   */
  public function getLanguageSwitchLinks(Request $request, $type, Url $url) {
    $links = [];
    $query = [];
    parse_str($request->getQueryString() ?? '', $query);

    foreach ($this->languageManager->getNativeLanguages() as $language) {
      $links[$language->getId()] = [
        // We need to clone the $url object to avoid using the same one for all
        // links. When the links are rendered, options are set on the $url
        // object, so if we use the same one, they would be set for all links.
        'url' => clone $url,
        'title' => $language->getName(),
        'language' => $language,
        'attributes' => ['class' => ['language-link']],
        'query' => $query,
      ];
    }

    return $links;
  }

}
