(function ($, Drupal, drupalSettings, cookies, once) {
  'use strict';
  Drupal.behaviors.advanced_pwa = {
    attach: function (context, settings) {
      const applicationServerPublicKey = drupalSettings.advanced_pwa.public_key;
      const status_all = drupalSettings.advanced_pwa.status_all;
      const baseUrl = (window.location.protocol + '//' + window.location.host) + (drupalSettings.path.baseUrl);

      navigator.serviceWorker.addEventListener('message', event => {
        if (event.data && event.data.type === 'reload') {
          // Reload the page
          location.reload(true);
        }
      });

      if (!(applicationServerPublicKey)) {
        return;
      }

      if (!('serviceWorker' in navigator)) {
        // Service Worker isn't supported on this browser, disable or hide UI.
        // console.log('service worker not supported');
        return;
      }

      if (!('PushManager' in window)) {
        // Push isn't supported on this browser, disable or hide UI.
        // console.log('PushManager not supported');
        return;
      }

      // Requesting notification permission
      if (!('Notification' in window)) {
        // Notification isn't supported on this browser, disable or hide UI.
        // console.log('Notification not supported');
        return;
      }
      else {
        // console.log('Notification is supported');
      }

      if (Notification.permission === 'denied') {
        // console.log('Notification permission denied');
        return;
      }


      // register service worker
      if ('serviceWorker' in navigator) {
        navigator.serviceWorker.register('/serviceworker-advanced_pwa_js', {scope: '/'})
          .then(function (registration) {
            // console.log('Registration successful, scope is:', registration.scope);
          })
          .catch(function (error) {
            // console.log('Service worker registration failed, error:', error);
          });

        // Then later, request a one-off sync:
        navigator.serviceWorker.ready.then(function (registration) {
          return registration.sync.register('synFirstSync');
        });
      }

      window.addEventListener('beforeinstallprompt', function (e) {
        e.userChoice.then(function (choiceResult) {
          // console.log(choiceResult.outcome);
          if (choiceResult.outcome === 'dismissed') {
            // console.log('User cancelled homescreen install');
          }
          else {
            // console.log('User added to homescreen');
          }
        });
      });


      window.addEventListener('appinstalled', (evt) => {
        app.logEvent('advanced_pwa', 'installed');
      });

      // To determine if the app was launched in standalone mode in non-Safari browsers.
      if (window.matchMedia('(display-mode: standalone)').matches) {
        // console.log('display-mode is standalone');
      }

      // To determine if the app was launched in standalone mode in Safari.
      if (window.navigator.standalone === true) {
        // console.log('display-mode is standalone');
      }

      function urlB64ToUint8Array(base64String) {
        const padding = '='.repeat((4 - base64String.length % 4) % 4);
        const base64 = (base64String + padding)
          .replace(/-/g, '+')
          .replace(/_/g, '/');

        const rawData = window.atob(base64);
        const outputArray = new Uint8Array(rawData.length);

        for (let i = 0; i < rawData.length; ++i) {
          outputArray[i] = rawData.charCodeAt(i);
        }
        return outputArray;
      }

      function updatePushSubscription() {
        navigator.serviceWorker.ready.then(function (registration) {
          registration.pushManager.getSubscription().then(function (sub) {
            if (!sub) {
              // console.log('Not subscribed to push service!');
              subscribeUser();
              return;
            }
            else {
              // We have a subscription, update the database
              // console.log('Subscription object: ', JSON.stringify(sub));
            }
          })
            .catch(function (e) {
              //  console.log('Error subscribing: ', e);
            });
        });
      }

      function subscribeUser() {
        // console.log('subscribeUser');
        const applicationServerKey = urlB64ToUint8Array(applicationServerPublicKey);

        navigator.serviceWorker.ready.then(function (registration) {
          registration.pushManager.subscribe({
            userVisibleOnly: true,
            applicationServerKey: applicationServerKey
          }).then(function (sub) {
            // console.log('Endpoint URL: ', JSON.stringify(sub));
            return subscribeToBackEnd(sub);
          }).catch(function (e) {
            if (Notification.permission === 'denied') {
              // console.warn('Permission for notifications was denied');
            }
            else {
              // console.error('Unable to subscribe to push', e);
            }
          });
        });
      }

      function subscribeToBackEnd(subscription) {
        const key = subscription.getKey('p256dh');
        const token = subscription.getKey('auth');
        var subcribe_url = baseUrl + 'advanced_pwa/subscribe';
        // console.log('sendSubscriptionToBackEnd ', subscription);

        return fetch(subcribe_url, {
          method: 'POST',
          body: JSON.stringify({
            endpoint: subscription.endpoint,
            key: key ? btoa(String.fromCharCode.apply(null, new Uint8Array(key))) : null,
            token: token ? btoa(String.fromCharCode.apply(null, new Uint8Array(token))) : null
          })
        }).then(function (resp) {
          // Transform the data into json.
          resp = resp.json();
          //  console.log('subscribeToBackEnd ', resp);
        }).then(function (data) {
          // console.log('subscribeToBackEnd ', data);
        }).catch(function (e) {
          // console.log('Unable to send subscription to backend:', e);
        });
      }


      /*
      function unsubscribeUser() {
        if ('serviceWorker' in navigator) {
          navigator.serviceWorker.ready.then(function(registration) {
            registration.pushManager.getSubscription()
            .then(function(subscription) {
              if (subscription) {
                unSubscribeFromBackEnd(subscription);
                return subscription.unsubscribe();
              }
            })
            .catch(function(e) {
              console.log('Error unsubscribing', e);
            })
            .then(function() {
              console.log('User is unsubscribed.');
              isSubscribed = false;
            });
          });
        }
      }

      function unSubscribeFromBackEnd(subscription) {
        const key = subscription.getKey('p256dh');
        const token = subscription.getKey('auth');
        var subcribe_url = baseUrl + 'advanced_pwa/unsubscribe';
        console.log('sendSubscriptionToBackEnd ', subscription);

        return fetch(subcribe_url, {
          method: 'POST',
          body: JSON.stringify({
            endpoint: subscription.endpoint,
          })
        }).then(function(resp) {
           // Transform the data into json.
           resp = resp.json();
           comsole.log('unSubscribeFromBackEnd ', resp);
        }).then(function(data) {
          console.log('unSubscribeFromBackEnd ', data);
        }).catch(function (e) {
          console.log('Unable to un-subscribe from backend:', e);
        });
      }
      */
     
      var promptText = drupalSettings.advanced_pwa.prompt_text;
      var promptTitle = drupalSettings.advanced_pwa.prompt_title;
      var confirmText = drupalSettings.advanced_pwa.confirm_text;
      var declineText = drupalSettings.advanced_pwa.decline_text;
      var repeatPrompt = parseInt(drupalSettings.advanced_pwa.repeat_prompt);
      // Here 3 ways to display confirmation: As modal, bootstrap modal or as embedded div.
      var confirmation = {
        modal: Drupal.dialog('<div class="advanced_pwa_message_div" style="display: none !important;">' + promptText + '</div>', {
          title: promptTitle,
          dialogClass: 'advanced_pwa-model-popup',
          resizable: false,
          buttons: [
            {
              text: confirmText,
              class: 'button button--allow',
              click: function () {
                updatePushSubscription();
                cookies.set('advanced_pwa_prompt_seen', '1', { expires: repeatPrompt, path: "/" });
                confirmation.modal.close();
              }
            },
            {
              text: declineText,
              class: 'button button--cancel',
              click: function () {
                cookies.set('advanced_pwa_prompt_seen', '1', { expires: repeatPrompt, path: "/" });
                confirmation.modal.close();
              }
            }
          ],
          closeOnEscape: false,
          create: function () {

          },
          beforeClose: false,
          close: function (event) {
            // Automatically destroy the DOM element that was used for the dialog.
            // $(event.target).remove();
          }
        }),
        embedded: {
          open: function() {
            $(once('apwa', 'body')).append('<div class="advanced_pwa-model-embedded">'
              + '<span class="text">' + promptText + '</span>'
              + '<span class="button js--apwa-allow">' + confirmText + '</span>'
              + '<span class="button js--apwa-cancel">' + declineText + '</span>'
            + '</div>');
            // Let's handle the event.
            $(document).on('click', '.advanced_pwa-model-embedded .button', function(event) {
              if ($(this).hasClass('js--apwa-allow')) {
                confirmation.embedded.accept();
              }
              if ($(this).hasClass('js--apwa-cancel')) {
                confirmation.embedded.cancel();
              }
            });
          },
          close: function() {
            cookies.set('advanced_pwa_prompt_seen', '1', { expires: repeatPrompt, path: "/" });
            $('body').find('.advanced_pwa-model-embedded').remove();
          },
          accept: function() {
            updatePushSubscription();
            confirmation.embedded.close();
          },
          cancel: function() {
            confirmation.embedded.close();
          }
        },
        bootstrap_modal: {
          open: function() {
            // Add the modal content to the page
            $(once('apwa', 'body')).append('<div id="apwa-bootstrap-modal" class="modal fade" tabindex="-1">'
            + '<div class="modal-dialog">'
            + '<div class="modal-content">'
            + '<div class="modal-header">'
            + '<h5 class="modal-title">' + promptTitle + '</h5>'
            + '</div>'
            + '<div class="modal-body">'
            + '<p>' + promptText + '</p>'
            + '<div class="modal-footer">'
            + '<button type="button" class="btn btn-outline-danger js--apwa-cancel" data-dismiss="modal">' + declineText + '</button>&nbsp; &nbsp;'
            + '<button type="button" class="btn btn-primary js--apwa-allow">' + confirmText + '</button></div>'
            + '</div></div></div></div>');
            
            // Trigger the modal dialog.
            setTimeout(function() {
		          $('#apwa-bootstrap-modal').modal('show');
            }, 15);
            
            // Let's handle the event.
            $(document).on('click', '#apwa-bootstrap-modal .btn', function(event) {
              if ($(this).hasClass('js--apwa-allow')) {
                confirmation.bootstrap_modal.accept();
              }
              if ($(this).hasClass('js--apwa-cancel')) {
                confirmation.bootstrap_modal.cancel();
              }
            });
          },
          close: function() {
            cookies.set('advanced_pwa_prompt_seen', '1', { expires: repeatPrompt, path: "/" });
            $('#apwa-bootstrap-modal').modal('hide');
          },
          accept: function() {
            updatePushSubscription();
            confirmation.bootstrap_modal.close();
          },
          cancel: function() {
            confirmation.bootstrap_modal.close();
          }
        }
    };
        
        // Checking if the user is subcribed for notification, if not popup will appear.
      navigator.serviceWorker.ready.then(serviceWorkerRegistration => serviceWorkerRegistration.pushManager.getSubscription())
        .then(subscription => {
          if (status_all === 1) {
            var hasPermission = drupalSettings.advanced_pwa.has_permission;
            var seen = cookies.get('advanced_pwa_prompt_seen');
            if (!subscription && hasPermission && !seen) {
              // We aren't subscribed to push, so enable subscription.
              var displayType = drupalSettings.advanced_pwa.display_type || 'modal';
              switch (displayType) {
                case 'modal':
                  confirmation.modal.showModal();
                  break;

                case 'embedded':
                  confirmation.embedded.open();
                  break;
                  
                case 'bootstrap_modal':
                  confirmation.bootstrap_modal.open();
                  break;
              }
              // return;.
            } else if (!subscription && hasPermission) {
                $(".push-prompt-link").show();
            } else {
                $(".push-prompt-link").hide();
            }
          }
          else {
            // console.log('notification feature disabled from configuration form');
          }
        })
        .then(subscription => subscription)
        .catch(e => {
          // console.error('Error when updating the subscription', e);
        });
        
        $(".trigger-push-prompt").click(function(){
            var displayType = drupalSettings.advanced_pwa.display_type || 'modal';
              switch (displayType) {
                case 'modal':
                  confirmation.modal.showModal();
                  break;

                case 'embedded':
                  confirmation.embedded.open();
                  break;
                  
                case 'bootstrap_modal':
                  confirmation.bootstrap_modal.open();
                  break;
              }
        });
    }
  };
})(jQuery, Drupal, drupalSettings, window.Cookies, once);
