<?php

namespace Drupal\advanced_sanitize\Form;

use Drupal\advanced_sanitize\AdvancedSanitizeService;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Advanced Sanitize settings for this site.
 */
class SanitizeSettingsForm extends ConfigFormBase {


  /**
   * The accounts sync service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Advanced sanitize service.
   *
   * @var \Drupal\advanced_sanitize\AdvancedSanitizeService
   */
  protected $advancedSanitizeService;

  /**
   * Constructs a CustomerSyncForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\advanced_sanitize\AdvancedSanitizeService $advanced_sanitize
   *   Advanced sanitize service.
   */
  public function __construct(ConfigFactoryInterface $configFactory,
                              FileSystemInterface $file_system,
                              AdvancedSanitizeService $advanced_sanitize) {
    parent::__construct($configFactory);
    $this->fileSystem = $file_system;
    $this->advancedSanitizeService = $advanced_sanitize;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('file_system'),
      $container->get('advanced_sanitize.service'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'advanced_sanitize_sanitize_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['advanced_sanitize.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    $form['config_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Path to config'),
      '#description' => $this->t('Specify path to sanitize config file relative to Drupal root.'),
      '#default_value' => $this->config('advanced_sanitize.settings')->get('config_path'),
    ];
    $form['actions']['sanitize'] = [
      '#type' => 'submit',
      '#value' => $this->t('Sanitize'),
      '#submit' => ['::launchSanitize'],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if (
      empty($form_state->getValue('config_path')) ||
      !$this->fileSystem->realpath(DRUPAL_ROOT . '/' . $form_state->getValue('config_path'))
    ) {
      $form_state->setErrorByName('config_path', $this->t('Found no config by given path!'));
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('advanced_sanitize.settings')
      ->set('config_path', $form_state->getValue('config_path'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function launchSanitize(array &$form, FormStateInterface $form_state) {
    if (!empty($this->config('advanced_sanitize.settings')
      ->get('config_path'))) {
      $this->advancedSanitizeService->sanitize(FALSE);
    }
    else {
      $this->messenger()->addWarning('Path to sanitize configuration file is not set!');
    }
  }

}
