<?php

namespace Drupal\advancedqueue\Entity;

use Drupal\advancedqueue\Job;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\EntityWithPluginCollectionInterface;

/**
 * Defines the interface for queues.
 *
 * The enqueueJob() and enqueueJobs() methods are copied from BackendInterface
 * for DX reasons. Most modules will only interact with those methods, so
 * this saves them a getBackend() call.
 */
interface QueueInterface extends ConfigEntityInterface, EntityWithPluginCollectionInterface {

  /**
   * Available queue processors.
   */
  const PROCESSOR_CRON = 'cron';
  const PROCESSOR_DAEMON = 'daemon';

  /**
   * Enqueues the given job.
   *
   * The job will be modified with the assigned queue ID, job ID, and
   * relevant timestamps.
   *
   * @param \Drupal\advancedqueue\Job $job
   *   The job.
   * @param int $delay
   *   The time, in seconds, after which the job will become available to
   *   consumers. Defaults to 0, indicating no delay.
   */
  public function enqueueJob(Job $job, $delay = 0);

  /**
   * Enqueues the given jobs.
   *
   * Each job will be modified with the assigned queue ID, job ID, and
   * relevant timestamps.
   *
   * @param \Drupal\advancedqueue\Job[] $jobs
   *   The jobs.
   * @param int $delay
   *   The time, in seconds, after which the jobs will become available to
   *   consumers. Defaults to 0, indicating no delay.
   */
  public function enqueueJobs(array $jobs, $delay = 0);

  /**
   * Gets the backend plugin.
   *
   * @return \Drupal\advancedqueue\Plugin\AdvancedQueue\Backend\BackendInterface
   *   The backend plugin.
   */
  public function getBackend();

  /**
   * Gets the backend plugin ID.
   *
   * @return string
   *   The backend plugin ID.
   */
  public function getBackendId();

  /**
   * Sets the backend plugin ID.
   *
   * @param string $backend_id
   *   The backend plugin ID.
   *
   * @return $this
   */
  public function setBackendId($backend_id);

  /**
   * Gets the backend plugin configuration.
   *
   * @return string
   *   The backend plugin configuration.
   */
  public function getBackendConfiguration();

  /**
   * Sets the backend plugin configuration.
   *
   * @param array $configuration
   *   The backend plugin configuration.
   *
   * @return $this
   */
  public function setBackendConfiguration(array $configuration);

  /**
   * Gets the queue processor.
   *
   * @return string
   *   The queue processor, one of the PROCESSOR_ constants.
   */
  public function getProcessor();

  /**
   * Sets the queue processor.
   *
   * @param string $processor
   *   The queue processor, one of the PROCESSOR_ constants.
   *
   * @return $this
   */
  public function setProcessor($processor);

  /**
   * Gets the queue processing time.
   *
   * Indicates how long the processor should process the queue.
   *
   * @return int
   *   The queue processing time, in seconds. 0 for unlimited.
   */
  public function getProcessingTime();

  /**
   * Sets the queue processing time.
   *
   * @param int $processing_time
   *   The queue processing time, in seconds. 0 for unlimited.
   *
   * @return $this
   */
  public function setProcessingTime($processing_time);

  /**
   * Gets whether the queue is locked.
   *
   * Locked queues cannot be deleted.
   *
   * @return bool
   *   TRUE if the queue is locked, FALSE otherwise.
   */
  public function isLocked();

}
