<?php

namespace Drupal\advban;

use Drupal\Component\Datetime\Time;
use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\PagerSelectExtender;

/**
 * Ban IP manager.
 */
class AdvbanIpManager extends ControllerBase implements AdvbanIpManagerInterface {

  /**
   * The database connection used to check the IP against.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactory
   */
  protected $config;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\Time
   */
  protected $time;

  /**
   * The IP ban metadata.
   *
   * @var array
   */
  protected $metadata;

  /**
   * Construct the AdvbanIpManager.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection which will be used to check the IP against.
   * @param \Drupal\Core\Config\ConfigFactory $config
   *   The configuration factory service.
   * @param \Drupal\Component\Datetime\Time $time
   *   The time service.
   */
  public function __construct(Connection $connection, ConfigFactory $config, Time $time) {
    $this->connection = $connection;
    $this->config = $config;
    $this->time = $time;
    $this->metadata = [];
  }

  /**
   * {@inheritdoc}
   */
  public function isBanned($ip, array $options = []) {
    // Merge in defaults.
    $options += [
      'expiry_check' => TRUE,
      'info_output' => FALSE,
      'no_limit' => FALSE,
    ];

    // Collect result info.
    if ($options['info_output']) {
      $result_info = ['iid' => '', 'expiry_date' => ''];
    }

    if (!$options['expiry_check']) {
      $ban_info = $this->connection->query("SELECT iid FROM {advban_ip} WHERE ip = :ip", [':ip' => $ip])->fetchField();
      $is_banned = (bool) $ban_info;
      if ($is_banned && $options['info_output']) {
        $result_info['iid'] = $ban_info;
      }
    }
    else {
      $ban_info = $this->connection->query("SELECT iid, expiry_date FROM {advban_ip} WHERE ip = :ip", [':ip' => $ip])->fetchAll();
      $is_banned = count($ban_info) && (empty($ban_info[0]->expiry_date) || $ban_info[0]->expiry_date > $this->time->getRequestTime());
      if ($is_banned && $options['info_output']) {
        $result_info['iid'] = $ban_info[0]->iid;
        $result_info['expiry_date'] = $ban_info[0]->expiry_date;
      }
    }

    if (!$is_banned) {
      // Check for a range ban.
      $ip_long = ip2long($ip);
      if ($ip_long) {
        $limit = $options['no_limit'] ? NULL : 1;
        if (!$options['expiry_check']) {
          if (!$limit) {
            $ban_info = $this->connection->query("SELECT iid FROM {advban_ip} WHERE ip_end <> '' AND ip <= $ip_long AND ip_end >= $ip_long")->fetchAll();
            $is_banned = count($ban_info);
            if ($is_banned && $options['info_output']) {
              $result_info['iid'] = [];
              foreach ($ban_info as $item) {
                $result_info['iid'][] = $item->iid;
              }
            }
          }
          else {
            $ban_info = $this->connection->queryRange("SELECT iid FROM {advban_ip} WHERE ip_end <> '' AND ip <= $ip_long AND ip_end >= $ip_long", 0, $limit)->fetchField();
            $is_banned = (bool) $ban_info;
            if ($is_banned && $options['info_output']) {
              $result_info['iid'] = $ban_info;
            }
          }
        }
        else {
          if ($limit) {
            $query = $this->connection->queryRange("SELECT iid, expiry_date FROM {advban_ip} WHERE ip_end <> '' AND ip <= $ip_long AND ip_end >= $ip_long", 0, $limit);
          }
          else {
            $query = $this->connection->query("SELECT iid, expiry_date FROM {advban_ip} WHERE ip_end <> '' AND ip <= $ip_long AND ip_end >= $ip_long");
          }
          $ban_info = $query->fetchAll();
          $is_banned = count($ban_info) && (empty($ban_info[0]->expiry_date) || $ban_info[0]->expiry_date > $this->time->getRequestTime());
          if ($is_banned && $options['info_output']) {
            $result_info['iid'] = [];
            foreach ($ban_info as $item) {
              $result_info['iid'][] = $item->iid;
            }
            $result_info['expiry_date'] = $ban_info[0]->expiry_date;
          }
        }
      }
    }

    if ($options['info_output']) {
      $result_info['is_banned'] = $is_banned;
      return $result_info;
    }
    else {
      return $is_banned;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function isProtected($ip, $ip_end = NULL) {
    $ip = ip2long($ip);
    if (!$ip) {
      return FALSE;
    }

    if (!empty($ip_end)) {
      $ip_end = ip2long($ip_end);
      if (!$ip_end) {
        return FALSE;
      }
    }

    $protected_ips = explode(PHP_EOL, trim($this->config->get('advban.settings')->get('advban_protected_ips') ?? ''));
    if (empty($protected_ips)) {
      return FALSE;
    }

    if (filter_var($ip, FILTER_VALIDATE_IP)) {
      $real_host = gethostbyaddr($ip);
    }

    foreach ($protected_ips as $protected_ip) {
      $protected_ip = trim($protected_ip);
      if (empty($protected_ip)) {
        continue;
      }

      // Block comment.
      if (substr($protected_ip, 0, 1) == '#') {
        continue;
      }

      // Inline comment.
      $protected_ip_arr = explode('#', $protected_ip);
      if (count($protected_ip_arr) > 1) {
        $protected_ip = trim($protected_ip_arr[0]);
      }

      // Check for CIDR.
      $protected_ip_arr = explode('/', $protected_ip);
      if (count($protected_ip_arr) > 1) {
        if (ip2long($protected_ip_arr[0])) {
          $in_list = $this->cidrMatch($ip, $protected_ip_arr[0], (int) $protected_ip_arr[1]);
          if (!empty($ip_end)) {
            $in_list_end = $this->cidrMatch($ip_end, $protected_ip_arr[0], (int) $protected_ip_arr[1]);
            if ($in_list && $in_list_end) {
              return TRUE;
            }
          }
          else {
            if ($in_list) {
              return TRUE;
            }
          }
        }
      }
      else {
        $ip_check = ip2long($protected_ip);

        if (empty($ip_end)) {
          if ($ip_check === $ip) {
            return TRUE;
          }
        }
        else {
          if ($ip_check <= $ip_end && $ip <= $ip_check) {
            return TRUE;
          }
        }
      }

      // Check for domain.
      if (!empty($real_host)) {
        $real_host_arr = explode($protected_ip, $real_host);
        if (count($real_host_arr) == 2 && empty($real_host_arr[1])) {
          return TRUE;
        }
      }
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function isBannedByReason($reason) {
    $result_info = ['iid' => '', 'expiry_date' => ''];

    $ban_info = $this->connection->query("SELECT iid FROM {advban_ip} WHERE reason LIKE :reason", [':reason' => '%' . $reason . '%'])->fetchField();
    $is_banned = (bool) $ban_info;
    if ($is_banned) {
      $result_info['iid'] = $ban_info;
    }
    $result_info['is_banned'] = $is_banned;

    return $result_info;
  }

  /**
   * {@inheritdoc}
   */
  public function findAll($limit = -1) {
    $query = $this->connection->select('advban_ip', 'a')
      ->fields('a');
    if ($limit > 0) {
      $query = $query->extend(PagerSelectExtender::class)
        ->limit($limit);
    }
    return $query->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function banIp($ip, $ip_end = '', $reason = '', $expiry_date = NULL) {
    if (!empty($ip_end)) {
      if (!is_numeric($ip)) {
        $ip = sprintf("%u", ip2long($ip));
      }

      if (!is_numeric($ip_end)) {
        $ip_end = sprintf("%u", ip2long($ip_end));
      }

      $fields = ['ip' => $ip, 'ip_end' => $ip_end];
    }
    else {
      $fields = ['ip' => $ip];
    }

    // Set reason field.
    if (empty($reason)) {
      $metadata = $this->getMetadata();
      if (!empty($metadata)) {
        $reason = $metadata['reporter'] . ':' . $metadata['id'];
      }
    }
    $fields['reason'] = $reason ?? '';

    // Set expiry date using defaut expiry durations.
    if (!$expiry_date) {
      $expiry_duration = $this->config->get('advban.settings')->get('default_expiry_duration');
      if ($expiry_duration && $expiry_duration != AdvbanHelper::ADVBAN_NEVER) {
        $expiry_date = strtotime($expiry_duration);
        if (!$expiry_date) {
          $this->messenger()->addMessage($this->t('Wrong expiry date for duration %expiry_duration', ['%expiry_duration' => $expiry_duration]),
            'error');
          return;
        }
      }
    }

    $fields['expiry_date'] = $expiry_date ?: 0;

    $this->connection->merge('advban_ip')
      ->key('ip', $ip)
      ->fields($fields)
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function unbanIp($ip, $ip_end = '') {
    $query = $this->connection->delete('advban_ip');
    if (!empty($ip_end)) {
      $query->condition('ip', $ip);
      $query->condition('ip_end', $ip_end);
    }
    else {
      $query->condition('ip', $ip);
    }
    $query->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function unbanIpAll(array $params = []) {
    $query = $this->connection->delete('advban_ip');
    if (!empty($params)) {
      // Range parameters.
      if (!empty($params['range']) && $params['range'] != 'all') {
        switch ($params['range']) {
          case 'simple':
            $query->condition('ip_end', '');
            break;

          case 'range':
            $query->condition('ip_end', '', '<>');
            break;

        }
      }

      // Expire parameters.
      if (!empty($params['expire']) && $params['expire'] != 'all') {
        switch ($params['expire']) {
          case 'expired':
            $query->condition('expiry_date', 0, '<>');
            break;

          case 'not_expired':
            $query->condition('expiry_date', 0);
            break;
        }
      }
    }
    return $query->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function findById($ban_id) {
    return $this->connection->query("SELECT * FROM {advban_ip} WHERE iid = :iid", [':iid' => $ban_id])->fetchAll();
  }

  /**
   * {@inheritdoc}
   */
  public function formatIp($ip, $ip_end = '') {
    if (!empty($ip_end)) {
      if (is_numeric($ip)) {
        $ip = long2ip($ip);
      }
      if (is_numeric($ip_end)) {
        $ip_end = long2ip($ip_end);
      }

      $format_text = $this->config->get('advban.settings')
        ->get('range_ip_format') ?: '@ip_start ... @ip_end';
      $text = new FormattableMarkup($format_text, [
        '@ip_start' => $ip,
        '@ip_end' => $ip_end,
      ]);
      return $text;
    }
    else {
      return $ip;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function expiryDurations($index = NULL) {
    $expiry_durations = $this->config->get('advban.settings')->get('expiry_durations');

    if (empty($expiry_durations)) {
      $durations = [
        '+1 hour',
        '+1 day',
        '+1 week',
        '+1 month',
        '+1 year',
      ];
      $expiry_durations = implode(PHP_EOL, $durations);

      $this->config->getEditable('advban.settings')
        ->set('expiry_durations', $expiry_durations)
        ->save();
    }

    $list = explode(PHP_EOL, $expiry_durations);
    $list = array_map('trim', $list);
    if ($index != NULL) {
      $list[] = AdvbanHelper::ADVBAN_NEVER;
      return $list[$index] ?? AdvbanHelper::ADVBAN_NEVER;
    }

    return $list;
  }

  /**
   * {@inheritdoc}
   */
  public function expiryDurationIndex(array $expiry_durations, $default_expiry_duration) {
    if (!$default_expiry_duration || $default_expiry_duration == AdvbanHelper::ADVBAN_NEVER) {
      $expiry_durations_index = AdvbanHelper::ADVBAN_NEVER;
    }
    else {
      $expiry_durations_index = array_search(trim($default_expiry_duration), $expiry_durations);
      if ($expiry_durations_index === FALSE) {
        $expiry_durations_index = AdvbanHelper::ADVBAN_NEVER;
      }
    }

    return $expiry_durations_index;
  }

  /**
   * {@inheritdoc}
   */
  public function unblockExpiredIp() {
    $query = $this->connection->delete('advban_ip');
    $query->condition('expiry_date', 0, '>');
    $query->condition('expiry_date', strtotime('now'), '<');
    return $query->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function banText(array $variables) {
    $ban_text = $this->config->get('advban.settings')->get('advban_ban_text') ?: '@ip has been banned';
    $ban_text_params = ['@ip' => $variables['ip']];
    $expiry_date = $variables['expiry_date'];
    if (!empty($expiry_date)) {
      $ban_text = $this->config->get('advban.settings')->get('advban_ban_expire_text') ?: '@ip has been banned up to @expiry_date';
      $ban_text_params['@expiry_date'] = date('r', $expiry_date);
    }
    return new FormattableMarkup($ban_text, $ban_text_params);
  }

  /**
   * {@inheritdoc}
   */
  public function getEntryStatus($ip) {
    if ($this->isProtected($ip->ip)) {
      return $this->t('Protected');
    }

    $now = $this->time->getRequestTime();
    return !empty($ip->expiry_date) && $ip->expiry_date < $now
      ? $this->t('Expired') : $this->t('Active');
  }

  /**
   * {@inheritdoc}
   */
  public function setMetadata(array $data) {
    $this->metadata = $data;
  }

  /**
   * {@inheritdoc}
   */
  public function getMetadata() {
    return $this->metadata;
  }

  /**
   * Is IP address in subnet?
   *
   * @param int $ip
   *   IP address for match check.
   * @param string $network
   *   IP subnet.
   * @param int $cidr
   *   CIDR.
   *
   * @return bool
   *   IP matches for subnet.
   */
  private function cidrMatch($ip, $network, $cidr) {
    return (($ip & ~((1 << (32 - $cidr)) - 1)) == ip2long($network));
  }

}
