<?php

/**
 * Implementation of hook_field_info().
 */
function aef_image_field_info() {
  return array(
    'aef_image' => array(
      'label' => t('Uberimage'),
      'description' => t('Advanced image field'),
    ),
  );
}

/**
 * Implementation of hook_field_settings().
 */
function aef_image_field_settings($op, $field) {
  switch ($op) {
    case 'form':
      $form = array();
      return $form;

    case 'save':
      return array();

    case 'database columns':
      $columns['nid'] = array('type' => 'int', 'unsigned' => TRUE, 'not null' => FALSE, 'views' => TRUE);
      $columns['fid'] = array('type' => 'int', 'unsigned' => TRUE, 'not null' => FALSE, 'views' => TRUE);
      $columns['data'] = array('type' => 'text', 'size' => 'big', 'not null' => FALSE, 'serialize' => true, 'views' => FALSE);
      $columns['title'] = array('type' => 'varchar', 'length' => '255', 'not null' => FALSE, 'description' => 'The title of the image', 'views' => TRUE);
      $columns['legend'] = array('type' => 'varchar', 'length' => '511', 'not null' => FALSE, 'description' => 'The legend of the image', 'views' => TRUE);
      $columns['credits'] = array('type' => 'varchar', 'length' => '255', 'not null' => FALSE, 'description' => 'The credits of the image', 'views' => TRUE);
      return $columns;

  }
}

/**
 * Implementation of hook_field().
 */
function aef_image_field($op, &$node, $field, &$items, $teaser, $page) {
  switch ($op) {
    case 'validate':
      return $items;

    case 'sanitize':
  }
}


/**
 * Implementation of hook_content_is_empty().
 */
function aef_image_content_is_empty($item, $field) {
  if ((empty($item['nid']) || (string)$item['nid'] === '0') && 
    (empty($item['fid']) || (string)$item['fid'] === '0')) {
    return TRUE;
  }
  return FALSE;
}

/**
 * Implementation of hook_field_formatter_info().
 */
function aef_image_field_formatter_info() {
  $result = array(
    'default' => array(
      'label' => t('Default: Image with title, legend and credits'),
      'field types' => array('aef_image'),
      'multiple values' => CONTENT_HANDLE_CORE,
    ),
    'imageonly' => array(
      'label' => t('Image only'),
      'field types' => array('aef_image'),
      'multiple values' => CONTENT_HANDLE_CORE,
    ),
  );

  //Custom imagecache formatters selected in the admin panel
  foreach(variable_get('aef_image_generated_imagecache_formatters', array()) as $imagecache_preset)
  {
    $result['default_' . $imagecache_preset] = array(
      'label' => t('Image with title, legend and credits (!preset preset)', array('!preset' => $imagecache_preset)),
      'field types' => array('aef_image'),
      'multiple values' => CONTENT_HANDLE_CORE,
    );
    $result['imageonly_' . $imagecache_preset] = array(
      'label' => t('Image only (!preset preset)', array('!preset' => $imagecache_preset)),
      'field types' => array('aef_image'),
      'multiple values' => CONTENT_HANDLE_CORE,
    );
  }
  return $result;
}

/**
 * Theme function for 'default' text field formatter.
 */
function theme_aef_image_formatter_default($element) {

	return _aef_image_get_formatter_output($element, 'aef_image_default');
}

/**
 * Theme function for 'imageonly' text field formatter.
 */
function theme_aef_image_formatter_imageonly($element) {

	return _aef_image_get_formatter_output($element, 'aef_image_imageonly');
}


function _aef_image_get_formatter_output($element, $theme_name)
{
	// Inside a View this function may be called with null data.  In that case,
	// just return.
	if (empty($element['#item'])) {
		return '';
	}

  $field = content_fields($element['#field_name'], $element['#type_name']);

	$html = "";

  _aef_image_unserialize_data($element['#item']['data']);

  //If preset selection is on, use this preset. Otherwise use the default one.
  $preset_name = $field['widget']['default_imagecache_preset'];
  if($field['widget']['preset_selection'] == 1)
    $preset_name = $element['#item']['data']['preset_selection'];

  $html = theme($theme_name, $element['#item']['nid'], $element['#item']['fid'], $element['#item']['data'], $preset_name, '', '', array(), $element['#item']['title'], $element['#item']['legend'], $element['#item']['credits']);

  return $html;
}

/**
 * Theme function of the preset formatters
 */
function theme_aef_image_formatter_default_preset($element) {
	// Inside a View this function may be called with null data.  In that case,
	// just return.
	if (empty($element['#item'])) {
		return '';
	}

  $field = content_fields($element['#field_name'], $element['#type_name']);

	$html = "";

  _aef_image_unserialize_data($element['#item']['data']);

  //Get the imagecache preset to use (remove "default_" at the beginning of the formatter name
  $preset_name = substr($element['#formatter'], 8); 

  $html = theme('aef_image_default', $element['#item']['nid'], $element['#item']['fid'], $element['#item']['data'], $preset_name, '', '', array(), $element['#item']['title'], $element['#item']['legend'], $element['#item']['credits']);

  return $html;
}

function theme_aef_image_formatter_imageonly_preset($element) {
	// Inside a View this function may be called with null data.  In that case,
	// just return.
	if (empty($element['#item'])) {
		return '';
	}

  $field = content_fields($element['#field_name'], $element['#type_name']);

	$html = "";

  _aef_image_unserialize_data($element['#item']['data']);

  //Get the imagecache preset to use (remove "imageonly_" at the beginning of the formatter name
  $preset_name = substr($element['#formatter'], 10); 

  $html = theme('aef_image_imageonly', $element['#item']['nid'], $element['#item']['fid'], $element['#item']['data'], $preset_name, '', '', array(), $element['#item']['title'], $element['#item']['legend'], $element['#item']['credits']);

  return $html;
}


/**
 * Implementation of hook_widget_info().
 *
 * Here we indicate that the content module will handle
 * the default value and multiple values for these widgets.
 *
 * Callbacks can be omitted if default handing is used.
 * They're included here just so this module can be used
 * as an example for custom modules that might do things
 * differently.
 */
function aef_image_widget_info() {
  return array(
    'aef_image_creator' => array(
      'label' => t('Uberimage advanced widget'),
      'field types' => array('aef_image'),
      'multiple values' => CONTENT_HANDLE_CORE,
      'callbacks' => array(
        'default value' => CONTENT_CALLBACK_DEFAULT,
      ),
    ),
  );
}

/**
 * Implementation of FAPI hook_elements().
 *
 * Any FAPI callbacks needed for individual widgets can be declared here,
 * and the element will be passed to those callbacks for processing.
 *
 * Drupal will automatically theme the element using a theme with
 * the same name as the hook_elements key.
 *
 * Autocomplete_path is not used by text_widget but other widgets can use it
 * (see nodereference and userreference).
 */
function aef_image_elements() {
  return array(
    'aef_image_creator' => array(
      '#input' => TRUE,
      '#columns' => array('nid', 'fid'), 
      '#delta' => 0,
      '#process' => array('aef_image_creator_process'),
      '#element_validate' => array('aef_image_creator_validate'),
      '#value_callback' => 'aef_image_widget_value',
      '#autocomplete_path' => FALSE,
    ),
  );
}

/**
 * Implementation of hook_widget_settings().
 */
function aef_image_widget_settings($op, $widget) {
  switch ($op) {
    case 'form':
      $form = array();

      $form['file_path'] = array(
        '#type' => 'textfield',
        '#title' => t('File path'),
        '#default_value' => is_string($widget['file_path']) ? $widget['file_path'] : '',
        '#description' => t('Optional subdirectory within the "%directory" directory where files will be stored. Do not include preceding or trailing slashes.', array('%directory' => variable_get('file_directory_path', 'files') . '/')),
        '#element_validate' => array('_aef_image_settings_file_path_validate'),
      );

      $default_values = aef_utilities_check_cck_settings_array($widget['image_infos_display']);
      if(count($widget['image_infos_display']) == 0)
        $default_values = array(
          'title' => 'title',
          'legend' => 'legend',
          'credits' => 'credits'
          );
      $form['image_infos_display'] = array(
        '#type' => 'checkboxes',
        '#title' => t('Image info fields to display'),
        '#default_value' => $default_values,
        '#options' => array(
          'title' => t('title'),
          'legend' => t('Legend'),
          'credits' => t('Credits'),
          ),
        '#description' => t('Select the image info fields that the user will be able to edit in the node edit page.'),
      );

      $form['image_noderef'] = array(
        '#type' => 'checkbox',
        '#title' => t('Use nodereference instead of upload'),
        '#default_value' => $widget['image_noderef'],
        '#description' => t('Are the image loaded via an upload widget, or are existing image in node referenced in a nodereference widget?'),
      );

      $form['preset_selection'] = array(
        '#type' => 'checkbox',
        '#title' => t('Allow the selection of the preset'),
        '#default_value' => $widget['preset_selection'],
        '#description' => t('If selected, the user will be able to select the formatter that will be used by default.'),
      );

      $options = array();
      foreach (imagecache_presets() as $preset) {
        //We only support preset with only one imagecache_scale_and_crop for now
        if(count($preset['actions']) == 1 && $preset['actions'][0]['action'] == "imagecache_scale_and_crop")
          $options[$preset['presetname']] = $preset['presetname'];
      }
      $form['imagecache_presets'] = array(
        '#type' => 'checkboxes',
        '#title' => t('Imagecache presets to use'),
        '#default_value' => aef_utilities_check_cck_settings_array($widget['imagecache_presets']),
        '#options' => $options,
        '#description' => t('Select the imagecache presets that will be displayed on the edit widget. You can add imagecache presets with the module "Imagecache UI".'),
      );

      $form['default_imagecache_preset'] = array(
        '#type' => 'select',
        '#title' => t('Default imagecache preset to use in the default display'),
        '#default_value' => $widget['default_imagecache_preset'],
        '#options' => $options,
        '#description' => t('Select the imagecache presets that will be used by default on the node page.'),
      );

      $form['group_imagecache_presets'] = array(
        '#type' => 'checkbox',
        '#title' => t('Group the presets of the same size'),
        '#default_value' => $widget['group_imagecache_presets'],
        '#description' => t('If selected, the presets doing the same resizing will be shown only once.'),
      );

      $options = array('none' => t('None'));
      $content_types = content_types();
      foreach ($content_types as $name => $info) {
        $options[$name] = $info['name'];
      }
      $form['create_new_nodetype'] = array(
        '#type' => 'select',
        '#title' => t('Node type of the new node link'),
        '#description' => t('Module Embedded Edit needed. Only useful when using the nodereference mode. Select the node type you will create using the "Create new" link. "None" means there won\'t be a link.'),
        '#options' => $options,
        '#default_value' => $widget['create_new_nodetype'],
      );

      return $form;

    case 'validate':

      return;

    case 'save':
      return array('file_path', 'image_infos_display', 'image_noderef', 'preset_selection', 'imagecache_presets', 'default_imagecache_preset', 'group_imagecache_presets', 'create_new_nodetype');
  }
}

function _aef_image_settings_file_path_validate($element, &$form_state) {
  // Strip slashes from the beginning and end of $widget['file_path']
  $form_state['values']['file_path'] = trim($form_state['values']['file_path'], '\\/');

  // Do not allow the file path to be the same as the file_directory_path().
  // This causes all sorts of problems with things like file_create_url().
  if (strpos($form_state['values']['file_path'], file_directory_path()) === 0) {
    form_error($element, t('The file path (@file_path) cannot start with the system files directory (@files_directory), as this may cause conflicts when building file URLs.', array('@file_path' => $form_state['values']['file_path'], '@files_directory' => file_directory_path())));
  }
}

/**
 * Implementation of hook_widget().
 *
 * Attach a single form element to the form. It will be built out and
 * validated in the callback(s) listed in hook_elements. We build it
 * out in the callbacks rather than here in hook_widget so it can be
 * plugged into any module that can provide it with valid
 * $field information.
 *
 * Content module will set the weight, field name and delta values
 * for each form element. This is a change from earlier CCK versions
 * where the widget managed its own multiple values.
 *
 * If there are multiple values for this field, the content module will
 * call this function as many times as needed.
 *
 * @param $form
 *   the entire form array, $form['#node'] holds node information
 * @param $form_state
 *   the form_state, $form_state['values'][$field['field_name']]
 *   holds the field's form values.
 * @param $field
 *   the field array
 * @param $items
 *   array of default values for this field
 * @param $delta
 *   the order of this item in the array of subelements (0, 1, 2, etc)
 *
 * @return
 *   the form item for a single element for this field
 */
function aef_image_widget(&$form, &$form_state, $field, $items, $delta = 0) {
  $element = array(
    '#type' => 'aef_image_creator',
    '#default_value' => isset($items[$delta]) ? $items[$delta] : '',
  );
  return $element;
}

/**
 * Process an individual element.
 *
 * Build the form element. When creating a form using FAPI #process,
 * note that $element['#value'] is already set.
 *
 * The $fields array is in $form['#field_info'][$element['#field_name']].
 */
function aef_image_creator_process($element, $edit, $form_state, $form) {
  //Add some JS. Collapse.js is needed because a collapsible fieldset appears with AHAH.
  jq_add('jcrop');
  drupal_add_js('misc/collapse.js');

  $item = $element['#value'];
  $field = $form['#field_info'][$element['#field_name']];
  $field_key = $element['#columns'][0];
  $delta = $element['#delta'];
  $infos = array();

  //Fetch the infos we need
  $infos = _aef_image_creator_fetch_data($item, aef_utilities_check_cck_settings_array($field['widget']['imagecache_presets']));
  //$item['data'] = $infos['data'];




  //Get the old element state, if any
  $old_element_state = 0;
  if(isset($element['#post'][$element['#field_name'] .'_'. $element['#delta'] . '_old_element_state']))
  {
    $old_element_state = $element['#post'][$element['#field_name'] .'_'. $element['#delta'] . '_old_element_state'];
  }

  //Determine in which state we are
  //State = 1 : Initial state, only the image selection widget is here
  //State = 2 : Image is loaded
  //State = 3 : Image is loaded, one preset is being edited.(the name is in $preset_being_edited)
  $preset_being_edited = '';
  $element_state = 1;
  if((!empty($infos['fid']) && (string)$infos['fid'] !== '0') || 
    (!empty($infos['nid']) && (string)$infos['nid'] !== '0'))
  {
    $element_state = 2;
  }

  if($element_state == 2 && is_array($field['widget']['imagecache_presets']))
  {
    foreach(aef_utilities_check_cck_settings_array($field['widget']['imagecache_presets']) as $preset_name)
    {
      $preset = imagecache_preset_by_name($preset_name);
      if(empty($preset) == false)
      {
        if(isset($element['#post'][$element['#field_name'] .'_'. $element['#delta'] . '_' . $preset_name . '_edit']))
        {
          $element_state = 3;
          $preset_being_edited = $preset_name;
        }
      }
    }
  }



  // Set up the buttons first since we need to check if they were clicked.
  $element['aef_image_upload'] = array(
    '#name' => $element['#field_name'] .'_'. $element['#delta'] .'_upload',
    '#type' => 'submit',
    '#value' => t('Upload'),
    '#ahah' => array(
        "path" => "aef_image/ahah/" . $element['#type_name'] . '/' . $element['#field_name'] . '/' . $delta,
        "effect" => "fade",
        "wrapper" => $element['#id'] . '-wrapper', 
    ),
    '#field_name' => $element['#field_name'],
    '#post' => $element['#post'],
    '#attributes' => array('class' => 'aef_image_upload'),
  );
  $element['aef_image_remove'] = array(
    '#name' => $element['#field_name'] .'_'. $element['#delta'] .'_remove',
    '#type' => 'submit',
    '#value' => t('Remove'),
    '#ahah' => array(
        "path" => "aef_image/ahah/" . $element['#type_name'] . '/' . $element['#field_name'] . '/' . $delta,
        "effect" => "fade",
        "wrapper" => $element['#id'] . '-wrapper', 
    ),
    '#field_name' => $element['#field_name'],
    '#post' => $element['#post'],
    '#attributes' => array('class' => 'aef_image_remove')
  );


/**
  // Because the output of this field changes depending on the button clicked,
  // we need to ask FAPI immediately if the remove button was clicked.
  // It's not good that we call this private function, but
  // $form_state['clicked_button'] is only available after this #process
  // callback is finished.
  if (_form_button_was_clicked($element['viid_viewer']['remove'])) {
  }
*/


  //Put a div wrapper here
  $element['#prefix'] = '<div class="form-item">' .
   (($element['#title'])?'<label>' . check_plain($element['#title']) . ':</label>':"") . 
   '<div id="' . $element['#id'] . '-wrapper" class="aef_image_wrapper" field_name="' . $element['#field_name'] . '" delta="' . $element['#delta'] . '">';
  $element['#suffix'] = '</div></div>';

  //Theme
  $element['#theme'] = 'aef_image_widget';

  //
  // At this points, state info is ready.
  //


  $element['upload'] = array(
    '#name' => 'files['. $element['#field_name'] .'_'. $element['#delta'] .']',
    '#type' => 'file',
    '#required' => $element['#required'],
    '#weight' => -1,
    '#description' => $element["#description"],
  );
  $element['preview'] = array(
    '#prefix' => '<div class="aef-image-preview">',
    '#value' => (($field['widget']['image_noderef'] == 0) ?
      theme('imagecache', 'aef_image_thumbnail', $infos['image_infos']['filepath']) :
      theme('imagecache', 'aef_image_thumbnail', $infos['data']['defaults_from_node']['image_infos']['filepath'])),
    '#weight' => -1,
    '#suffix' => '</div>',
  );

  //For the noderef widget, add the "Create new" link if aef_embedded_edit is present, if the noderef is empty
  // and if a content type was selected in the settings  
  if(module_exists('aef_embedded_edit') && 
    $item['nid'] == 0 &&
    $field['widget']['image_noderef'] == 1 && $field['widget']['create_new_nodetype'] != '' && 
    $field['widget']['create_new_nodetype'] != 'none')
  {
    $link_attrs = array('class' => 'create_new_button', 'field' => $element['#field_name']);

    //Get the field multigroupgroup name, if any
    if(substr($element['#parents'][0], 0, 6) == "group_")
      $link_attrs['multigroup'] = $element['#parents'][0];

    $content_types = content_types();
    $create_new_button = l(t('Create new @node_type', array('@node_type' => $content_types[$field['widget']['create_new_nodetype']]['name'])), 'aef_embedded_edit/node/add/' . $content_types[$field['widget']['create_new_nodetype']]['url_str'],
    array('attributes' => $link_attrs));
  }

  //The nodereference widget
  $element['noderef'] = array(
    '#prefix' => '<div class="container-inline" id="' . $element['#field_name'] . '_values">',
    '#suffix' => '</div><div>' . $create_new_button . '</div>',
  );
  $noderef_node = node_load($item['nid']);
  $element['noderef']['noderef'] = array(
    '#type' => 'textfield',
    '#value' => (($noderef_node->nid > 0)?check_plain($noderef_node->title) . ' [nid:' . $noderef_node->nid . ']':''),
    '#autocomplete_path' => 'aef_image/noderef_autocomplete',
    '#ahah' => array(
        "path" => "aef_image/ahah/" . $element['#type_name'] . '/' . $element['#field_name'] . '/' . $delta,
        "effect" => "fade",
        "wrapper" => $element['#id'] . '-wrapper', 
        "event" => "change",
    ),
    '#attributes' => array("class" => 'aef-nodeselect-droppable'),
    '#required' => $element['#required'],
  );

  $image_infos_display = $field['widget']['image_infos_display'];
  $element['infos']['title'] = array(
    '#title' => t('Image title'),
    '#type' => ((isset($image_infos_display['title']) == false || $image_infos_display['title'])?'textfield':'hidden'),
    '#value' => (isset($infos['data']['defaults_from_node']) && trim($item['title']) == "<default>")?
      $infos['data']['defaults_from_node']['title']:$item['title'],
    '#size' => 40,
    '#maxlength' => 128,
  );

  $element['infos']['legend'] = array(
    '#title' => t('Image legend'),
    '#type' => ((isset($image_infos_display['legend']) == false || $image_infos_display['legend'])?'textfield':'hidden'),
    '#value' => (isset($infos['data']['defaults_from_node']) && trim($item['legend']) == "<default>")?
      $infos['data']['defaults_from_node']['legend']:$item['legend'],
    '#size' => 40,
    '#maxlength' => 256,
  );

  $element['infos']['credits'] = array(
    '#title' => t('Image credits'),
    '#type' => ((isset($image_infos_display['credits']) == false || $image_infos_display['credits'])?'textfield':'hidden'),
    '#value' => (isset($infos['data']['defaults_from_node']) && trim($item['credits']) == "<default>")?
      $infos['data']['defaults_from_node']['credits']:$item['credits'],
    '#maxlength' => 64,
  );

  $element['nid'] = array(
    '#type' => 'hidden',
    '#value' => $item['nid'],
    );
  $element['fid'] = array(
    '#type' => 'hidden',
    '#value' => $item['fid'],
    );


  $element['presets_area'] = array();
  if($element_state != 1 && is_array($field['widget']['imagecache_presets']))
  {
    //Put the theming of the preset area
    $element['presets_area']['#type'] = 'fieldset';
    $element['presets_area']['#title'] = t('Image formats');
    $element['presets_area']['#collapsible'] = 'true';
    $element['presets_area']['#collapsed'] = ($element_state != 3 && $old_element_state != 3);
    $element['presets_area']['#theme'] = 'aef_image_widget_presets';

    //If we are grouping presets of the same size, do it here.
    $presets = array();
    if($field['widget']['group_imagecache_presets'] == 1)
    {
      $group_infos = _aef_image_group_presets(aef_utilities_check_cck_settings_array($field['widget']['imagecache_presets']));
      foreach($group_infos['names'] as $preset_name => $preset_group)
      {
        $presets[$preset_name] = array('group' => $preset_group);
      }
    }
    else
    {
      //No grouping
      foreach(aef_utilities_check_cck_settings_array($field['widget']['imagecache_presets']) as $preset_name)
      {
        $presets[$preset_name] = array();
      }
    }

    //Output each preset preview and buttons
    foreach($presets as $preset_name => $preset_infos)
    {
      $preset = imagecache_preset_by_name($preset_name);
      if(empty($preset) == false)
      {
        $preset_selector_id = $element['#field_name'] .'_'. $element['#delta'] . '_presetselector_' . $preset_name;

        $element['presets_area'][$preset_name] = array(
          '#prefix' => '<div class="aef-image-preset">',
          '#suffix' => '</div>',
        );

        //Preset infos.
        //If this preset is being edited, show part of the full image that will mirror the selection
        //Otherwise show the result of the preset.
        $imagefid = _aef_image_get_preset_fid($item['fid'], $infos['data'], $preset_name);
        $image = aef_image_fid_load($imagefid);
        $image_path = $image['filepath'];
        $image_infos = image_get_info($image_path);
        if($element_state == 3 && $preset_being_edited == $preset_name)
        {
          $image_preview = theme('image', $image_path, '', '', 
            array('id' => $preset_selector_id . '_preview',
              //TODO: Works because we only support one kind of preset now
              'preset_width' => $preset['actions'][0]['data']['width'],
              'preset_height' => $preset['actions'][0]['data']['height'],
              'img_width' => $image_infos['width'],
              'img_height' => $image_infos['height'],
            ));
        }
        else
        {
          $image_preview = theme('aef_image_imageonly', $item['nid'], $item['fid'], $item['data'], $preset_name, '', '');
        }
        $preset_title = (($preset['title'])?$preset['title']:$preset_name);
        if(is_array($preset_infos['group']))
        {
          $titles = array();
          foreach($preset_infos['group'] as $preset_title_name)
          {
            $preset_title_object = imagecache_preset_by_name($preset_title_name);
            $titles[] = (($preset_title_object['title'])?$preset_title_object['title']:$preset_title_name);
          }
          $preset_title = implode('<br />', $titles);
        }

        $element['presets_area'][$preset_name]['infos'] = array(
          '#value' => 
            '<div class="aef-image-preset-title">' .
            $preset_title .
            '</div>' .
            '<div class="aef-image-preset-preview" style="width:' . $preset['actions'][0]['data']['width'] . 'px; height:' . $preset['actions'][0]['data']['height'] . 'px">' .
            $image_preview . 
            '</div>'
        );

        $element['presets_area'][$preset_name]['actions']['edit'] = array(
        '#name' => $element['#field_name'] .'_'. $element['#delta'] . '_' . $preset_name . '_edit',
        '#type' => 'submit',
        '#value' => t('Edit'),
        '#ahah' => array(
          "path" => "aef_image/ahah/" . $element['#type_name'] . '/' . $element['#field_name'] . '/' . $delta,
          "effect" => "fade",
          "wrapper" => $element['#id'] . '-wrapper', 
          ),
        '#access' => ($element_state != 3),
        );

        if($field['widget']['preset_selection'] == 1)
        {
          $parents = $element['#parents'];
          $element['presets_area'][$preset_name]['actions']['preset_selection'] = array(
            '#title' => t('Use this format'),
            '#type' => (($element_state == 3)?'hidden':'radio'),
            '#name' => array_shift($parents) . '[' .  implode('][', $parents) . '][preset_selection]',
            '#value' => (($item['data']['preset_selection'] != "")?$item['data']['preset_selection']:$field['widget']['default_imagecache_preset']),
            '#return_value' => $preset_name,
            '#attributes' => array('class' => 'aef_image_preset_selection'),
          );
        }

        $element['presets_area'][$preset_name]['area_selector'] = array(
          '#title' => t('Region selection'),
          '#type' => 'fieldset', 
          '#collapsible' => false,
          '#collapsed' => false,
          '#access' => ($element_state == 3 && $preset_being_edited == $preset_name),
        );

        $element['presets_area'][$preset_name]['area_selector']['image'] = array(
          '#value' => theme('image', $image_path, '', '', 
            array('class'=>'aef-image-crop-image', 
            'id' => $preset_selector_id)
          ),
        );

        $element['presets_area'][$preset_name]['area_selector']['upload'] = array(
          '#prefix' => '<div class="container-inline">',
          '#suffix' => '</div>',
          '#access' => ($infos['data']['presets'][$preset_name]['fid'] == 0),
        );
        $element['presets_area'][$preset_name]['area_selector']['upload']['file'] = array(
          '#name' => 'files['. $element['#field_name'] .'_'. $element['#delta'] . '_presets_' . $preset_name . ']',
          '#type' => 'file',
          '#size' => '20',
        );
        $element['presets_area'][$preset_name]['area_selector']['upload']['button'] = array(
        '#name' => $element['#field_name'] .'_'. $element['#delta'] . '_' . $preset_name . '_upload',
        '#type' => 'submit',
        '#value' => t('Upload new image'),
        '#ahah' => array(
          "path" => "aef_image/ahah/" . $element['#type_name'] . '/' . $element['#field_name'] . '/' . $delta,
          "effect" => "fade",
          "wrapper" => $element['#id'] . '-wrapper', 
          ),
        );

        $element['presets_area'][$preset_name]['actions']['save'] = array(
          '#name' => $element['#field_name'] .'_'. $element['#delta'] . '_' . $preset_name . '_save',
          '#type' => 'submit',
          '#value' => t('Save'),
          '#ahah' => array(
            "path" => "aef_image/ahah/" . $element['#type_name'] . '/' . $element['#field_name'] . '/' . $delta,
            "effect" => "fade",
            "wrapper" => $element['#id'] . '-wrapper', 
            ),
          '#access' => ($element_state == 3 && $preset_being_edited == $preset_name),
        );

        $element['presets_area'][$preset_name]['area_selector']['remove_custom_img'] = array(
          '#name' => $element['#field_name'] .'_'. $element['#delta'] . '_' . $preset_name . '_remove_custom_img',
          '#type' => 'submit',
          '#value' => t('Remove custom image'),
          '#ahah' => array(
            "path" => "aef_image/ahah/" . $element['#type_name'] . '/' . $element['#field_name'] . '/' . $delta,
            "effect" => "fade",
            "wrapper" => $element['#id'] . '-wrapper', 
            ),
          '#access' => ($infos['data']['presets'][$preset_name]['fid'] > 0),
        );

        //Get the coords
        $coords = _aef_image_get_preset_coords($infos['data'], $preset_name, true);
        $element['presets_area'][$preset_name]['x'] = array(
          '#type' => 'hidden',
          '#value' => $coords['x'],
          '#attributes' => array('class' => $preset_selector_id . '_x'),
        );
        $element['presets_area'][$preset_name]['y'] = array(
          '#type' => 'hidden',
          '#value' =>  $coords['y'],
          '#attributes' => array('class' => $preset_selector_id . '_y'),
        );
        $element['presets_area'][$preset_name]['x2'] = array(
          '#type' => 'hidden',
          '#value' => $coords['width'] +$coords['x'],
          '#attributes' => array('class' => $preset_selector_id . '_x2'),
        );
        $element['presets_area'][$preset_name]['y2'] = array(
          '#type' => 'hidden',
          '#value' => $coords['height'] + $coords['y'],
          '#attributes' => array('class' => $preset_selector_id . '_y2'),
        );
        $element['presets_area'][$preset_name]['fid'] = array(
          '#type' => 'hidden',
          '#value' => $infos['data']['presets'][$preset_name]['fid'],
          '#attributes' => array('class' => $preset_selector_id . '_fid'),
        );

        //Set the settings for the awesome JCrop JQuery plugin.
        //Compute first the minSize: if the image is smaller than the formatter, make the minSize smaller
        $min_size = array();
        if($infos['data']['presets'][$preset_name]['preset_infos']['width'] > $image_infos['width'] || 
          $infos['data']['presets'][$preset_name]['preset_infos']['height'] > $image_infos['height'])
        {
          $min_size = array($infos['data']['presets'][$preset_name]['default_values']['width'], $infos['data']['presets'][$preset_name]['default_values']['height']);
        }
        else
        {
          $min_size = array($infos['data']['presets'][$preset_name]['preset_infos']['width'], $infos['data']['presets'][$preset_name]['preset_infos']['height']);
        }
        //Compute the selected area
        $select_area =  array(
          $coords['x'], 
          $coords['y'], 
          $coords['x'] + $coords['width'], 
          $coords['y'] + $coords['height'],
        );
	      drupal_add_js(array(
		      "jcrop" => array(
			      $preset_selector_id => array(
  			      'args' => array(
                'setSelect' => $select_area,
                //The aspect ratio of the preset
                'aspectRatio' => $infos['data']['presets'][$preset_name]['preset_infos']['width']/$infos['data']['presets'][$preset_name]['preset_infos']['height'],
                //The minimum size is the size of the preset.
                'minSize' => $min_size,
                //onChange and onSelect, update the coordinates in our hidden inputs
                'onChange' => 'aef_image_update_coords',
                'onSelect' => 'aef_image_update_coords',
                'boxWidth' => 600,
              ),
            ),
			    ),
	      ), "setting");
      }
    }

    //If the image selection setting is on, add another preset: original format (but downsized if necessary)
    if($field['widget']['preset_selection'] == 1)
    {
      $preset_name = 'aef_image_original_format';
      $preset = imagecache_preset_by_name($preset_name);

      $element['presets_area'][$preset_name] = array(
        '#prefix' => '<div class="aef-image-preset">',
        '#suffix' => '</div>',
      );

      //Preset infos.
      //If this preset is being edited, show part of the full image that will mirror the selection
      //Otherwise show the result of the preset.
      $imagefid = _aef_image_get_preset_fid($item['fid'], $infos['data'], $preset_name);
      $image = aef_image_fid_load($imagefid);
      $image_path = $image['filepath'];
      $image_infos = image_get_info($image_path);

      $image_preview = theme('aef_image_imageonly', $item['nid'], $item['fid'], $item['data'], $preset_name, '', '');

      $preset_title = (($preset['title'])?$preset['title']:$preset_name);

      $element['presets_area'][$preset_name]['infos'] = array(
        '#value' => 
          '<div class="aef-image-preset-title">' .
          $preset_title .
          '</div>' .
          '<div class="aef-image-preset-preview">' .
          $image_preview . 
          '</div>'
      );

      $parents = $element['#parents'];
      $element['presets_area'][$preset_name]['actions']['preset_selection'] = array(
        '#title' => t('Use this format'),
        '#type' => 'radio',
        '#name' => array_shift($parents) . '[' .  implode('][', $parents) . '][preset_selection]',
        '#value' => (($item['data']['preset_selection'] != "")?$item['data']['preset_selection']:$field['widget']['default_imagecache_preset']),
        '#return_value' => $preset_name,
        '#attributes' => array('class' => 'aef_image_preset_selection'),
      );
    }

  }



  //Save the element state, can be useful to have the old element state
  $element['old_element_state'] = array(
    '#name' => $element['#field_name'] .'_'. $element['#delta'] . '_old_element_state',
    '#type' => 'hidden',
    '#value' => $element_state,
  );


  $element['upload']['#access'] = ($element_state == 1 && $field['widget']['image_noderef'] == 0);
  $element['aef_image_upload']['#access'] = ($element_state == 1 && $field['widget']['image_noderef'] == 0);
  $element['aef_image_remove']['#access'] = ($element_state != 1 && $field['widget']['image_noderef'] == 0);
  $element['noderef']['#access'] = ($field['widget']['image_noderef'] == 1);
  $element['preview']['#access'] = ($element_state != 1);
  $element['infos']['#access'] = ($element_state != 1);
  $element['presets_area']['#access'] = ($element_state != 1);


  return $element;
}


/**
 * FAPI theme for an individual text elements.
 *
 * $element['#field_name'] contains the field name
 * $element['#delta]  is the position of this element in the group
 */
function theme_aef_image_creator($element) {
  return $element['#children'];
}


/**
 * The #value_callback for the aef_image_creator type element.
 */
function aef_image_widget_value($element, $edit = FALSE) {
  global $user;
  $upload_name = $element['#field_name'] .'_'. $element['#delta'];
  $upload_name_presets = $element['#field_name'] .'_'. $element['#delta'] . '_presets';
  if(count($_FILES['files']['name']) > 0)
  {
    $upload_first_key = array_slice(array_keys($_FILES['files']['name']),0,1);
    $upload_first_key = $upload_first_key[0];
  }
  $field = content_fields($element['#field_name'], $element['#type_name']);

  //Get the nid from the noderef
  $new_node_loaded = false;
  preg_match('/^(?:\s*|(.*) )?\[\s*nid\s*:\s*(\d+)\s*\]$/', $edit['noderef']['noderef'], $matches);
  list(, $noderef_title, $noderef_nid) = $matches;
  if($noderef_nid > 0)
    $noderef_node = node_load($noderef_nid);




  if(is_array($edit))
    $item = array('fid' => $edit['fid'], 'nid' => $edit['nid']);



  if($edit && 
      (
        empty($_FILES['files']['name'][$upload_name]) == false
      )
    )
  {

    $dest = file_directory_path() . "/" . $field['widget']['file_path'];

    if (!file_check_directory($dest, FILE_CREATE_DIRECTORY)) 
    {
      watchdog('aef_image', 'The upload directory %directory for the file field %field (content type %type) could not be created or is not accessible. A newly uploaded file could not be saved in this directory as a consequence, and the upload was canceled.', array('%directory' => $dest, '%field' => $element['#field_name'], '%type' => $element['#type_name']));
      form_set_error($upload_name, t('The file could not be uploaded.'));
    }
    else
    {
      if (!$file = file_save_upload($upload_name, array(), $dest)) {
        watchdog('aef_image', 'The file upload failed. %upload', array('%upload' => $upload_name));
        form_set_error($upload_name, t('The file in the @field field was unable to be uploaded.', array('@field' => $element['#title'])));
      }
      else
      {
        //Ok, file uploaded!
        //Change the status of the upload to permanent
        $file->status = FILE_STATUS_PERMANENT;
        drupal_write_record('files', $file, 'fid');
        //Save the fid
        $item['fid'] = $file->fid;
      }
    }
    
  }
  else if($edit && 
      (
        empty($_FILES['files']['name']) == false && 
        substr($upload_first_key, 0, strlen($upload_name_presets)) == $upload_name_presets
      )
    )
  {
    //A new image for a preset was uploaded.
    $preset_name = substr($upload_first_key, strlen($upload_name_presets)+1);
    $dest = file_directory_path() . "/" . $field['widget']['file_path'];

    if (!file_check_directory($dest, FILE_CREATE_DIRECTORY)) 
    {
      watchdog('aef_image', 'The upload directory %directory for the file field %field (content type %type) could not be created or is not accessible. A newly uploaded file could not be saved in this directory as a consequence, and the upload was canceled.', array('%directory' => $dest, '%field' => $element['#field_name'], '%type' => $element['#type_name']));
      form_set_error($upload_name, t('The file could not be uploaded.'));
    }
    else
    {
      if (!$file = file_save_upload($upload_first_key, array(), $dest)) {
        watchdog('aef_image', 'The file upload failed. %upload', array('%upload' => $upload_first_key));
        form_set_error($upload_name, t('The file in the @field field was unable to be uploaded.', array('@field' => $element['#title'])));
      }
      else
      {
        //Ok, file uploaded!
        //Change the status of the upload to permanent
        $file->status = FILE_STATUS_PERMANENT;
        drupal_write_record('files', $file, 'fid');
        //We unset the coordinates so that the default one are used.
        unset($edit['presets_area'][$preset_name]['x']);
        unset($edit['presets_area'][$preset_name]['y']);
        unset($edit['presets_area'][$preset_name]['x2']);
        unset($edit['presets_area'][$preset_name]['y2']);
        //Save the fid
        $item['data']['presets'][$preset_name]['fid'] = $file->fid;
      }
    }

  }
  //"Remove" button clicked
  else if($edit && 
      (
        isset($element['#post'][$element['#field_name'] .'_'. $element['#delta'] .'_remove']) 
      )
    )
  {

      $item = array('fid' => 0);
    
  }
  //New node reference entered
  else if($edit && 
      (
        is_numeric($noderef_nid) && $noderef_nid > 0 && $item['nid'] == 0 && 
        $noderef_nid = _aef_image_is_node_valid($noderef_node)
      )
    )
  {
    $item = array('nid' => $noderef_nid);
    $new_node_loaded = true;
  }
  //Node reference changed
  else if($edit && 
      (
        is_numeric($noderef_nid) && $noderef_nid > 0 && $item['nid'] != $noderef_nid &&
        $noderef_nid = _aef_image_is_node_valid($noderef_node)
      )
    )
  {
    $item = array('nid' => $noderef_nid);
    $new_node_loaded = true;
    unset($edit['presets_area']);
    unset($edit['infos']);
  }
  //Node reference removed
  else if($edit && 
      (
        $noderef_nid == 0 && $item['nid'] != 0
      )
    )
  {
    $item = array('nid' => 0);
  }
  

 
  if($edit && ($item['fid'] > 0 || $item['nid'] > 0)){

    //First update the custom image status
    foreach(aef_utilities_check_cck_settings_array($field['widget']['imagecache_presets']) as $preset_name)
    {
      $preset = imagecache_preset_by_name($preset_name);
      if(empty($preset) == false)
      {
        //If there is a custom image, and if it is not being removed, forward its fid
        if($edit['presets_area'][$preset_name]['fid'] > 0)
        {
          if(isset($element['#post'][$element['#field_name'] .'_'. $element['#delta'] . '_' . $preset_name . '_remove_custom_img']))
          {
            //The custom image is being removed, remove the custom region coordinates
            unset($edit['presets_area'][$preset_name]['x']);
            unset($edit['presets_area'][$preset_name]['y']);
            unset($edit['presets_area'][$preset_name]['x2']);
            unset($edit['presets_area'][$preset_name]['y2']);
          }
          else
          {
            $item['data']['presets'][$preset_name]['fid'] = $edit['presets_area'][$preset_name]['fid'];
          }
        }
      }
    }


    //Then save the coordinates
    $field_infos = _aef_image_creator_fetch_data($item, aef_utilities_check_cck_settings_array($field['widget']['imagecache_presets']));
    foreach(aef_utilities_check_cck_settings_array($field['widget']['imagecache_presets']) as $preset_name)
    {
      $preset = imagecache_preset_by_name($preset_name);
      if(empty($preset) == false)
      {
        //The new incoming size
        $new_size = array(
          "x" => $edit['presets_area'][$preset_name]['x'],
          "y" => $edit['presets_area'][$preset_name]['y'],
          "width" => $edit['presets_area'][$preset_name]['x2'] - $edit['presets_area'][$preset_name]['x'],
          "height" => $edit['presets_area'][$preset_name]['y2'] - $edit['presets_area'][$preset_name]['y'],
        );
        //The default size
        $default_values = $field_infos['data']['presets'][$preset_name]['default_values'];
        if(isset($field_infos['data']['defaults_from_node']))
          $default_values = _aef_image_get_preset_coords($field_infos['data']['defaults_from_node']['data'], $preset_name, true);

        //If the input is different from the default values, save them
        if(isset($edit['presets_area'][$preset_name]['x']) && 
          _aef_image_creator_cmp_coords($new_size, $default_values) == false)
        {
          $item['data']['presets'][$preset_name] = 
            ((is_array($item['data']['presets'][$preset_name]))?$item['data']['presets'][$preset_name]:array()) +
            $new_size;
        }
      }
    }

    //Save the title/legend/credits if necessary
    if(!isset($field_infos['data']['defaults_from_node']) || 
      ( 
        isset($field_infos['data']['defaults_from_node']) &&
        $edit['infos']['title'] != $field_infos['data']['defaults_from_node']['title'] &&
        $new_node_loaded == false
      ))
      $item['title'] = $edit['infos']['title'];
    else
      $item['title'] = '<default>';

    if(!isset($field_infos['data']['defaults_from_node']) || 
      ( 
        isset($field_infos['data']['defaults_from_node']) &&
        $edit['infos']['legend'] != $field_infos['data']['defaults_from_node']['legend'] &&
        $new_node_loaded == false
      ))
      $item['legend'] = $edit['infos']['legend'];
    else
      $item['legend'] = '<default>';

    if(!isset($field_infos['data']['defaults_from_node']) || 
      ( 
        isset($field_infos['data']['defaults_from_node']) &&
        $edit['infos']['credits'] != $field_infos['data']['defaults_from_node']['credits'] &&
        $new_node_loaded == false
      ))
      $item['credits'] = $edit['infos']['credits'];
    else
      $item['credits'] = '<default>';

  }

  //If we are grouping the presets of the same size, ungroup here
  if($field['widget']['group_imagecache_presets'] == 1)
  {
    $group_infos = _aef_image_group_presets(aef_utilities_check_cck_settings_array($field['widget']['imagecache_presets']));
    foreach($group_infos['names'] as $preset_name => $preset_group)
    {
      if(isset($item['data']['presets'][$preset_name]))
      {
        foreach($preset_group as $identical_preset_name)
        {
          $item['data']['presets'][$identical_preset_name] = $item['data']['presets'][$preset_name];
        }
      }
    }
  }

  //Save the preset selection
  if($field['widget']['preset_selection'] == 1 && $edit['preset_selection'])
    $item['data']['preset_selection'] = $edit['preset_selection'];


  if(!$edit){
    $item = $element['#default_value'];
  }

  _aef_image_unserialize_data($item['data']);

  return $item;
}

/**
 * AEF Easy view creator validation function.
 */
function aef_image_creator_validate(&$element, &$form_state) 
{

}

/**
 * Theme the widget
 */
function theme_aef_image_widget($element) {
  $html = "";

  $no_image_infos_field = (($element['infos']['title']['#type'] == "hidden") && 
    ($element['infos']['legend']['#type'] == "hidden") && 
    ($element['infos']['credits']['#type'] == "hidden"));

  $inside .= '<div class="aef-image-informations">';

  if($no_image_infos_field)
  {
    //No image fields: display all (as hidden), display the preview
    $title_render .= drupal_render($element['infos']['title']);
    $inside .= $title_render;
    $inside .= drupal_render($element['infos']['credits']);
    $inside .= drupal_render($element['preview']);
    $inside .= drupal_render($element['infos']['legend']);
  }
  else
  {
    //There are fields: put them in a table
    $inside .= '<table class="aef-image-informations-infos">';
    $inside .= '<tr><td>';
    $title_render .= drupal_render($element['infos']['title']);
    $inside .= $title_render;
    $inside .= '</td><td class="aef-image-informations-credits">';
    $inside .= drupal_render($element['infos']['credits']);
    $inside .= '</td><td rowspan="2" class="aef-image-preview-area">';
    $inside .= drupal_render($element['preview']);
    $inside .= '</td></tr>';
    $inside .= '<tr><td colspan="2" class="aef-image-infos-legend">';
    $inside .= drupal_render($element['infos']['legend']);
    $inside .= '</td></tr>';
    $inside .= '</table>';
  }
  $inside .= drupal_render($element['presets_area']);
  $inside .= drupal_render($element['aef_image_remove']);
  $inside .= '</div>';

  $html .= drupal_render($element);
  //Put the inside (title,legend,source, presets) if it is not empty
  if($title_render)
    $html .= $inside;

  return $html;
}

/**
 * Theme a list of presets
 */
function theme_aef_image_widget_presets($element) {
  $id = $element['#id'] . '-presets';

  $items = array();
  foreach (element_children($element) as $key) {
    $items[] = &$element[$key];
  }

  $header = array(
    t('Formats'),
    t('Actions'),
  );
  $rows = array();



  // Add the items as table rows.
  foreach ($items as $key => $item) {
    $fixed_element = drupal_render($item['actions']);
    $cells = array(
      drupal_render($item),
      array('data' => $fixed_element, 'class' => 'aef_image_fixed_checkbox'),
    );
    $rows[] = array(
      'data' => $cells,
    );
  }

  $output .= theme('table', $header, $rows, array('id' => $id, 'class' => 'aef-image-widget-presets'));

  return $output;
}

/**
 * Menu callback; Retrieve a pipe delimited string of autocomplete suggestions for existing users
 */
function aef_image_noderef_autocomplete($string = '') {
  $where = array();
  $args = array();
  $limit = 10;

  if ($string !== '') {
    $where[] = 'n.title '. "LIKE '%%%s%%'";
    $args[] = $string;
  }

  $where_clause = $where ? 'WHERE ('. implode(') AND (', $where) .')' : '';
  $sql = db_rewrite_sql("SELECT n.nid, n.title AS node_title, n.type AS node_type FROM {node} n $where_clause ORDER BY n.title, n.type");
  $result = $limit ? db_query_range($sql, $args, 0, $limit) : db_query($sql, $args);
  $references = array();
  while ($node = db_fetch_object($result)) {
    $references[$node->nid] = array(
      'title' => $node->node_title,
      'rendered' => check_plain($node->node_title),
    );
  }

  $matches = array();
  foreach ($references as $id => $row) {
    // Add a class wrapper for a few required CSS overrides.
    $matches[$row['title'] ." [nid:$id]"] = '<div class="reference-autocomplete">'. $row['rendered'] . '</div>';
  }
  drupal_json($matches);
}

/**
 * Is this node a valid container of image?
 * Return the nid of the valid container (A node could have a AEF image noderef to another AEF image noderef,
 * and so forth... Recursively find the real image node)
 */
function _aef_image_is_node_valid($node)
{
  $result = false;

  if($node != null)
  {
    $fields = content_types($node->type);
    foreach($fields['fields'] as $field_name => $field_infos)
    {
      if($field_infos['type'] == 'aef_image')
      {
        $item = $node->{$field_name}[0];
        _aef_image_unserialize_data($item['data']);
        if(aef_image_content_is_empty($item, null) == false)
        {
          if($item['fid'] > 0)
            $result = $node->nid;
          else //if($item['nid'])
            $result = _aef_image_is_node_valid(node_load($item['nid']));
        }
      }
    }
  }

  return $result;
}

/**
 * Unserialize the data part, in a while loop: I spotted a case of a double serialized data field.
 * Should be fixed by now.
 * This should really be the job of CCK... :s
 */
function _aef_image_unserialize_data(&$data)
{
  $max_count = 0;

  while(is_array($data) == false && $data != "" && $max_count < 5)
  {
    $data = unserialize($data);
    $max_count++;
  }
  if(is_array($data) == false && $data == "")
    $data = null;
}
