<?php

declare(strict_types=1);

namespace Drupal\aemet;

use Drupal\aemet\Clients\PredictionsSpecific;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use GuzzleHttp\ClientInterface;

/**
 * Perform requests through aemet.
 *
 * @see https://opendata.aemet.es/dist/index.html?#/
 */
final class Client {

  const BASE_URL = 'https://opendata.aemet.es/opendata';

  /**
   * Constructs a Client object.
   */
  public function __construct(
    private readonly ClientInterface $httpClient,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly CacheBackendInterface $cacheBackend,
    private readonly TimeInterface $time,
  ) {}

  /**
   * Returns API to make specific predictions.
   */
  public function predictionsSpecific() : PredictionsSpecific {
    return new PredictionsSpecific($this->httpClient, $this->cacheBackend, $this->time, $this->getApiKey(), (int) $this->getConfig()->get('requests_max_age'));
  }

  /**
   * Gets configuration.
   */
  protected function getConfig() : ImmutableConfig {
    return $this->configFactory->get('aemet.settings');
  }

  /**
   * Gets the API key from configuration.
   */
  public function getApiKey() : string {
    return $this->getConfig()->get('api_key');
  }

}
