<?php

declare(strict_types=1);

namespace Drupal\aemet\Plugin\Block;

use Drupal\aemet\Model\SkyStatus;
use Drupal\aemet\Model\PredictionSpecificHourly;
use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\aemet\Client;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a hourly prediction block.
 */
#[Block(
  id: 'aemet_weather_prediction_hourly',
  admin_label: new TranslatableMarkup('Aemet Weather: Hourly prediction'),
  category: new TranslatableMarkup('Aemet'),
)]
final class PredictionHourlyBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs the plugin instance.
   */
  public function __construct(
    array                                   $configuration,
                                            $plugin_id,
                                            $plugin_definition,
    private readonly Client                 $aemetClient,
    private readonly LoggerChannelInterface $loggerChannel,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('aemet.client'),
      $container->get('logger.channel.aemet'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'municipality_code' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state): array {
    $form['municipality_code'] = [
      '#type' => 'textfield',
      '#required' => TRUE,
      '#title' => $this->t('Municipality code'),
      '#default_value' => $this->configuration['municipality_code'],
      '#description' => $this->t('Municipality code is build joining the fields CPRO and CMUN from <a href="https://www.ine.es/daco/daco42/codmun/diccionario24.xlsx" target="_blank">municipality list</a>. Example for Madrid: 28079'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state): void {
    $this->configuration['municipality_code'] = $form_state->getValue('municipality_code');
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $build = [];
    $build['content'] = [];
    try {
      $date = new \DateTime();
      $prediction = $this->aemetClient
        ->predictionsSpecific()
        ->getMunicipalityPredictionHourly($this->configuration['municipality_code']);
      if ($prediction instanceof PredictionSpecificHourly) {
        $sky_status = $prediction->getSkyStatusByDate($date);
        $build['content'] = [
          '#type' => 'component',
          '#component' => 'aemet:weather-info',
          '#props' => [
            'minimum_temperature' => $prediction->getMinimumTemperature($date) ?? '',
            'maximum_temperature' => $prediction->getMaximumTemperature($date) ?? '',
            'temperature' => $prediction->getTemperatureByDate($date) ?? '',
            'humidity' => $prediction->getHumidityByDate($date) ?? '',
            'cloud_status_description' => $sky_status instanceof SkyStatus ? $sky_status->getDescription() : '',
            'cloud_status_value' => $sky_status instanceof SkyStatus ? $sky_status->getValue() : '',
          ],
        ];
      }
    }
    catch (\Exception $e) {
      $this->loggerChannel->error('Error getting weather information: ' . $e->getMessage());
    }
    return $build;
  }

}
