<?php

declare(strict_types=1);

namespace Drupal\aemet\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Aemet settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'aemet_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['aemet.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['api_key'] = [
      '#type' => 'textarea',
      '#title' => $this->t('API Key'),
      '#default_value' => $this->config('aemet.settings')->get('api_key'),
      '#required' => TRUE,
      '#description' => $this->t('Required API Key to do requests to Aemet. Get your own API key at <a target="_blank" href="https://opendata.aemet.es/centrodedescargas/inicio">Aemet OpenData</a>'),
    ];

    $form['requests_max_age'] = [
      '#type' => 'select',
      '#title' => $this->t('Requests max age'),
      '#default_value' => $this->config('aemet.settings')->get('requests_max_age'),
      '#required' => TRUE,
      '#options' => [
        3600 => $this->t('One hour'),
        43200 => $this->t('Half day'),
        86400 => $this->t('One day'),
        -1 => $this->t('Disabled'),
      ],
      '#description' => $this->t('Max age of API requests.'),
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('aemet.settings')
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('requests_max_age', $form_state->getValue('requests_max_age'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
