<?php

namespace Drupal\aemet\Clients;

use Drupal\aemet\Client;
use Drupal\aemet\Model\AemetModelInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use GuzzleHttp\ClientInterface;
use Psr\Http\Message\ResponseInterface;

/**
 * Base class to have specific APIs.
 */
abstract class ClientBase {

  /**
   * Constructs the Base Client.
   *
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   Used to perform requests against Aemet.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   Cache backend.
   * @param \Drupal\Component\Datetime\TimeInterface $dateTime
   *   Date Time.
   * @param string $api_key
   *   API key.
   * @param int $max_age
   *   Max age.
   *
   * @parm string $api_key
   *   API key.
   */
  public function __construct(
    private readonly ClientInterface $httpClient,
    private readonly CacheBackendInterface $cacheBackend,
    private readonly TimeInterface $dateTime,
    private string $api_key,
    private int $max_age,
  ) {}

  /**
   * Do an API request.
   *
   * @param string $path
   *   Request path.
   * @param string $model_class
   *   Class that will wrap the results.
   */
  protected function doRequest(string $path, string $model_class) {
    $cid = sprintf('aemet.request.%s', md5($path . $model_class));
    $cache = $this->cacheBackend->get($cid);
    if (empty($cache->data)) {
      assert(class_exists($model_class) && is_subclass_of($model_class, AemetModelInterface::class));
      $response = $this->httpClient->request('GET', $this->buildUrl($path), ['query' => ['api_key' => $this->api_key]]);
      $data = $this->getDataFromResponse($response);
      $model_data = new $model_class($data);
      $this->cacheBackend->set($cid, $model_data, $this->dateTime->getCurrentTime() + $this->max_age);
      return $model_data;
    }
    return $cache->data;
  }

  /**
   * Get the data obtained from the API callback.
   *
   * For each endpoint, Aemet API returns a URL to a JSON file
   * that contains all the requested data.
   * This method is a helper to extract the information.
   *
   * @param \Psr\Http\Message\ResponseInterface $response
   *   API Response.
   *
   * @return array
   *   Json data.
   */
  protected function getDataFromResponse(ResponseInterface $response) {
    $json = $this->parseJsonResponse($response);
    if (empty($json['datos'])) {
      throw new \RuntimeException('Data from Aemet not found in response');
    }

    $data_response = $this->httpClient->request('GET', $json['datos']);
    return $this->parseJsonResponse($data_response);
  }

  /**
   * Converts a JSON response to array.
   *
   * @param \Psr\Http\Message\ResponseInterface $response
   *   Response.
   *
   * @return array
   *   JSON response converted to array.
   */
  protected function parseJsonResponse(ResponseInterface $response) {
    // Data coming from opendata needs to be converted because
    // otherwise there may be malformed characters,
    // which would block json decoding.
    $contents = mb_convert_encoding($response->getBody()->getContents(), 'UTF-8', 'UTF-8');
    return json_decode($contents, TRUE);
  }

  /**
   * Builds URL for a specific endpoint.
   *
   * @param string $path
   *   Path.
   */
  protected function buildUrl(string $path) {
    return sprintf('%s%s%s', Client::BASE_URL, $this->getBasePath(), $path);
  }

  /**
   * Gets the base path of the endpoints.
   *
   * Every group of endpoints shares the same base path.
   *
   * @return string
   *   Base path.
   */
  abstract protected function getBasePath() : string;

}
