<?php

namespace Drupal\aemet\Model;

/**
 * Gets predictions for a specific day.
 */
class PredictionSpecificHourly extends Base {

  /**
   * Gets the maximum temperature of a specific day.
   *
   * @param \DateTime $date
   *   Date.
   *
   * @return int
   *   Celsius temperature
   */
  public function getMaximumTemperature(\DateTime $date) : ?int {
    $temperatures = $this->getAllTemperatures($this->getDayPrediction($date) ?? []);
    return !empty($temperatures) ? max($temperatures) : NULL;
  }

  /**
   * Gets the minimum temperature of a specific day.
   *
   * @param \DateTime $date
   *   Date.
   *
   * @return int
   *   Celsius temperature
   */
  public function getMinimumTemperature(\DateTime $date) : ?int {
    $temperatures = $this->getAllTemperatures($this->getDayPrediction($date) ?? []);
    return !empty($temperatures) ? min($temperatures) : NULL;
  }

  /**
   * Gets the prediction for a specific day.
   *
   * @param \DateTime $date
   *   Date.
   */
  protected function getDayPrediction(\DateTime $date) : ?array {
    $day_predictions = $this->data[0]['prediccion']['dia'];
    foreach ($day_predictions as $day_prediction) {
      $prediction_date = new \DateTime($day_prediction['fecha']);
      if ($prediction_date->format('Y-m-d') == $date->format('Y-m-d')) {
        return $day_prediction;
      }
    }
    return NULL;
  }

  /**
   * Gets all the temperatures of a specific day.
   *
   * Used to get maximum / minimum.
   *
   * @param array $day_prediction
   *   Day prediction.
   *
   * @return array
   *   List of temperatures in celsius.
   */
  protected function getAllTemperatures(array $day_prediction) : array {
    return array_map(function ($temperature) {
      return $temperature['value'];
    }, $day_prediction['temperatura'] ?? []);
  }

  /**
   * Get the temperature for a specific date and hour.
   *
   * @param \DateTime $date
   *   Full date containing date + hour.
   *
   * @return int
   *   Temperature.
   */
  public function getTemperatureByDate(\DateTime $date) : ?int {
    $temperature = $this->getPredictionDataByDateHour($date, 'temperatura')['value'] ?? NULL;
    return !empty($temperature) ? (int) $temperature : NULL;
  }

  /**
   * Get the humidity for a specific date and hour.
   *
   * @param \DateTime $date
   *   Full date containing date + hour.
   *
   * @return int
   *   Humidity percentage.
   */
  public function getHumidityByDate(\DateTime $date) : ?int {
    $humidity = $this->getPredictionDataByDateHour($date, 'humedadRelativa')['value'] ?? NULL;
    return !empty($humidity) ? (int) $humidity : NULL;
  }

  /**
   * Get the sky status for a specific date and hour.
   *
   * @param \DateTime $date
   *   Full date containing date + hour.
   *
   * @return SkyStatus
   *   Sky status..
   */
  public function getSkyStatusByDate(\DateTime $date) : ?SkyStatus {
    $sky_status = $this->getPredictionDataByDateHour($date, 'estadoCielo');
    return !empty($sky_status['descripcion']) && !empty($sky_status['value']) ? new SkyStatus($sky_status['descripcion'], $sky_status['value']) : NULL;
  }

  /**
   * Use this method to get a prediction for a specific hour and property.
   *
   * For example, getting the temperature / humidity.
   *
   * @param \DateTime $date
   *   Date.
   * @param string $property
   *   Property we want get its information.
   *
   * @return array
   *   Property value.
   */
  protected function getPredictionDataByDateHour(\DateTime $date, string $property) : ?array {
    $hour = $date->format('H');
    $day_prediction = $this->getDayPrediction($date);
    $data = $day_prediction[$property] ?? [];
    foreach ($data as $item) {
      if ($hour <= $item['periodo']) {
        return $item;
      }
    }
    return NULL;
  }

}
