<?php

/**
 * Trait for settings that need to be used by multiple classes.
 *
 * The admin screen keeps things simple and enables all clients to have
 * different settings; this stems from a time when a completely different
 * authentication scheme was introduced. (We used to authenticate using NTLM
 * rather than through 'App Connectors'.) However, it is extremely likely that
 * all clients use a base set of settings that are common for all clients.
 * Those are defined here.
 */
trait ClientCommonSettingsTrait {

  /**
   * Checks / modifies options. Meant to be called from constructor.
   *
   * This is a static function; it's not meant to touch $this->options because
   * that's the business of the parent class in the library.
   *
   * @param array $options
   *   The options as passed into the constructor. (Often, the  argument is
   *   empty and options will be fetched from Drupal variables.)
   *
   * @return array
   *   The options, possibly modified.
   *
   * @throws \InvalidArgumentException
   *   If some option values are missing / incorrect.
   * @throws \RuntimeException
   *   If the current user has no permission to use this client to access the
   *   configured endpoint.
   */
  public static function checkCommonSettings(array $options) {
    // If options are not provided (which in practice will be always), populate
    // them from our global Drupal configuration variables.
    foreach (array(
               'customerId' => array('afas_api_customer_id', ''),
               'appToken' => array('afas_api_app_token', ''),
             ) as $required_key => $var) {
      if (!isset($options[$required_key])) {
        $options[$required_key] = variable_get($var[0], $var[1]);
      }
      // We know the parent will throw an exception. We'll set a more specific
      // message.
      if (empty($options[$required_key])) {
        throw new InvalidArgumentException("Required configuration parameter missing for AFAS client: $required_key. Maybe you forgot to set the module configuration?", 1);
      }
    }

    // Check access to the environment.
    $environment = variable_get('afas_api_environment', '');
    if (in_array($environment, array('test', 'accept'), TRUE)) {
      if (!user_access('access afas test')) {
        throw new RuntimeException("The current user has no permission to access AFAS non-live environments through Drupal.");
      }
    }
    elseif ($environment === '') {
      if (!user_access('access afas')) {
        throw new RuntimeException("The current user has no permission to access the AFAS live environment through Drupal.");
      }
    }
    else {
      throw new InvalidArgumentException("Unknown environment '" . var_export($environment, TRUE) , "'.");
    }

    return $options;
  }

  /**
   * Adds class specific options to the settings form.
   *
   * @return array
   *   Extra form elements to merge into the settings form.
   */
  public static function settingsForm() {
    $form['afas_api_environment'] = array(
      '#type' => 'select',
      '#title' => t('Connect to environment'),
      '#options' => array(
        'test' => t('Test'),
        'accept' => t('Accept'),
        '' => t('Live'),
      ),
      '#default_value' => variable_get('afas_api_environment', ''),
    );
    $form['afas_api_customer_id'] = array(
      '#type' => 'textfield',
      '#title' => t('Customer / User ID'),
      '#description' => t('This is usually 5 digits.'),
      '#default_value' => variable_get('afas_api_customer_id', ''),
      '#size' => 20,
    );
    $form['afas_api_app_token'] = array(
      '#type' => 'textfield',
      '#title' => t('The token for the app connector'),
      '#description' => t('A note: if it expires, this module has no automatic way yet of creating a new one.'),
      '#default_value' => variable_get('afas_api_app_token', ''),
    );

    return $form;
  }

}
