<?php
/**
 * @file
 * Admin screens for AFAS API.
 */

use PracticalAfas\Connection;
use PracticalAfas\UpdateConnector\UpdateObject;

/**
 * Form definition for global settings.
 */
function afas_api_settings_form($form, &$form_state) {

  $form['logging'] = array(
    '#type' => 'fieldset',
    '#title' => t('Logging'),
    '#description' => t('These logging settings are followed by the standard functions in afas_api.module. (Custom code may choose to use or ignore these standard functions/settings.)'),
  );
  // These are various combinations of the 'watchdog' + 'watchdog_extra' options
  // for the logger class.
  $form['logging']['afas_api_log_watchdog'] = array(
    '#type' => 'select',
    '#title' => t('Log AFAS errors to watchdog'),
    '#options' => array(
      0 => t('No'),
      AFAS_LOG_ERRORS_WATCHDOG => t('Log errors'),
      AFAS_LOG_ERRORS_WATCHDOG+AFAS_LOG_ERROR_DETAILS_WATCHDOG => t('Log all messages including all context info + backtrace'),
    ),
    '#default_value' => variable_get('afas_api_log_watchdog', 1),
  );
  $form['logging']['afas_api_log_screen'] = array(
    '#type' => 'select',
    '#title' => t('Show AFAS errors on screen'),
    '#options' => array(
      0 => t('No'),
      AFAS_LOG_ERRORS_SCREEN => t('Show errors'),
      AFAS_LOG_ERRORS_SCREEN+AFAS_LOG_ERROR_DETAILS_SCREEN => t('Show all messages including all context info + backtrace'),
    ),
    '#default_value' => variable_get('afas_api_log_screen', 0),
  );

  $form['class'] = array(
    '#type' => 'fieldset',
    '#title' => t('Client class'),
    '#collapsible' => TRUE,
    '#collapsed' => TRUE,
  );
  // Get class name from submitted AJAX form on reload, so the right form gets
  // displayed.
  if (!empty($form_state['values']['afas_api_client_class'])) {
    $current_classname = $form_state['values']['afas_api_client_class'];
  }
  else {
    $current_classname = variable_get('afas_api_client_class', 'DrupalRestCurlClient');
  }
  $info = module_invoke_all('afas_api_client_info');
  $form['class']['afas_api_client_class'] = array(
    '#type' => 'select',
    '#title' => t('The name of the AFAS soap client class to use'),
    '#description' => t('This class is used by the standard functions in afas_api.module. (Custom code may choose to use or ignore these standard functions/settings.)'),
    '#options' => array_combine(array_keys($info), array_keys($info)),
    '#default_value' => $current_classname,
    '#ajax' => array(
      'callback' => 'afas_api_admin_class_options_callback',
      'wrapper' => 'class-specific',
      'method' => 'replace',
    ),
  );
  $form['class']['info'] = array(
    '#markup' => '<p>' . t('More information:') . '</p><p>',
    '#prefix' => '<div class="description">',
    '#suffix' => '</div>',
  );
  foreach ($info as $classname => $description) {
    $form['class']['info']['#markup'] .= '<li>' . $classname . ': '
                                         . $description . '</li>';
  }
  $form['class']['info']['#markup'] .= '</p>';

  $form['class_specific'] = array(
    '#type' => 'fieldset',
    '#prefix' => '<div id="class-specific">',
    '#suffix' => '</div>',
    '#title' => t('Client specific options'),
    '#collapsible' => FALSE,
    '#description' => t("Options for the selected client class. Changing an option for one class may affect other classes' options too."),
  );
  if (method_exists($current_classname, 'settingsForm')) {
    $form['class_specific'] += $current_classname::settingsForm();
  }
  //@TODO Debug AJAX stuff? We can only change the class selector once.
  //      The second time, the screen is not updated.

  $form['#validate'][] = 'afas_api_settings_form_validate';

  return system_settings_form($form);
}

/**
 * AJAX form callback for the settings form.
 */
function afas_api_admin_class_options_callback($form, $form_state) {
  $form['class_specific']['#prefix'] = '<div class="messages warning">' . t("Press '!button' to make the change of client class permanent.", array('!button' => t('Save configuration'))) . '</div>' . $form['class_specific']['#suffix'];
  return $form['class_specific'];
}

/**
 * Validate function for our global settings form.
 */
function afas_api_settings_form_validate($form, &$form_state) {
  if (!empty($form_state['values']['afas_api_client_class'])) {
    $current_classname = $form_state['values']['afas_api_client_class'];
  }
  else {
    $current_classname = variable_get('afas_api_client_class', 'DrupalRestCurlClient');
  }
  if (method_exists($current_classname, 'settingsFormValidate')) {
    $current_classname::settingsFormValidate($form, $form_state);
  }
}

/**
 * Return the client type of the currently configured default client.
 *
 * @return string
 *   'SOAP', 'REST', or '' on error
 */
function afas_api_test_client_type() {
  $current_classname = variable_get('afas_api_client_class', 'DrupalRestCurlClient');
  $type = '';
  try {
    $type = $current_classname::getClientType();
  }
  catch (Exception $e) {
  }
  return $type;
}

/**
 * Form definition for tests form.
 */
function afas_api_test_form($form, &$form_state) {
  $type_soap = afas_api_test_client_type() === 'SOAP';

  $form['help'] = array(
    '#markup' => t('On this form you can perform certain remote calls, which eases testing API access and accessing certain data. The options are slightly different depending on the type of selected client class (SOAP or REST).'),
    '#weight' => 0,
  );

  $form['getconnector'] = array(
    '#type' => 'fieldset',
    '#title' => t('Get data from a Get Connector'),
  );
  $form['getconnector']['getconnector_name'] = array(
    '#type' => 'textfield',
    '#title' => t('Connector name'),
  );
  $form['getconnector']['getconnector_skip'] = array(
    '#type' => 'textfield',
    '#title' => t('Skip # of items before reading data'),
  );
  $form['getconnector']['getconnector_take'] = array(
    '#type' => 'textfield',
    // Adjust title to the specific client's behavior.
    '#title' => t('Read/return # of items') . ' ' . ($type_soap ? t('(cannot be 0)') : t('(default: 100)')),
  );
  $form['getconnector']['getconnector_orderby'] = array(
    '#type' => 'textfield',
    '#title' => t('Order by fields'),
    '#description' => t('Syntax example: Field1,-Field2 for ascending and then descending ordering.'),
  );
  if (user_access('access afas eval')) {
    $form['getconnector']['getconnector_filter'] = [
      '#type' => 'textarea',
      '#rows' => 5,
      '#title' => t('Filter'),
      '#description' => "Fill one or more (comma separated) array definitions to apply filters. (This format is a slightly simplified version of the argument to Connection::getData().)<br>    
One example, to get only items that changed in the past week:<br><em>[<br>
'A-DATE-FIELD' => '" . date('Y-m-d\TH:i:s', time() - 7 * 86400) . "',<br>
'#op' => \\PracticalAfas\\Connection::OP_LARGER_OR_EQUAL,<br>
],</em>",
      '#default_value' => "[\n],",
    ];
  }
  $form['getconnector']['getconnector_get_metadata'] = array(
    '#type' => 'checkbox',
    '#title' => t('Include metadata'),
  );
  if (!$type_soap) {
    $form['getconnector']['getconnector_get_metadata']['#states']['enabled'] = array(
      ':input[name="getconnector_get_literal"]' => array('checked' => FALSE),
    );
  }
  else {
    // 'Metadata' only makes sense for XML output; if we're converting to an
    // array, the Metdadata == Schema information gets swallowed.
    $form['getconnector']['getconnector_get_metadata']['#states']['disabled'] = array(
      ':input[name="getconnector_get_literal"]' => array('checked' => FALSE),
      ':input[name="getconnector_output_pretty"]' => array('checked' => FALSE)
    );
    $form['getconnector']['getconnector_get_empty'] = array(
      '#type' => 'checkbox',
      '#title' => t('Include empty field values'),
    );
  }
  $form['getconnector']['getconnector_get_literal'] = array(
    '#type' => 'checkbox',
    '#title' => t('Output literal return value from remote service'),
    '#states' => array('enabled' => array(':input[name="getconnector_output_pretty"]' => array('checked' => FALSE))),
  );
  if ($type_soap) {
    $form['getconnector']['getconnector_output_pretty'] = array(
      '#type' => 'checkbox',
      '#title' => t('Output pretty printed XML.'),
    );
  }
  $form['getconnector']['getconnector_get'] = array(
    '#type' => 'submit',
    '#value' => t('Get Data'),
  );

  if (user_access('access afas eval')) {
    $form['updateconnector'] = [
      '#type' => 'fieldset',
      '#title' => t('Send data through an Update Connector'),
    ];
    $form['updateconnector']['updateconnector_name'] = [
      '#type' => 'textfield',
      '#title' => t('Connector name'),
    ];
    $form['updateconnector']['updateconnector_action'] = [
      '#type' => 'textfield',
      '#title' => t('Action to take'),
      '#description' => "'update' or 'insert'.",
      '#default_value' => 'update',
    ];
    $form['updateconnector']['updateconnector_data'] = [
      '#type' => 'textarea',
      '#rows' => 5,
      '#title' => t('Payload data'),
      '#description' => "Enter data to send to AFAS (JSON or XML, depending on connector type), or an array value to convert first, which must start/end with [].",
    ];
    $form['updateconnector']['updateconnector_send'] = [
      '#type' => 'submit',
      '#value' => t('Send Data'),
    ];
    $form['updateconnector']['updateconnector_convert'] = [
      '#type' => 'submit',
      '#value' => t('Convert without sending'),
    ];
  }

  if ($type_soap) {
    $form['get_schema'] = array(
      '#type' => 'fieldset',
      '#title' => t('Get XSD schema for Update Connector'),
    );
    $form['get_schema']['get_schema_function'] = array(
      '#type' => 'textfield',
      '#title' => 'Connector name',
    );
    $form['get_schema']['get_schema_submit'] = array(
      '#type' => 'submit',
      '#value' => t('Get XSD'),
    );
  }
  else {
    $form['get_schema'] = array(
      '#type' => 'fieldset',
      '#title' => t('Get meta info'),
    );
    $form['get_schema']['get_schema_get'] = array(
      '#type' => 'textfield',
      '#title' => 'Get Connector name',
    );
    $form['get_schema']['get_schema_update'] = array(
      '#type' => 'textfield',
      '#title' => 'Update Connector name',
    );
    $form['get_schema']['get_schema_submit'] = array(
      '#type' => 'submit',
      '#value' => t('Get Meta Info'),
    );
  }

  // Token isn't implemented for REST yet in the library but since we haven't
  // even seen it work for SOAP yet, we'll just keep it visible for now and have
  // it throw an exception.
  $form['get_token'] = array(
    '#type' => 'fieldset',
    '#title' => t('Get new token'),
    '#weight' => 11,
  );
  $form['get_token']['get_token_user_id'] = array(
    '#type' => 'textfield',
    '#title' => t('User ID'),
  );
  $form['get_token']['get_token_api_key'] = array(
    '#type' => 'textfield',
    '#title' => t('API key'),
  );
  $form['get_token']['get_token_environment_key'] = array(
    '#type' => 'textfield',
    '#title' => t('Environment key'),
  );
  $form['get_token']['get_token_description'] = array(
    '#type' => 'textfield',
    '#title' => t('Description'),
  );
  $form['get_token']['get_token_submit'] = array(
    '#type' => 'submit',
    '#value' => t('Get Token'),
  );

  $form['get_version'] = array(
    '#type' => 'fieldset',
    '#title' => t('Get version info'),
    '#weight' => 11,
  );
  $form['get_version']['get_version_submit'] = array(
    '#type' => 'submit',
    '#value' => t('Get Version'),
  );

  return $form;
}

/**
 * Form valiate callback for tests form.
 */
function afas_api_test_form_validate(&$form, &$form_state) {
  switch ($form_state['clicked_button']['#value']) {
    case t('Get Data'):
      // We can't make fields required, because we only want to check them
      // if the corresponding button is clicked. If we let this through,
      // Connection::getData() will throw an error eventually, but it'll be
      // strange and we'll lose all the data in the form. (Even if we pressed
      // <Enter> in the form while on a random input element, which gets picked
      // up by the "Get Data" button.)
      if (empty($form_state['values']['getconnector_name'])) {
        form_error($form['getconnector']['getconnector_name'], t('Connector Name is required for "Get Data".'));
      }

      // Ignore POST data for filter if we don't have the right permisson.
      // Effect is that 'getconnector_filter_parsed' value will never be set.
      if (!empty($form_state['values']['getconnector_filter']) && user_access('access afas eval')) {
        $eval = '';
        // We embed the evaluated code in [] because then we can tell the user
        // to enter comma separated single-depth arrays, which fits what we
        // support (which is arrays 2 layser deep) and is easier than letting
        // them mess with more complicated arrays. The eval can cause a parse
        // arror that is uncatchable in PHP5. We could catch it with the
        // following try/catch construct but that itself cannot be parsed on
        // PHP5 because \Error does not exist. Uncomment at some point.
        //    try {
        eval('$eval = [' . $form_state['values']['getconnector_filter'] . '];');
        //    }
        //    catch (\Error $e) {
        //      form_error($form['getconnector']['getconnector_filter'], t("Error parsing 'filter': @e", ['@e' => $e->getMessage()]));
        //    }
        $first = TRUE;
        foreach ($eval as $key => $filter) {
          // The user is encouraged to enter one or more comma separated arrays.
          // This will lead to $key being numeric and $filter being an array.
          // We will however also accept single "key => value" filter constructs
          // as input because the Connection can interpret those. We will even
          // accept numeric keys because the Connection also accepts those
          // (because we do not know whether people can define custom fields
          // with numeric names?)... but one thing we will throw an error for,
          // is if the first key is zero. That is too much of an indicator that
          // the user has just entered a single scalar value by accident.
          if (!is_array($filter) && $first && $key === 0) {
            form_error($form['getconnector']['getconnector_filter'], t("Invalid 'filter'; is just a single un-keyed value: %p", ['%p' => var_export($filter, TRUE)]));
          }
          // Likewise, check the first member inside the array.
          if (is_array($filter) && key($filter) === 0) {
            form_error($form['getconnector']['getconnector_filter'], t("Invalid 'filter' array with key $key; is just a un-keyed scalar value: %p", ['%p' => var_export($filter, TRUE)]));
          }
          $first = FALSE;
        }
        if (!form_get_errors()) {
          // This doesn't guarantee the filter is problem free but it's an array
          // we can pass into the Connection, which will do the other checks.
          $form_state['values']['getconnector_filter_parsed'] = $eval;
        }
      }
      break;

    case t('Send Data'):
    case t('Convert without sending'):
      if (empty($form_state['values']['updateconnector_name'])) {
        form_error($form['updateconnector']['updateconnector_name'], t('Connector Name is required for "Send Data".'));
      }
      if (empty($form_state['values']['updateconnector_action'])) {
        form_error($form['updateconnector']['updateconnector_action'], t('Action is required for "Send Data".'));
      }
      elseif (!is_string($form_state['values']['updateconnector_action'])
        || !in_array(strtolower($form_state['values']['updateconnector_action']), ['insert', 'update'])) {
        form_error($form['updateconnector']['updateconnector_action'], t("Action must be 'insert' or 'update'."));
      }

      // Ignore POST data for filter if we don't have the right permisson.
      // Effect is that 'updateconnector_data_parsed' value will never be set.
      if (empty($form_state['values']['updateconnector_data']) || !is_string($form_state['values']['updateconnector_data'])) {
        form_error($form['updateconnector']['updateconnector_data'], t('Payload Data is required for "Send Data".'));
      }
      elseif (user_access('access afas eval')) {
        $eval = trim($form_state['values']['updateconnector_data']);
        // Expect array delimeters; be strict. Evaluated output is funky. If
        // this is not the format, interpret it as a literal message.
        if ($eval[0] === '[' && substr($eval, -1) === ']') {
          // The eval can cause a parse arror that is uncatchable in PHP5. We
          // could catch it with the following try/catch construct but that
          // itself cannot be parsed on PHP5 because \Error does not exist.
          // Uncomment at some point.
          //    try {
          eval('$eval = ' . $form_state['values']['updateconnector_data'] . ';');
          //    }
          //    catch (\Error $e) {
          //      form_error($form['getconnector']['getconnector_filter'], t("Error parsing 'filter': @e", ['@e' => $e->getMessage()]));
          //    }
          if (!form_get_errors()) {
            // Parse the array notation through UpdateConnector. In order to
            // support custom fields, a form will have to implement a
            // form_alter, maybe to add a validation function before this one,
            // that sets UpdateConnector::$classMap.
            try {
              $type_soap = afas_api_test_client_type() === 'SOAP';
              afas_api_set_custom_fields();
              $form_state['values']['updateconnector_data_parsed'] = UpdateObject::create(
                $form_state['values']['updateconnector_name'],
                $eval,
                $form_state['values']['updateconnector_action'])->output($type_soap ? 'xml' : 'json');
            }
            catch (\Exception $e) {
              form_error($form['updateconnector']['updateconnector_data'], t('While evaluating payload array: ' . $e->getMessage()));
            }
          }
        }
        else {
          if ($form_state['clicked_button']['#value'] === t('Convert without sending')) {
            form_error($form['updateconnector']['updateconnector_data'], t('Payload Data must be array starting/ending with [].'));
          }
          elseif (afas_api_test_client_type() === 'SOAP') {
            // We're not actually checking the XML.
            if ($eval[0] !== '<' && substr($eval, -1) !== '>') {
              form_error($form['updateconnector']['updateconnector_data'], t('Payload Data must be XML starting/ending with {} (or array starting/ending with []).'));
            }
          }
          else {
            // Check the JSON; must be object.
            if ($eval[0] !== '{' && substr($eval, -1) !== '}') {
              form_error($form['updateconnector']['updateconnector_data'], t('Payload Data must be JSON starting/ending with {} (or array starting/ending with []).'));
            }
            $data = json_decode($eval);
            if (!$data) {
              form_error($form['updateconnector']['updateconnector_data'], t('Payload Data is invalid JSON.'));
            }
          }
        }
      }
      break;

    case t('Get Meta Info'):
      if (!empty($form_state['values']['get_schema_get']) && !empty($form_state['values']['get_schema_update'])) {
        form_error($form['get_schema']['get_schema_update'], t('Enter either a GetConnector or an UpdateConnector, not both.'));
      }
  }
}

/**
 * Form submit callback for tests form.
 */
function afas_api_test_form_submit(&$form, &$form_state) {
  $type_soap = afas_api_test_client_type() === 'SOAP';
  $result =  FALSE;
  $pretty = FALSE;
  switch ($form_state['clicked_button']['#value']) {
    case t('Get Data'):
      $args = array();
      if (isset($form_state['values']['getconnector_skip']) && $form_state['values']['getconnector_skip'] !== '') {
        $args['skip'] = $form_state['values']['getconnector_skip'];
      }
      if (isset($form_state['values']['getconnector_take']) && $form_state['values']['getconnector_take'] !== '') {
        $args['take'] = $form_state['values']['getconnector_take'];
      }
      if (isset($form_state['values']['getconnector_orderby']) && $form_state['values']['getconnector_orderby'] !== '') {
        // Supported for SOAP as well as REST; the library converts it for SOAP.
        $args['OrderbyFieldIds'] = $form_state['values']['getconnector_orderby'];
      }
      if (!empty($form_state['values']['getconnector_get_literal']) || !empty($form_state['values']['getconnector_output_pretty'])) {
        $args['options']['Outputmode'] = Connection::GET_OUTPUTMODE_LITERAL;
      }
      if (!empty($form_state['values']['getconnector_get_metadata'])) {
        $args['options']['Metadata'] = Connection::GET_METADATA_YES;
      }
      if (!empty($form_state['values']['getconnector_get_empty'])) {
        $args['options']['Outputoptions'] = Connection::GET_OUTPUTOPTIONS_XML_INCLUDE_EMPTY;
      }
      $filters = array();
      if (!empty($form_state['values']['getconnector_filter_parsed'])) {
        $filters = $form_state['values']['getconnector_filter_parsed'];
      }

      $result = afas_api_get_data($form_state['values']['getconnector_name'], $filters, Connection::DATA_TYPE_GET, $args, AFAS_LOG_ERROR_DETAILS_SCREEN);

      $pretty = !empty($form_state['values']['getconnector_output_pretty']);
      break;

    case t('Convert without sending'):
      // Output the parsed data in the "Payload data" box below, so we can
      // immediately press "Send Data", unformatted.
      $form['updateconnector']['updateconnector_data']['#value'] = $form_state['values']['updateconnector_data_parsed'];
      // Also output it on top of the screen so we see the results immediately,
      // formatted. If we json-decode the result then it will be pretty-printed
      // below.
      $pretty = TRUE;
      if ($type_soap) {
        $result = $form_state['values']['updateconnector_data_parsed'];
      }
      else {
        $result = json_decode($form_state['values']['updateconnector_data_parsed'], TRUE);
        if ($result === NULL) {
          drupal_set_message('JSON could not be decoded?', 'error');
          // If we don't give $result a value, we'll rebuild and the "Payload
          // data" element also will contain nothing.
          $result = '?';
        }
      }
      break;

    case t('Send Data'):
      $result = afas_api_send_data(
        $form_state['values']['updateconnector_name'],
        !empty($form_state['values']['updateconnector_data_parsed'])
          ? $form_state['values']['updateconnector_data_parsed']
          : $form['updateconnector']['updateconnector_data'],
        $form_state['values']['updateconnector_action'],
        AFAS_LOG_ERROR_DETAILS_SCREEN
      );
      break;

    case t('Get Version'):
      $args = array();
      if (!$type_soap) {
        $args['options']['Outputmode'] = Connection::GET_OUTPUTMODE_LITERAL;
      }
      $result = afas_api_get_data('', array(), Connection::DATA_TYPE_VERSION_INFO, $args, AFAS_LOG_ERROR_DETAILS_SCREEN);
      break;

    case t('Get Token'):
      $extra_args = array(
        'apiKey' => $form_state['values']['get_token_api_key'],
        'environmentKey' => $form_state['values']['get_token_environment_key'],
        'description' => $form_state['values']['get_token_description'],
      );
      $result = afas_api_get_data($form_state['values']['get_token_user_id'], array(), 'token', $extra_args, AFAS_LOG_ERROR_DETAILS_SCREEN);
      break;

    case t('Get Meta Info'):
      // Empty get connector == empty update connector; there is no difference.
      $args['options']['Outputmode'] = Connection::GET_OUTPUTMODE_LITERAL;
      if (!empty($form_state['values']['get_schema_get'])) {
        $result = afas_api_get_data($form_state['values']['get_schema_get'], array(), 'metainfo_get', $args, AFAS_LOG_ERROR_DETAILS_SCREEN);
      }
      else {
        $data_id = !empty($form_state['values']['get_schema_update']) ? $form_state['values']['get_schema_update'] : '';
        $result = afas_api_get_data($data_id, array(), 'data', $args, AFAS_LOG_ERROR_DETAILS_SCREEN);
      }
      break;

    case t('Get XSD'):
      // Retrieve / display XSD;
      $result = afas_api_get_data($form_state['values']['get_schema_function'], array(), 'data', array(), AFAS_LOG_ERROR_DETAILS_SCREEN);
      if ($result) {

    /* We got an XML string returned, with the XSD encoded into one of the
     * attributes. This is an example string we got back:
<AfasDataConnector>
  <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema">
    <xs:element name="AfasDataConnector">
      <xs:complexType>
        <xs:choice maxOccurs="unbounded">
          <xs:element name="ConnectorData">
            <xs:complexType>
              <xs:sequence>
                <xs:element name="Id" type="xs:string" minOccurs="0"/>
                <xs:element name="Schema" type="xs:string" minOccurs="0"/>
              </xs:sequence>
            </xs:complexType>
          </xs:element>
        </xs:choice>
      </xs:complexType>
    </xs:element>
  </xs:schema>
  <ConnectorData>
    <Id>knOrganisation</Id>
    <Schema> [ THE SCHEMA DATA ] </Schema>
  </ConnectorData>
</AfasDataConnector>
     * Let's decode that. If we find the schema, we will only print that and
     * silently ignore the rest.
     */
      $doc_element = new SimpleXMLElement($result);
      if (isset($doc_element->ConnectorData->Schema)) {
        $result = strval($doc_element->ConnectorData->Schema);
      }
    }
    break;
  }

  if ($result || is_array($result)) {
    // Do not redirect; print the value directly into the form.
    $form_state['redirect'] = FALSE;
    $form['result'] = array(
      '#type' => 'fieldset',
      '#title' => t('RESULTS:'),
      '#weight' => -99,
    );
    // We'll just ignore 'pretty' if the dom extension is not installed.
    if ($pretty && $type_soap && extension_loaded('dom')) {
      // To pretty print it, we need to make a document from it.
      // A NOTE - when trying this in 2016/2017 on output from AFAS, it worked.
      // Pretty printing does not seem to work for the XML generated for
      // payload data (by "Convert without sending"), though: it comes out
      // still unformatted. We're not looking at that for now.
      $doc_element = new DomDocument('1,0');
      if ($doc_element->loadXML($result)) {
        $doc_element->formatOutput = TRUE;
        $result = $doc_element->saveXML();
        // The original did not contain the XML header though.
        $len = strlen("<?xml version=\"1.0\"?>\n");
        if (substr($result, 0, $len) === "<?xml version=\"1.0\"?>\n") {
          $result = substr($result, $len);
        }
      }
    }
    if (!is_scalar($result)) {
      // In case of a REST client, we got JSON that was just decoded...
      $result = function_exists('json_encode') ? json_encode($result, JSON_PRETTY_PRINT) : var_export($result, TRUE);
    }
    // NOTE - the literal value from the REST API already contains CRLFs,
    // unlike the SOAP API.

    // Since there will be no rebuild (we will just exit the form builder after
    // this submit callback, and the renderer will work on it), use #value.
    $form['result']['output'] = array(
      '#type' => 'textarea',
      '#rows' => 20,
      '#weight' => -99,
      '#value' => $result,
    );
  }
}
