# Affiliated

A Drupal 10.3+ module for tracking affiliate referrals and conversions.

Based on [affiliate_ng](https://www.drupal.org/project/affiliate_ng) (Drupal 7) with significant improvements.

## Table of Contents

- [Introduction](#introduction)
- [How It Works](#how-it-works)
- [Quick Start](#quick-start)
- [Architecture](#architecture)
  - [Clicks](#affiliate_click)
  - [Conversions](#affiliate_conversion)
  - [Campaigns](#affiliate-campaigns-affiliate_campaign)
- [Conversion Types](#conversion-types)
- [Conversion Status](#conversion-status-approved--cancelled)
- [Creating Conversions](#creating-a-conversion)
- [Events](#events)
  - [ConversionPreCreateEvent](#conversionprecreateevent)
  - [AffiliateAccountLookupEvent](#affiliateaccountlookupevent)
  - [AffiliateCodeLookupEvent](#affiliatecodelookupevent)
- [Entity Hooks](#entity-hooks)
- [Submodules](#submodules)

## Introduction

- 'Affiliates' are user accounts.
- The permission 'act as an affiliate' will give a user access to an 'Affiliate
  Center' dashboard on their profile at `/user/UID/affiliate`
- The global config settings are located at `/admin/config/affiliate/settings`
- There are also a handful of permissions for creating campaigns and viewing stats on the permissions page, so make sure that's configured to your liking

### How It Works

The affiliate tracking flow:

```
1. CLICK           2. COOKIE         3. ACTION         4. CONVERSION
   │                   │                 │                 │
   ▼                   ▼                 ▼                 ▼
Visitor clicks     Cookie saved     Visitor does      Conversion entity
affiliate link     with affiliate   something you     created, crediting
(?affiliate=CODE)  & campaign ID    want to track     the affiliate
```

**Step 1 - Click**: A visitor arrives via an affiliate link (e.g.,
`example.com?affiliate=john123`). An `affiliate_click` entity is created
(optional, can be disabled in settings).

**Step 2 - Cookie**: A cookie is saved on the visitor's device containing the
affiliate user ID and campaign. This persists across sessions (configurable
duration).

**Step 3 - Action**: The visitor performs an action you want to track - makes a
purchase, registers an account, submits a form, etc. This can happen
immediately or days/weeks later.

**Step 4 - Conversion**: Your code (or a submodule) creates an
`affiliate_conversion` entity, linking the action to the affiliate stored in
the cookie. The affiliate is credited for the referral.

## Quick Start

1. **Install the module** `drush en affiliated` and enable any submodules you
   need (affiliate_commerce, affiliate_registrations, affiliate_webform)

2. **Configure settings** at `/admin/config/affiliate/settings`:
   - Set the tracking parameter (default: `affiliate`)
   - Set cookie duration
   - Enable/disable click tracking

3. **Grant permissions**: Grant users that you want to be affiliates the "act
   as an affiliate" permission

4. **Conversion types**: Submodules automatically create their own conversion
   types on install. For custom conversions, create types at
   `/admin/structure/affiliate/conversion/types` and set:
   - Default commission amount
   - Default status (approved/pending)

5. **Generate affiliate links**: Affiliates access their dashboard at
   `/user/UID/affiliate` to get tracking links like
   `example.com?affiliate=CODE` (parameter name is configurable)

6. **Track conversions**: Use a submodule or create conversions programmatically:
   ```php
   $manager = \Drupal::service('affiliate.manager');
   if ($affiliate = $manager->getStoredAccount()) {
     $conversion = \Drupal::entityTypeManager()
       ->getStorage('affiliate_conversion')
       ->create([
         'type' => 'my_conversion_type',
         'affiliate' => $affiliate,
         'campaign' => $manager->getStoredCampaign(),
       ]);
     $conversion->save();
   }
   ```

### Architecture

This module creates the following 3 content entity types:

| Entity Type | Description |
|-------------|-------------|
| `affiliate_click` | Records when a visitor arrives via an affiliate link |
| `affiliate_conversion` | Records when a tracked action occurs (sale, signup, etc.) |
| `affiliate_campaign` | Groups/categorizes affiliate traffic sources |

#### affiliate_click

- Every visit to the site using an affiliate link creates an affiliate_click
  entity (if enabled) and saves a cookie on the visitor's device.
- Click entities are useful for reporting (tracking click volume, sources,
  etc.) but are **not required** for tracking conversions - only the cookie
  matters for attributing conversions.
- This happens automatically, you shouldn't ever need to create a click
  entity yourself.

**Click storage settings** (`/admin/config/affiliate/settings`):
- **Store click entities**: Enable/disable click entity creation. Disable to
  reduce database size if you don't need click reporting.
- **Click retention**: Automatically delete click entities older than X hours.
  Set to 0 to keep forever. Runs during cron. Default: 720 hours (30 days).

#### affiliate_conversion

- When an action on the site occurs that you
  consider a "conversion", you need to create a conversion entity. The most
  obvious case for this is a commerce sale, but any action you want could be a conversion.

- Conversions are bundled entities, you can create different bundles/types of
  conversions with their own rules at `/admin/structure/affiliate/conversion/types`

- For Instance: a bundle named 'commerce_orders' could add conversions for
  orders
  while 'webform_submissions' could add conversions for
  submitting a
  webform, both awarding a different commission amount (or no commission at
  all if you just want to track how many times the action occurred).
- The main module itself does not
  automatically create conversions. The creation of conversions should be
  handled by
  submodules. For example, there are two included
  submodules: **affiliate_commerce** creates conversions for commerce orders.
  And **affiliate_registrations** creates conversions for new user account
  registrations.
  But it's mainly left up to you to decide what you consider a 'conversion'

### Conversion Types

Each conversion type (bundle) has configurable settings at
`/admin/structure/affiliate/conversion/types/{type}/edit`:

- **Default commission**: A default commission amount applied to new conversions
  of this type. Can be overridden per-conversion by calling `setCommission()`.
  Leave empty for no default. Note: This is simply a numeric value - how you
  interpret it (dollars, cents, points, percentage, etc.) is up to your
  implementation.

- **Default status**: Whether new conversions are automatically approved.
  - **Approved** (default): Conversions are immediately approved when created.
    Use this for automated/trusted conversion sources.
  - **Pending**: Conversions require manual approval before they count toward
    affiliate payouts. Use this when you need to review conversions before
    crediting the affiliate.

- **Label pattern**: A token pattern for auto-generating conversion labels.
  Available tokens include `[affiliate_conversion:id]`,
  `[affiliate_conversion:affiliate]`, etc.

### Conversion Status (Approved / Cancelled)

Conversions use Drupal's publish status to track approval:
- **Approved** (published): The conversion counts toward affiliate payouts
- **Cancelled** (unpublished): The conversion is excluded from payouts

#### Checking status

```php
// Check if a conversion is approved.
if ($conversion->isApproved()) {
  // Conversion counts toward payouts.
}

// Check if cancelled (opposite of approved).
if (!$conversion->isApproved()) {
  // Conversion is cancelled/pending.
}
```

#### Approving a conversion

```php
// Approve a pending conversion.
$conversion->setPublished();
$conversion->save();
```

#### Cancelling a conversion

Use the `cancel()` method to unpublish and record a reason:

```php
// Cancel with a reason (stored in notes field).
$conversion->cancel('Order was refunded');
$conversion->save();

// Or cancel without the helper method.
$conversion->setUnpublished();
$conversion->setNotes('Fraudulent order');
$conversion->save();
```

Common cancellation reasons: `refund`, `fraud`, `duplicate`, `test order`,
`returned item`.

#### Administrative UI

The admin list pages are views, so they can be customized to your liking.

Conversions can be managed through operation links in the admin UI:
- **Conversions list**: `/admin/config/affiliate/conversions` shows all
  conversions with operation links (view, edit, cancel)
- **Cancel form**: `/affiliate/conversion/{id}/cancel` prompts for a
  cancellation reason
- **Edit form**: Change status, commission, notes, and other fields directly

#### Permissions

There are separate permissions for editing vs approving/cancelling:
- **Edit affiliate conversions**: Full access to edit all conversion fields
- **Approve affiliate conversions**: Only allows approving or cancelling
  conversions without access to edit other details (commission, notes, etc.)

This allows you to give moderators the ability to approve or reject conversions
without granting them full edit access.

### Creating a Conversion

Example for creating a conversion.
```php
// To get an affiliate user account and campaign saved in the cookie.
$affiliateManager = \Drupal::service('affiliate.manager');
$affiliate = $affiliateManager->getStoredAccount();
$campaign = $affiliateManager->getStoredCampaign();

// Only create a conversion if we have an affiliate.
if($affiliate){

  $conversion = \Drupal::entityTypeManager('affiliate_conversion')->create(
    'type' => THE_CONVERSION_BUNDLE
    'affiliate' => $affiliate, // A user account
    'campaign' => $campaign,
  );

  // set an optional entity as a parent.
  // Typically this would be the entity that caused the conversion to be created.
  $conversion->setParentEntity($entity);

  // Set an optional commission amount.
  $conversion->setCommission(123.99);

  // Set an optional label on the conversion
  $conversion->setLabel('Larry bought a t-shirt');

  $conversion->save();
}

```

### Affiliate Campaigns (affiliate_campaign)

- These are entities and act as categories to group where your
  affiliate clicks are coming from (in addition to the affiliate user account
  itself).
- These are fieldable.
- Campaigns can be either global or specific to an affiliate.
- When you install the module a default global campaign is created.
- You can create global campaigns at
  `/admin/config/affiliate/campaigns` or campaigns specific to your affiliate account at `/user/UID/affiliate/campaigns`

## Events

The module dispatches events that allow other modules to interact with the
affiliate system.

### ConversionPreCreateEvent

Dispatched before a conversion is saved. Subscribers can reject the conversion
to prevent it from being recorded. This is useful for validating whether a
sale qualifies for commission.

**Event name:** `affiliated_conversion_pre_create`

**When it fires:** During `preSave()` on new conversions, after all data
(label, commission) has been populated.

**Key methods:**
- `$event->getConversion()` - Get the conversion entity being created
- `$event->reject($reason)` - Reject the conversion with an optional reason
- `$event->isRejected()` - Check if the conversion has been rejected
- `$event->getRejectionReason()` - Get the rejection reason

**Rejection behavior:** When a conversion is rejected:
- `save()` returns `FALSE` instead of the entity ID
- The rejection is automatically logged
- The rejection reason is available via `$conversion->getRejectionReason()`
- No exception is thrown - the save fails silently

#### Example: Reject conversions for specific product types

```php
<?php

namespace Drupal\my_module\EventSubscriber;

use Drupal\affiliated\Event\ConversionPreCreateEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Validates conversions before they are created.
 */
class ConversionValidationSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      ConversionPreCreateEvent::EVENT_NAME => ['onConversionPreCreate'],
    ];
  }

  /**
   * Validates the conversion before creation.
   */
  public function onConversionPreCreate(ConversionPreCreateEvent $event): void {
    $conversion = $event->getConversion();

    // Only check certain conversion types.
    if ($conversion->bundle() !== 'product_sales') {
      return;
    }

    $parent = $conversion->getParentEntity();
    if (!$parent || $parent->getEntityTypeId() !== 'commerce_order_item') {
      return;
    }

    // Check the product type.
    $purchased_entity = $parent->getPurchasedEntity();
    if ($purchased_entity) {
      $product = $purchased_entity->getProduct();
      if ($product && $product->bundle() === 'excluded_type') {
        $event->reject('This product type is not eligible for commission');
      }
    }
  }

}
```

Register the subscriber in your module's services.yml:

```yaml
services:
  my_module.conversion_validation_subscriber:
    class: '\Drupal\my_module\EventSubscriber\ConversionValidationSubscriber'
    tags:
      - { name: 'event_subscriber' }
```

#### Example: Check for rejection after save

```php
$conversion = \Drupal::entityTypeManager()
  ->getStorage('affiliate_conversion')
  ->create([
    'type' => 'product_sales',
    'affiliate' => $affiliate->id(),
  ]);
$conversion->setParentEntity($order_item);

if (!$conversion->save()) {
  // Conversion was rejected.
  $reason = $conversion->getRejectionReason();
  \Drupal::logger('my_module')->notice('Conversion rejected: @reason', [
    '@reason' => $reason,
  ]);
}
```

### AffiliateAccountLookupEvent

Dispatched when looking up an affiliate account from a code. Allows modules to
provide custom code-to-account resolution.

**Event name:** `affiliated_account_lookup`

**Key methods:**
- `$event->getCode()` - Get the affiliate code being looked up
- `$event->setAccount($user)` - Set the resolved user account
- `$event->getAccount()` - Get the resolved account (if set)

#### Example: Look up affiliate by custom profile field

```php
public function onAccountLookup(AffiliateAccountLookupEvent $event): void {
  $code = $event->getCode();

  // Look up by custom profile field.
  $profiles = \Drupal::entityTypeManager()
    ->getStorage('profile')
    ->loadByProperties(['field_affiliate_code' => $code]);

  if ($profiles) {
    $profile = reset($profiles);
    $event->setAccount($profile->getOwner());
  }
}
```

### AffiliateCodeLookupEvent

Dispatched when looking up an affiliate code for a user account. Allows modules
to provide custom account-to-code resolution.

**Event name:** `affiliated_code_lookup`

**Key methods:**
- `$event->getAccount()` - Get the user account
- `$event->setCode($code)` - Set the affiliate code
- `$event->getCode()` - Get the resolved code (if set)

#### Example: Get code from custom profile field

```php
public function onCodeLookup(AffiliateCodeLookupEvent $event): void {
  $account = $event->getAccount();

  $profile = \Drupal::entityTypeManager()
    ->getStorage('profile')
    ->loadByUser($account, 'affiliate');

  if ($profile && $code = $profile->field_affiliate_code->value) {
    $event->setCode($code);
  }
}
```

## Entity Hooks

Since conversions, clicks, and campaigns are standard Drupal content entities,
you can react to them using Drupal's entity hook system. This is useful when
you need to perform actions after a conversion has been successfully saved.

### Available hooks

For conversions (`affiliate_conversion`):
- `hook_affiliate_conversion_insert()` - After a new conversion is created
- `hook_affiliate_conversion_update()` - After a conversion is updated
- `hook_affiliate_conversion_delete()` - After a conversion is deleted
- `hook_affiliate_conversion_presave()` - Before a conversion is saved, but 
  after the `ConversionPreCreateEvent` (If the event rejects the conversion, 
  the save operation is canceled and it never gets to this hook).

Similar hooks exist for `affiliate_click` and `affiliate_campaign`.

### Example: Send notification when conversion is created

```php
<?php

/**
 * Implements hook_affiliate_conversion_insert().
 */
function my_module_affiliate_conversion_insert(\Drupal\affiliated\Entity\AffiliateConversionInterface $conversion) {
  // Get the affiliate user.
  $affiliate = $conversion->getAffiliate();

  // Send them an email notification.
  $mailManager = \Drupal::service('plugin.manager.mail');
  $mailManager->mail('my_module', 'conversion_notification', $affiliate->getEmail(), 'en', [
    'conversion' => $conversion,
    'affiliate' => $affiliate,
  ]);
}
```

### Example: Log conversion details

```php
<?php

/**
 * Implements hook_affiliate_conversion_insert().
 */
function my_module_affiliate_conversion_insert(\Drupal\affiliated\Entity\AffiliateConversionInterface $conversion) {
  \Drupal::logger('my_module')->info('New @type conversion created for affiliate @name. Commission: @amount', [
    '@type' => $conversion->bundle(),
    '@name' => $conversion->getAffiliate()->getDisplayName(),
    '@amount' => $conversion->getCommission() ?? 'none',
  ]);
}
```

### Example: Update external CRM on conversion

```php
<?php

/**
 * Implements hook_affiliate_conversion_insert().
 */
function my_module_affiliate_conversion_insert(\Drupal\affiliated\Entity\AffiliateConversionInterface $conversion) {
  // Only for certain conversion types.
  if ($conversion->bundle() !== 'product_sales') {
    return;
  }

  // Push to external CRM.
  $client = \Drupal::httpClient();
  $client->post('https://api.example.com/conversions', [
    'json' => [
      'affiliate_id' => $conversion->getAffiliate()->id(),
      'amount' => $conversion->getCommission(),
      'reference' => $conversion->id(),
    ],
  ]);
}
```

### Example: React to status changes (approved/cancelled)

```php
<?php

/**
 * Implements hook_affiliate_conversion_update().
 */
function my_module_affiliate_conversion_update(\Drupal\affiliated\Entity\AffiliateConversionInterface $conversion) {
  // React when a conversion is approved.
  if ($conversion->isApproved() && !$conversion->original->isApproved()) {
    \Drupal::logger('my_module')->info('Conversion @id was approved', [
      '@id' => $conversion->id(),
    ]);
    // Notify the affiliate, update external systems, etc.
  }

  // React when a conversion is cancelled.
  if ($conversion->isCancelled() && !$conversion->original->isCancelled()) {
    \Drupal::logger('my_module')->warning('Conversion @id was cancelled', [
      '@id' => $conversion->id(),
    ]);
  }
}
```

### Events vs Hooks

Use **ConversionPreCreateEvent** when you need to:
- Validate or reject a conversion before it's saved

Use **entity hooks** when you need to:
- React after a conversion is successfully saved
- Trigger side effects (notifications, external API calls, logging)
- Access the saved entity ID

## Submodules

### affiliate_commerce

Creates conversions for Commerce orders. Can be configured to create one
conversion per order or one per order item.

### affiliate_registrations

Creates conversions when new user accounts are registered via an affiliate
link.

### affiliate_webform

Creates conversions when webforms are submitted via an affiliate link.
Configure which webforms trigger conversions on the conversion type settings.
