<?php

namespace Drupal\affiliate_commerce\OrderProcessor;

use Drupal\affiliated\AffiliateManager;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\OrderProcessorInterface;
use Drupal\Core\Entity\EntityTypeManager;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Class CommerceAffiliateOrderProcessor.
 *
 * This is run whenever the cart is recalculated. Here we check if we have an
 * affiliate/campaign cookie and set a field on the order that we check later
 * in the order complete subscriber.
 *
 * @see \Drupal\affiliate_commerce\EventSubscriber\CommerceAffiliateCompleteConversion
 *   where conversion is saved on order complete.
 */
class CommerceAffiliateSetOrderAffiliate implements OrderProcessorInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManager
   */
  protected $entityTypeManager;

  /**
   * The affiliate manager service.
   *
   * @var \Drupal\affiliated\AffiliateManager
   */
  protected $affiliateManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManager $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\affiliated\AffiliateManager $affiliate_manager
   *   The affiliate manager service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(EntityTypeManager $entity_type_manager, AffiliateManager $affiliate_manager, AccountProxyInterface $current_user) {
    $this->entityTypeManager = $entity_type_manager;
    $this->affiliateManager = $affiliate_manager;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public function process(OrderInterface $order): void {
    if ($order->getCustomerId() == $this->currentUser->id()) {
      if ($order->hasField('affiliate_account') && $order->affiliate_account->isEmpty()) {
        if ($affiliate = $this->affiliateManager->getStoredAccount()) {
          $log_vars = [
            'affiliate' => $affiliate->toLink(),
            'campaign' => $this->t('none'),
          ];
          $order->set('affiliate_account', $affiliate->id());
          if ($campaign = $this->affiliateManager->getStoredCampaign()) {
            $order->set('affiliate_campaign', $campaign->id());
            $log_vars['campaign'] = $campaign->toLink();
          }
          // Add a commerce log that an affiliate was assigned to the order.
          $logStorage = $this->entityTypeManager->getStorage('commerce_log');
          $logStorage->generate($order, 'affiliate_commerce_added', $log_vars)->save();
        }
      }
    }
  }

}
