<?php

namespace Drupal\Tests\affiliate_registrations\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\user\Entity\User;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpFoundation\Session\Storage\MockArraySessionStorage;

/**
 * Tests the affiliate registrations module.
 *
 * @group affiliated
 */
class AffiliateRegistrationsTest extends KernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'options',
    'datetime',
    'views',
    'affiliated',
    'affiliate_registrations',
  ];

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * An affiliate user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $affiliateUser;

  /**
   * The default campaign.
   *
   * @var \Drupal\affiliated\Entity\AffiliateCampaignInterface
   */
  protected $defaultCampaign;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('affiliate_click');
    $this->installEntitySchema('affiliate_conversion');
    $this->installEntitySchema('affiliate_campaign');
    $this->installConfig(['affiliated', 'affiliate_registrations']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');

    // Create an affiliate user with proper permission.
    $this->affiliateUser = User::create([
      'name' => 'affiliate_user',
      'mail' => 'affiliate@example.com',
      'status' => 1,
    ]);
    $this->affiliateUser->save();

    // Grant the affiliate permission via role.
    $role = $this->entityTypeManager->getStorage('user_role')->create([
      'id' => 'affiliate',
      'label' => 'Affiliate',
    ]);
    $role->grantPermission('act as an affiliate');
    $role->save();
    $this->affiliateUser->addRole('affiliate');
    $this->affiliateUser->save();

    // Create a default campaign.
    $this->defaultCampaign = $this->entityTypeManager
      ->getStorage('affiliate_campaign')
      ->create([
        'user_id' => 0,
        'name' => 'Default Campaign',
        'is_default' => 1,
        'is_global' => 1,
        'status' => 1,
      ]);
    $this->defaultCampaign->save();
  }

  /**
   * Creates a request with a mock session attached.
   *
   * @param string $uri
   *   The request URI.
   *
   * @return \Symfony\Component\HttpFoundation\Request
   *   The request with session.
   */
  protected function createRequestWithSession(string $uri): Request {
    $request = Request::create($uri);
    $request->setSession(new Session(new MockArraySessionStorage()));
    return $request;
  }

  /**
   * Tests that the user_registration conversion type is installed.
   */
  public function testConversionTypeInstalled(): void {
    $conversionType = $this->entityTypeManager
      ->getStorage('affiliate_conversion_type')
      ->load('user_registration');

    $this->assertNotNull($conversionType);
    $this->assertEquals('User Registration', $conversionType->label());
  }

  /**
   * Tests conversion is created when user self-registers with affiliate cookie.
   */
  public function testConversionCreatedOnSelfRegistration(): void {
    // Set up request with affiliate cookie and session.
    $request = $this->createRequestWithSession('/user/register');
    $request->cookies->set('affiliate_id', $this->affiliateUser->id());
    $request->cookies->set('affiliate_campaign', $this->defaultCampaign->id());
    $this->container->get('request_stack')->push($request);

    // Current user is anonymous by default in kernel tests.
    // Create a new user (simulating registration).
    $newUser = User::create([
      'name' => 'new_user',
      'mail' => 'new@example.com',
      'status' => 1,
    ]);
    $newUser->save();

    // Verify conversion was created for this specific user.
    $conversions = $this->entityTypeManager
      ->getStorage('affiliate_conversion')
      ->loadByProperties([
        'affiliate' => $this->affiliateUser->id(),
        'type' => 'user_registration',
        'entity_id' => $newUser->id(),
      ]);

    $this->assertCount(1, $conversions);
    $conversion = reset($conversions);
    $this->assertEquals($this->defaultCampaign->id(), $conversion->get('campaign')->target_id);
    $this->assertEquals($newUser->id(), $conversion->getParentEntityId());
  }

  /**
   * Tests no conversion when admin creates a user.
   */
  public function testNoConversionWhenAdminCreatesUser(): void {
    // First, create admin user WITHOUT affiliate cookie in request.
    // This prevents a conversion being created for the admin user itself.
    $request = $this->createRequestWithSession('/admin/people/create');
    $this->container->get('request_stack')->push($request);

    $admin = User::create([
      'name' => 'admin',
      'mail' => 'admin@example.com',
      'status' => 1,
    ]);
    $admin->save();

    // Set admin as current user (use setAccount, not setUpCurrentUser).
    $this->container->get('current_user')->setAccount($admin);

    // NOW add the affiliate cookie for subsequent requests.
    $request = $this->createRequestWithSession('/admin/people/create');
    $request->cookies->set('affiliate_id', $this->affiliateUser->id());
    $this->container->get('request_stack')->push($request);

    // Verify current user is NOT anonymous before creating new user.
    $this->assertFalse($this->container->get('current_user')->isAnonymous());

    // Create a new user as admin - this should NOT create a conversion
    // because the current user is authenticated (admin), not anonymous.
    $newUser = User::create([
      'name' => 'admin_created_user',
      'mail' => 'admincreated@example.com',
      'status' => 1,
    ]);
    $newUser->save();

    // Verify NO conversion was created for the admin-created user.
    $conversions = $this->entityTypeManager
      ->getStorage('affiliate_conversion')
      ->loadByProperties([
        'type' => 'user_registration',
        'entity_id' => $newUser->id(),
      ]);

    $this->assertCount(0, $conversions);
  }

  /**
   * Tests no conversion when user registers without affiliate cookie.
   */
  public function testNoConversionWithoutAffiliateCookie(): void {
    // Set up request WITHOUT affiliate cookie.
    $request = $this->createRequestWithSession('/user/register');
    $this->container->get('request_stack')->push($request);

    // Current user is anonymous by default in kernel tests.
    // Create a new user.
    $newUser = User::create([
      'name' => 'organic_user',
      'mail' => 'organic@example.com',
      'status' => 1,
    ]);
    $newUser->save();

    // Verify NO conversion was created for this user.
    $conversions = $this->entityTypeManager
      ->getStorage('affiliate_conversion')
      ->loadByProperties([
        'type' => 'user_registration',
        'entity_id' => $newUser->id(),
      ]);

    $this->assertCount(0, $conversions);
  }

  /**
   * Tests conversion without campaign still works.
   */
  public function testConversionWithoutCampaign(): void {
    // Set up request with affiliate cookie but no campaign.
    $request = $this->createRequestWithSession('/user/register');
    $request->cookies->set('affiliate_id', $this->affiliateUser->id());
    $this->container->get('request_stack')->push($request);

    // Current user is anonymous by default in kernel tests.
    // Create a new user.
    $newUser = User::create([
      'name' => 'no_campaign_user',
      'mail' => 'nocampaign@example.com',
      'status' => 1,
    ]);
    $newUser->save();

    // Verify conversion was created for this specific user.
    $conversions = $this->entityTypeManager
      ->getStorage('affiliate_conversion')
      ->loadByProperties([
        'affiliate' => $this->affiliateUser->id(),
        'type' => 'user_registration',
        'entity_id' => $newUser->id(),
      ]);

    $this->assertCount(1, $conversions);
    $conversion = reset($conversions);
    $this->assertEquals($newUser->id(), $conversion->getParentEntityId());
  }

}
