<?php

namespace Drupal\affiliated\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\user\EntityOwnerTrait;

/**
 * Provides the Affiliate Click entity.
 *
 * @ContentEntityType(
 *   id = "affiliate_click",
 *   label = @Translation("Affiliate Click"),
 *   label_collection = @Translation("Affiliate Clicks"),
 *   label_singular = @Translation("affiliate click"),
 *   label_plural = @Translation("affiliate clicks"),
 *   label_count = @PluralTranslation(
 *     singular = "@count affiliate click",
 *     plural = "@count affiliate clicks",
 *   ),
 *   base_table = "affiliate_click",
 *   handlers = {
 *     "form" = {
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *     },
 *   "route_provider" = {
 *     "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider"
 *   },
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "access" =
 *   "Drupal\affiliated\Entity\Handler\AffiliateClickAccessControlHandler",
 *     "list_builder" = "Drupal\Core\Entity\EntityListBuilder",
 *   },
 *   admin_permission = "administer affiliate_click entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *     "owner" = "user_id",
 *     "uid" = "user_id",
 *   },
 *   links = {
 *     "delete-form" = "/affiliate/click/{affiliate_click}/delete",
 *     "collection" = "/admin/config/affiliate/clicks",
 *   },
 * )
 */
class AffiliateClick extends ContentEntityBase implements AffiliateClickInterface {

  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields += static::ownerBaseFieldDefinitions($entity_type);

    $fields['affiliate'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(new TranslatableMarkup('Affiliate'))
      ->setDescription(new TranslatableMarkup('The affiliate account of the click'))
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setDisplayConfigurable("view", TRUE)
      ->setDisplayConfigurable("form", TRUE);

    $fields['campaign'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(new TranslatableMarkup('Campaign'))
      ->setDescription(new TranslatableMarkup('The referenced campaign.'))
      ->setSetting('target_type', 'affiliate_campaign')
      ->setSetting('handler', 'default')
      ->setDisplayConfigurable("view", TRUE)
      ->setDisplayConfigurable("form", TRUE);

    $fields['hostname'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Hostname'))
      ->setDescription(new TranslatableMarkup("The IP address that made the click (clicked the affiliate's link)."));

    $fields['referrer'] = BaseFieldDefinition::create('string_long')
      ->setLabel(new TranslatableMarkup('Referrer'))
      ->setDescription(new TranslatableMarkup('The referring site.'));

    $fields['destination'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Destination'))
      ->setDescription(new TranslatableMarkup('The click destination.'));

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(new TranslatableMarkup("Created"))
      ->setDescription(new TranslatableMarkup("The time of the click."));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    return new TranslatableMarkup('Affiliate Click #%id', ['%id' => $this->id()]);
  }

}
