<?php

namespace Drupal\affiliated\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityPublishedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Session\AnonymousUserSession;
use Drupal\affiliated\Event\ConversionPreCreateEvent;
use Drupal\affiliated\Exception\ConversionRejectedException;
use Drupal\user\EntityOwnerTrait;

/**
 * Provides the Affiliate Conversion entity.
 *
 * @ContentEntityType(
 *   id = "affiliate_conversion",
 *   label = @Translation("Affiliate Conversion"),
 *   label_collection = @Translation("Affiliate Conversions"),
 *   label_singular = @Translation("Affiliate Conversion"),
 *   label_plural = @Translation("Affiliate Conversions"),
 *   label_count = @PluralTranslation(
 *     singular = "@count Affiliate Conversion",
 *     plural = "@count Affiliate Conversions",
 *   ),
 *   bundle_label = @Translation("Affiliate Conversion Type"),
 *   base_table = "affiliate_conversion",
 *   bundle_entity_type = "affiliate_conversion_type",
 *   field_ui_base_route = "entity.affiliate_conversion_type.edit_form",
 *   permission_granularity = "bundle",
 *   handlers = {
 *     "storage" = "Drupal\affiliated\Storage\AffiliateConversionStorage",
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *     "views_data" = "Drupal\affiliated\Entity\AffiliateConversionViewsData",
 *     "list_builder" = "Drupal\affiliated\Entity\Handler\AffiliateConversionListBuilder",
 *     "form" = {
 *       "default" = "Drupal\Core\Entity\ContentEntityForm",
 *       "add" = "Drupal\Core\Entity\ContentEntityForm",
 *       "edit" = "Drupal\Core\Entity\ContentEntityForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "approve" = "Drupal\affiliated\Form\ApproveConversionForm",
 *       "cancel" = "Drupal\affiliated\Form\CancelConversionForm",
 *     },
 *   },
 *   admin_permission = "administer affiliate_conversion entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "bundle" = "type",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *     "langcode" = "langcode",
 *     "owner" = "user_id",
 *     "uid" = "user_id",
 *     "published" = "status",
 *   },
 *   links = {
 *     "add-page" = "/affiliate/conversion/add",
 *     "add-form" = "/affiliate/conversion/add/{affiliate_conversion_type}",
 *     "canonical" = "/affiliate/conversion/{affiliate_conversion}",
 *     "collection" = "/admin/config/affiliate/conversions",
 *     "delete-form" = "/affiliate/conversion/{affiliate_conversion}/delete",
 *     "edit-form" = "/affiliate/conversion/{affiliate_conversion}/edit",
 *     "approve-form" = "/affiliate/conversion/{affiliate_conversion}/approve",
 *     "cancel-form" = "/affiliate/conversion/{affiliate_conversion}/cancel",
 *   },
 * )
 */
class AffiliateConversion extends ContentEntityBase implements AffiliateConversionInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;
  use EntityPublishedTrait;

  /**
   * The rejection reason if save() was rejected.
   *
   * This is set by AffiliateConversionStorage when a
   * ConversionPreCreateEvent subscriber rejects the conversion.
   *
   * @var string
   */
  public string $rejectionReason = '';

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'affiliate' => 0,
    ];

    // Apply the bundle's default status if not explicitly set.
    if (!isset($values['status']) && !empty($values['type'])) {
      $bundle = \Drupal::entityTypeManager()
        ->getStorage('affiliate_conversion_type')
        ->load($values['type']);
      if ($bundle) {
        $values['status'] = $bundle->getDefaultStatus();
        // If starting unapproved, set a default note.
        if (!$values['status'] && empty($values['notes'])) {
          $values['notes'] = 'Pending admin approval';
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    parent::preSave($storage);

    if (empty($this->label->value)) {
      $label = $this->getBundle()->getLabelPattern();
      $label = \Drupal::token()->replace($label, ['affiliate_conversion' => $this]);
      $this->set('label', $label);
    }

    // Apply default commission from the conversion type if not set.
    // Use NULL check instead of empty() so explicit 0 values are preserved.
    if ($this->amount->value === NULL) {
      $default = $this->getBundle()->getDefaultCommission();
      if ($default !== NULL) {
        $this->set('amount', $default);
      }
    }

    // Validate campaign ownership for new conversions.
    // Non-global campaigns can only be used by their owner.
    if ($this->isNew() && !$this->isSyncing()) {
      $campaign = $this->getCampaign();
      $affiliate_id = $this->getAffiliateId();
      if ($campaign && !$campaign->isGlobal() && $campaign->getOwnerId() != $affiliate_id) {
        // Replace with default campaign.
        $default = \Drupal::service('affiliate.manager')->getDefaultCampaign();
        $this->set('campaign', $default?->id());
      }
    }

    // Dispatch event for new conversions AFTER all data is populated.
    // This allows implementing modules to inspect the full conversion
    // (including commission from hooks) before deciding to reject.
    if ($this->isNew() && !$this->isSyncing()) {
      $event = new ConversionPreCreateEvent($this);
      \Drupal::service('event_dispatcher')->dispatch($event, $event::EVENT_NAME);

      if ($event->isRejected()) {
        $reason = $event->getRejectionReason() ?? 'Conversion rejected';
        \Drupal::logger('affiliated')->notice('Conversion rejected for affiliate @affiliate: @reason', [
          '@affiliate' => $this->getAffiliateId() ?? 'unknown',
          '@reason' => $reason,
        ]);
        throw new ConversionRejectedException($reason);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);
    $fields += static::ownerBaseFieldDefinitions($entity_type);
    $fields += static::publishedBaseFieldDefinitions($entity_type);

    // Make the status field visible on the form with "Approved" label.
    $fields['status']
      ->setLabel(t('Approved'))
      ->setDescription(t('Whether this conversion is approved.'))
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'boolean',
        'weight' => -1,
        'settings' => [
          'format' => 'yes-no',
        ],
      ])
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'weight' => 90,
        'settings' => [
          'display_label' => TRUE,
        ],
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayConfigurable('form', TRUE);

    $fields['label'] = BaseFieldDefinition::create('string')
      ->setLabel(t("Label"))
      ->setDescription(t("The line item label for the conversion. Empty labels will be auto generated based on bundle settings."))
      ->setRequired(FALSE)
      ->setTranslatable(TRUE)
      ->setSetting("max_length", 255)
      ->setDisplayOptions("view", [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions("form", [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable("view", TRUE)
      ->setDisplayConfigurable("form", TRUE);

    // Configure the owner field from EntityOwnerTrait.
    $fields['user_id']
      ->setLabel(t('Converted User'))
      ->setDescription(t('The user who performed the converting action (e.g., registered, purchased).'))
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'entity_reference_label',
        'weight' => 5,
      ])
      ->setDisplayConfigurable("view", TRUE)
      ->setDisplayConfigurable("form", TRUE);

    $fields['affiliate'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Affiliate'))
      ->setDescription(t('The affiliate user account'))
      ->setTranslatable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'entity_reference_label',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => -4,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable("view", TRUE)
      ->setDisplayConfigurable("form", TRUE);

    $fields['campaign'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Campaign'))
      ->setDescription(t('The referenced campaign.'))
      ->setSetting('target_type', 'affiliate_campaign')
      ->setSetting('handler', 'default')
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'entity_reference_label',
        'weight' => 1,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => -3,
      ])
      ->setDisplayConfigurable("view", TRUE)
      ->setDisplayConfigurable("form", TRUE);

    $fields['entity_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Parent Entity Id'))
      ->setDescription(t("The Id of the parent entity."))
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
        'weight' => 15,
      ])
      ->setDisplayConfigurable("view", TRUE)
      ->setDisplayConfigurable("form", TRUE);

    $fields['entity_type'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Parent Entity Type'))
      ->setDescription(t("The Type of the parent entity."))
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
        'weight' => 16,
      ])
      ->setDisplayConfigurable("view", TRUE)
      ->setDisplayConfigurable("form", TRUE);

    $fields['amount'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t("Commission"))
      ->setDescription(t("The commission amount for this conversion."))
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'number_decimal',
        'weight' => 2,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => -2,
      ])
      ->setDisplayConfigurable("view", TRUE)
      ->setDisplayConfigurable("form", TRUE);

    $fields['currency'] = BaseFieldDefinition::create('string')
      ->setLabel(t("Currency"))
      ->setDescription(t("The currency of the conversion amount."))
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
        'weight' => 3,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -1,
      ])
      ->setDisplayConfigurable("view", TRUE)
      ->setDisplayConfigurable("form", TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t("Created"))
      ->setDescription(t("The time the conversion was created."))
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'timestamp',
        'weight' => 10,
      ])
      ->setDisplayConfigurable("view", TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t("Changed"))
      ->setDescription(t("The time the conversion was updated."))
      ->setTranslatable(TRUE)
      ->setDisplayConfigurable("view", TRUE);

    $fields['notes'] = BaseFieldDefinition::create('string')
      ->setLabel(t("Admin Notes"))
      ->setDescription(t("Administrative notes about this conversion (e.g., refunded, fraud, duplicate)."))
      ->setSetting("max_length", 255)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
        'weight' => 20,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 95,
      ])
      ->setDisplayConfigurable("view", TRUE)
      ->setDisplayConfigurable("form", TRUE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function getBundle() {
    return $this->type->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getAffiliate() {
    $user = $this->affiliate->entity ?? NULL;
    return $user ?? new AnonymousUserSession();
  }

  /**
   * {@inheritdoc}
   */
  public function getAffiliateId() {
    return $this->affiliate->target_id ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getCampaign() {
    return $this->campaign->entity ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getCampaignId() {
    return $this->campaign->target_id ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getParentEntity() {
    $type = $this->getParentEntityTypeId();
    $id = $this->getParentEntityId();

    if ($type && $id) {
      return $this->entityTypeManager()->getStorage($type)->load($id);
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getParentEntityTypeId() {
    return $this->entity_type->value ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getParentEntityId() {
    return $this->entity_id->value ?? NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function setParentEntity(EntityInterface $entity) {
    $this->set('entity_id', $entity->id());
    $this->set('entity_type', $entity->getEntityTypeId());
  }

  /**
   * {@inheritDoc}
   */
  public function setCommission(float $value, string $currency = '') {
    $this->set('amount', $value);
    $this->set('currency', $currency);
  }

  /**
   * {@inheritDoc}
   */
  public function getCommission() {
    return [
      'amount' => $this->amount->value ?? 0,
      'currency' => $this->currency->value ?? '',
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function isApproved() {
    return $this->isPublished();
  }

  /**
   * {@inheritDoc}
   */
  public function getNotes(): ?string {
    return $this->notes->value ?? NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function setNotes(?string $notes): static {
    $this->set('notes', $notes);
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function cancel(string $reason): static {
    $this->setUnpublished();
    $this->setNotes($reason);
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getRejectionReason(): ?string {
    return $this->rejectionReason ?: NULL;
  }

}
