<?php

namespace Drupal\affiliated\Storage;

use Drupal\affiliated\Exception\ConversionRejectedException;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\Sql\SqlContentEntityStorage;

/**
 * Storage handler for affiliate conversions.
 *
 * This handler catches ConversionRejectedException thrown during preSave()
 * and handles it silently, returning FALSE instead of throwing. The rejection
 * reason is stored on the entity for optional inspection.
 *
 * @code
 * $conversion->save();
 *
 * // Optional: check if rejected and get reason
 * if ($reason = $conversion->getRejectionReason()) {
 *   \Drupal::logger('my_module')->notice('Rejected: @reason', ['@reason' => $reason]);
 * }
 * @endcode
 */
class AffiliateConversionStorage extends SqlContentEntityStorage {

  /**
   * {@inheritdoc}
   */
  public function save(EntityInterface $entity) {
    try {
      return parent::save($entity);
    }
    catch (\Exception $e) {
      $previous = $e->getPrevious();
      if ($previous instanceof ConversionRejectedException) {
        // Store the rejection reason on the entity for optional inspection.
        $entity->rejectionReason = $previous->getMessage();
        return FALSE;
      }
      // Re-throw other exceptions (DB errors, etc.).
      throw $e;
    }
  }

}
