<?php

namespace Drupal\Tests\affiliated\Kernel;

/**
 * Tests the AffiliateManager service.
 *
 * @group affiliated
 * @coversDefaultClass \Drupal\affiliated\AffiliateManager
 */
class AffiliateManagerTest extends AffiliatedKernelTestBase {

  /**
   * Tests isActiveAffiliate() with a user who has permission.
   *
   * @covers ::isActiveAffiliate
   */
  public function testIsActiveAffiliateWithPermission(): void {
    $user = $this->createAffiliateUser();
    $this->assertTrue($this->affiliateManager->isActiveAffiliate($user));
  }

  /**
   * Tests isActiveAffiliate() with a user who lacks permission.
   *
   * @covers ::isActiveAffiliate
   */
  public function testIsActiveAffiliateWithoutPermission(): void {
    // Create a second user (not user 1) without affiliate permission.
    // User 1 always has all permissions.
    $this->createUser([]);
    $user = $this->createUser([]);
    $this->assertFalse($this->affiliateManager->isActiveAffiliate($user));
  }

  /**
   * Tests getDefaultCampaign() returns the default campaign.
   *
   * @covers ::getDefaultCampaign
   */
  public function testGetDefaultCampaign(): void {
    // Create a default global campaign.
    $campaign = $this->entityTypeManager->getStorage('affiliate_campaign')->create([
      'name' => 'Default Campaign',
      'is_default' => 1,
      'is_global' => 1,
    ]);
    $campaign->save();

    $default = $this->affiliateManager->getDefaultCampaign();
    $this->assertNotNull($default);
    $this->assertEquals('Default Campaign', $default->label());
    $this->assertTrue($default->isDefault());
  }

  /**
   * Tests getAccountFromCode() with user ID code type.
   *
   * @covers ::getAccountFromCode
   */
  public function testGetAccountFromCodeWithUserId(): void {
    $user = $this->createAffiliateUser();

    // Default config uses user_id.
    $account = $this->affiliateManager->getAccountFromCode($user->id());
    $this->assertNotNull($account);
    $this->assertEquals($user->id(), $account->id());
  }

  /**
   * Tests getAccountFromCode() with invalid code.
   *
   * @covers ::getAccountFromCode
   */
  public function testGetAccountFromCodeWithInvalidCode(): void {
    $account = $this->affiliateManager->getAccountFromCode('invalid');
    $this->assertNull($account);
  }

  /**
   * Tests getAccountFromCode() with non-affiliate user.
   *
   * @covers ::getAccountFromCode
   */
  public function testGetAccountFromCodeWithNonAffiliate(): void {
    // Create a second user (not user 1) without affiliate permission.
    // User 1 always has all permissions.
    $this->createUser([]);
    $user = $this->createUser([]);

    $account = $this->affiliateManager->getAccountFromCode($user->id());
    $this->assertNull($account);
  }

  /**
   * Tests getCodeFromAccount() with user ID code type.
   *
   * @covers ::getCodeFromAccount
   */
  public function testGetCodeFromAccount(): void {
    $user = $this->createAffiliateUser();

    $code = $this->affiliateManager->getCodeFromAccount($user);
    $this->assertEquals($user->id(), $code);
  }

  /**
   * Tests getCampaignFromCode() returns default for invalid code.
   *
   * @covers ::getCampaignFromCode
   */
  public function testGetCampaignFromCodeReturnsDefault(): void {
    // Create a default global campaign.
    $defaultCampaign = $this->entityTypeManager->getStorage('affiliate_campaign')->create([
      'name' => 'Default',
      'is_default' => 1,
      'is_global' => 1,
    ]);
    $defaultCampaign->save();

    $campaign = $this->affiliateManager->getCampaignFromCode('invalid');
    $this->assertNotNull($campaign);
    $this->assertTrue($campaign->isDefault());
  }

  /**
   * Tests getCampaignFromCode() with valid campaign ID.
   *
   * @covers ::getCampaignFromCode
   */
  public function testGetCampaignFromCodeWithValidId(): void {
    $campaign = $this->entityTypeManager->getStorage('affiliate_campaign')->create([
      'user_id' => 0,
      'name' => 'Test Campaign',
      'status' => 1,
    ]);
    $campaign->save();

    $loaded = $this->affiliateManager->getCampaignFromCode($campaign->id());
    $this->assertNotNull($loaded);
    $this->assertEquals('Test Campaign', $loaded->label());
  }

  /**
   * Tests getCampaignFromCode() with valid campaign code.
   *
   * @covers ::getCampaignFromCode
   */
  public function testGetCampaignFromCodeWithValidCode(): void {
    $campaign = $this->entityTypeManager->getStorage('affiliate_campaign')->create([
      'user_id' => 0,
      'name' => 'Code Campaign',
      'code' => 'summer-promo',
      'status' => 1,
    ]);
    $campaign->save();

    $loaded = $this->affiliateManager->getCampaignFromCode('summer-promo');
    $this->assertNotNull($loaded);
    $this->assertEquals('Code Campaign', $loaded->label());
  }

  /**
   * Tests getCampaignFromCode() with unpublished campaign code.
   *
   * @covers ::getCampaignFromCode
   */
  public function testGetCampaignFromCodeWithUnpublishedCode(): void {
    // Create a default global campaign for fallback.
    $defaultCampaign = $this->entityTypeManager->getStorage('affiliate_campaign')->create([
      'name' => 'Default',
      'is_default' => 1,
      'is_global' => 1,
      'status' => 1,
    ]);
    $defaultCampaign->save();

    // Create an unpublished campaign with code.
    $campaign = $this->entityTypeManager->getStorage('affiliate_campaign')->create([
      'user_id' => 0,
      'name' => 'Unpublished Campaign',
      'code' => 'inactive',
      'status' => 0,
    ]);
    $campaign->save();

    // Should return default since the coded campaign is unpublished.
    $loaded = $this->affiliateManager->getCampaignFromCode('inactive');
    $this->assertNotNull($loaded);
    $this->assertTrue($loaded->isDefault());
  }

}
