<?php

namespace Drupal\Tests\affiliated\Kernel\Entity;

use Drupal\affiliated\Event\ConversionPreCreateEvent;
use Drupal\node\Entity\NodeType;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\Tests\affiliated\Kernel\AffiliatedKernelTestBase;

/**
 * Tests the AffiliateConversion entity.
 *
 * @group affiliated
 * @coversDefaultClass \Drupal\affiliated\Entity\AffiliateConversion
 */
class AffiliateConversionTest extends AffiliatedKernelTestBase {

  /**
   * A conversion type for testing.
   *
   * @var \Drupal\affiliated\Entity\AffiliateConversionTypeInterface
   */
  protected $conversionType;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['affiliated']);

    // Create a conversion type for testing with a default commission.
    $this->conversionType = $this->entityTypeManager->getStorage('affiliate_conversion_type')->create([
      'id' => 'test_conversion',
      'label' => 'Test Conversion',
      'description' => 'A test conversion type',
      'label_pattern' => 'Conversion [affiliate_conversion:id]',
      'default_commission' => 5.00,
    ]);
    $this->conversionType->save();
  }

  /**
   * Tests creating a conversion entity.
   */
  public function testCreateConversion(): void {
    $user = $this->createAffiliateUser();
    $campaign = $this->entityTypeManager->getStorage('affiliate_campaign')->create([
      'user_id' => 0,
      'name' => 'Test Campaign',
      'is_default' => 1,
    ]);
    $campaign->save();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
      'campaign' => $campaign->id(),
    ]);
    $conversion->save();

    $this->assertNotEmpty($conversion->id());
    $this->assertEquals('test_conversion', $conversion->bundle());
    $this->assertEquals($user->id(), $conversion->getAffiliateId());
  }

  /**
   * Tests setting commission on a conversion.
   *
   * @covers ::setCommission
   * @covers ::getCommission
   */
  public function testSetCommission(): void {
    $user = $this->createAffiliateUser();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
    ]);
    $conversion->setCommission(25.50, 'USD');
    $conversion->save();

    $commission = $conversion->getCommission();
    $this->assertEquals(25.50, $commission['amount']);
    $this->assertEquals('USD', $commission['currency']);
  }

  /**
   * Tests getAffiliate() returns the user entity.
   *
   * @covers ::getAffiliate
   */
  public function testGetAffiliate(): void {
    $user = $this->createAffiliateUser();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
    ]);
    $conversion->save();

    $affiliate = $conversion->getAffiliate();
    $this->assertNotNull($affiliate);
    $this->assertEquals($user->id(), $affiliate->id());
  }

  /**
   * Tests conversion without campaign.
   */
  public function testConversionWithoutCampaign(): void {
    $user = $this->createAffiliateUser();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
    ]);
    $conversion->save();

    $this->assertNotEmpty($conversion->id());
    $this->assertNull($conversion->get('campaign')->entity);
  }

  /**
   * Tests that created timestamp is set automatically.
   */
  public function testCreatedTimestamp(): void {
    $user = $this->createAffiliateUser();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
    ]);
    $conversion->save();

    $this->assertNotEmpty($conversion->get('created')->value);
    $this->assertLessThanOrEqual(time(), $conversion->get('created')->value);
  }

  /**
   * Tests that default commission from conversion type is applied on save.
   */
  public function testDefaultCommissionApplied(): void {
    // Conversion type has default_commission of 5.00 from setUp.
    $user = $this->createAffiliateUser();

    // Create conversion with NULL amount to trigger default commission.
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
      'amount' => NULL,
    ]);
    $conversion->save();

    // Verify the default commission (5.00) was applied.
    $this->assertEquals(5.00, $conversion->get('amount')->value);
  }

  /**
   * Tests that explicit commission overrides default.
   */
  public function testExplicitCommissionOverridesDefault(): void {
    // Conversion type has default_commission of 5.00 from setUp.
    $user = $this->createAffiliateUser();

    // Create conversion with explicit amount (10.00).
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
      'amount' => 10.00,
    ]);
    $conversion->save();

    // Verify explicit amount was kept, not overwritten by default (5.00).
    $this->assertEquals(10.00, $conversion->get('amount')->value);
  }

  /**
   * Tests that explicit zero commission is preserved (not replaced by default).
   */
  public function testExplicitZeroCommissionPreserved(): void {
    // Conversion type has default_commission of 5.00 from setUp.
    $user = $this->createAffiliateUser();

    // Create conversion with explicit zero amount.
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
      'amount' => 0,
    ]);
    $conversion->save();

    // Verify explicit zero was kept, not overwritten by default (5.00).
    $this->assertEquals(0, $conversion->get('amount')->value);
  }

  /**
   * Tests that NULL amount with no default commission stays NULL.
   */
  public function testNullAmountWithNoDefaultStaysNull(): void {
    // Clear the default commission for this test.
    $this->conversionType->set('default_commission', NULL);
    $this->conversionType->save();

    $user = $this->createAffiliateUser();

    // Create conversion with NULL amount and no default commission configured.
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
      'amount' => NULL,
    ]);
    $conversion->save();

    // Amount should remain NULL since no default is configured.
    $this->assertNull($conversion->get('amount')->value);
  }

  /**
   * Tests setting and getting a parent entity.
   *
   * @covers ::setParentEntity
   * @covers ::getParentEntity
   * @covers ::getParentEntityId
   * @covers ::getParentEntityTypeId
   */
  public function testParentEntity(): void {
    $affiliate = $this->createAffiliateUser();

    // Create another user to be the parent entity.
    $parentUser = $this->entityTypeManager->getStorage('user')->create([
      'name' => 'parent_user',
      'mail' => 'parent@example.com',
      'status' => 1,
    ]);
    $parentUser->save();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $affiliate->id(),
    ]);
    $conversion->setParentEntity($parentUser);
    $conversion->save();

    // Verify parent entity info is stored correctly.
    $this->assertEquals($parentUser->id(), $conversion->getParentEntityId());
    $this->assertEquals('user', $conversion->getParentEntityTypeId());

    // Verify we can retrieve the actual entity.
    $loadedParent = $conversion->getParentEntity();
    $this->assertNotNull($loadedParent);
    $this->assertEquals($parentUser->id(), $loadedParent->id());
    $this->assertEquals('parent_user', $loadedParent->getAccountName());
  }

  /**
   * Tests conversion without parent entity.
   */
  public function testConversionWithoutParentEntity(): void {
    $affiliate = $this->createAffiliateUser();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $affiliate->id(),
    ]);
    $conversion->save();

    // Parent entity methods should return null.
    $this->assertNull($conversion->getParentEntityId());
    $this->assertNull($conversion->getParentEntityTypeId());
    $this->assertNull($conversion->getParentEntity());
  }

  /**
   * Tests parent entity works with node.
   *
   * The parent can be any entity - a node that triggered the conversion.
   */
  public function testParentEntityWithNode(): void {
    // Install node module dependencies.
    $this->enableModules(['node', 'field', 'text', 'filter']);
    $this->installEntitySchema('node');

    // Create a content type.
    $nodeType = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $nodeType->save();

    // Create a node - use container to get fresh entity type manager.
    $node = \Drupal::entityTypeManager()->getStorage('node')->create([
      'type' => 'article',
      'title' => 'Test Article',
      'uid' => 1,
    ]);
    $node->save();

    $affiliate = $this->createAffiliateUser();
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $affiliate->id(),
    ]);
    $conversion->setParentEntity($node);
    $conversion->save();

    // Verify it stores the correct entity type.
    $this->assertEquals($node->id(), $conversion->getParentEntityId());
    $this->assertEquals('node', $conversion->getParentEntityTypeId());

    // Verify we can load the node back.
    $loadedParent = $conversion->getParentEntity();
    $this->assertNotNull($loadedParent);
    $this->assertEquals('Test Article', $loadedParent->label());
  }

  /**
   * Tests parent entity works with taxonomy term.
   *
   * The parent can be any entity - a term representing what was converted.
   */
  public function testParentEntityWithTaxonomyTerm(): void {
    // Install taxonomy module dependencies.
    $this->enableModules(['taxonomy', 'text', 'field', 'filter']);
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('taxonomy_vocabulary');

    // Create a vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'products',
      'name' => 'Products',
    ]);
    $vocabulary->save();

    // Create a term - use container to get fresh entity type manager.
    $term = \Drupal::entityTypeManager()->getStorage('taxonomy_term')->create([
      'vid' => 'products',
      'name' => 'Product A',
    ]);
    $term->save();

    $affiliate = $this->createAffiliateUser();
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $affiliate->id(),
    ]);
    $conversion->setParentEntity($term);
    $conversion->save();

    // Verify it stores the correct entity type.
    $this->assertEquals($term->id(), $conversion->getParentEntityId());
    $this->assertEquals('taxonomy_term', $conversion->getParentEntityTypeId());

    // Verify we can load the term back.
    $loadedParent = $conversion->getParentEntity();
    $this->assertNotNull($loadedParent);
    $this->assertEquals('Product A', $loadedParent->label());
  }

  /**
   * Tests setting and getting notes.
   *
   * @covers ::getNotes
   * @covers ::setNotes
   */
  public function testNotes(): void {
    $user = $this->createAffiliateUser();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
    ]);
    $conversion->save();

    // Initially notes should be null.
    $this->assertNull($conversion->getNotes());

    // Set notes.
    $conversion->setNotes('Test note');
    $conversion->save();

    // Reload and verify.
    $loaded = $this->entityTypeManager->getStorage('affiliate_conversion')->load($conversion->id());
    $this->assertEquals('Test note', $loaded->getNotes());
  }

  /**
   * Tests the cancel method.
   *
   * @covers ::cancel
   */
  public function testCancel(): void {
    $user = $this->createAffiliateUser();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
      'status' => 1,
    ]);
    $conversion->save();

    // Initially should be published.
    $this->assertTrue($conversion->isPublished());

    // Cancel the conversion.
    $conversion->cancel('refund');
    $conversion->save();

    // Reload and verify.
    $loaded = $this->entityTypeManager->getStorage('affiliate_conversion')->load($conversion->id());
    $this->assertFalse($loaded->isPublished());
    $this->assertEquals('refund', $loaded->getNotes());
  }

  /**
   * Tests default status from conversion type.
   */
  public function testDefaultStatusApplied(): void {
    // Set default_status to FALSE (unapproved).
    $this->conversionType->set('default_status', FALSE);
    $this->conversionType->save();

    $user = $this->createAffiliateUser();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
    ]);
    $conversion->save();

    // Conversion should start as unpublished.
    $this->assertFalse($conversion->isPublished());
    // And should have the pending note.
    $this->assertEquals('Pending admin approval', $conversion->getNotes());
  }

  /**
   * Tests default status TRUE (approved) from conversion type.
   */
  public function testDefaultStatusApprovedByDefault(): void {
    // Ensure default_status is TRUE (the default).
    $this->conversionType->set('default_status', TRUE);
    $this->conversionType->save();

    $user = $this->createAffiliateUser();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
    ]);
    $conversion->save();

    // Conversion should start as published.
    $this->assertTrue($conversion->isPublished());
    // Notes should be empty.
    $this->assertNull($conversion->getNotes());
  }

  /**
   * Tests explicit status overrides default_status.
   */
  public function testExplicitStatusOverridesDefault(): void {
    // Set default_status to FALSE.
    $this->conversionType->set('default_status', FALSE);
    $this->conversionType->save();

    $user = $this->createAffiliateUser();

    // Explicitly set status to 1 (published).
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
      'status' => 1,
    ]);
    $conversion->save();

    // Explicit status should be honored.
    $this->assertTrue($conversion->isPublished());
  }

  /**
   * Tests isApproved is alias for isPublished.
   *
   * @covers ::isApproved
   */
  public function testIsApproved(): void {
    $user = $this->createAffiliateUser();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
      'status' => 1,
    ]);
    $conversion->save();

    $this->assertTrue($conversion->isApproved());

    $conversion->setUnpublished();
    $this->assertFalse($conversion->isApproved());
  }

  /**
   * Tests that ConversionPreCreateEvent is dispatched.
   */
  public function testPreCreateEventDispatched(): void {
    $user = $this->createAffiliateUser();
    $eventFired = FALSE;

    // Add an event subscriber to verify the event is dispatched.
    $listener = function ($event) use (&$eventFired) {
      $eventFired = TRUE;
      $this->assertInstanceOf(ConversionPreCreateEvent::class, $event);
      $this->assertNotNull($event->getConversion());
    };

    $this->container->get('event_dispatcher')->addListener(
      ConversionPreCreateEvent::EVENT_NAME,
      $listener
    );

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
    ]);
    $conversion->save();

    $this->assertTrue($eventFired, 'ConversionPreCreateEvent was dispatched');
  }

  /**
   * Tests that conversion can be rejected via event.
   */
  public function testConversionRejectedViaEvent(): void {
    $user = $this->createAffiliateUser();

    // Add an event subscriber that rejects conversions.
    $listener = function ($event) {
      $event->reject('Test rejection reason');
    };

    $this->container->get('event_dispatcher')->addListener(
      ConversionPreCreateEvent::EVENT_NAME,
      $listener
    );

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
    ]);

    // Save returns FALSE when rejected (silent approach).
    $result = $conversion->save();
    $this->assertFalse($result);

    // Rejection reason is available on the entity.
    $this->assertEquals('Test rejection reason', $conversion->getRejectionReason());
  }

  /**
   * Tests that rejected conversion is not saved.
   */
  public function testRejectedConversionNotSaved(): void {
    $user = $this->createAffiliateUser();

    // Add an event subscriber that rejects conversions.
    $listener = function ($event) {
      $event->reject('Not qualified');
    };

    $this->container->get('event_dispatcher')->addListener(
      ConversionPreCreateEvent::EVENT_NAME,
      $listener
    );

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
    ]);

    // Save returns FALSE when rejected.
    $result = $conversion->save();
    $this->assertFalse($result);
    $this->assertEquals('Not qualified', $conversion->getRejectionReason());

    // Verify no conversion was saved.
    $conversions = $this->entityTypeManager->getStorage('affiliate_conversion')->loadMultiple();
    $this->assertCount(0, $conversions);
  }

  /**
   * Tests that event has access to fully populated conversion data.
   */
  public function testEventHasFullConversionData(): void {
    $user = $this->createAffiliateUser();
    $capturedConversion = NULL;

    // Add an event subscriber to capture the conversion.
    $listener = function ($event) use (&$capturedConversion) {
      $capturedConversion = $event->getConversion();
    };

    $this->container->get('event_dispatcher')->addListener(
      ConversionPreCreateEvent::EVENT_NAME,
      $listener
    );

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
    ]);
    $conversion->save();

    // Verify the event had access to the populated data.
    $this->assertNotNull($capturedConversion);
    // Default commission (5.00) should be applied before event.
    $this->assertEquals(5.00, $capturedConversion->get('amount')->value);
    // Label should be generated before event.
    $this->assertNotEmpty($capturedConversion->get('label')->value);
  }

  /**
   * Tests that event is not dispatched for syncing entities.
   */
  public function testEventNotDispatchedWhenSyncing(): void {
    $user = $this->createAffiliateUser();
    $eventFired = FALSE;

    $listener = function ($event) use (&$eventFired) {
      $eventFired = TRUE;
    };

    $this->container->get('event_dispatcher')->addListener(
      ConversionPreCreateEvent::EVENT_NAME,
      $listener
    );

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
    ]);
    $conversion->setSyncing(TRUE);
    $conversion->save();

    $this->assertFalse($eventFired, 'Event should not fire for syncing entities');
  }

  /**
   * Tests that event is not dispatched for updates.
   */
  public function testEventNotDispatchedForUpdates(): void {
    $user = $this->createAffiliateUser();
    $eventCount = 0;

    $listener = function ($event) use (&$eventCount) {
      $eventCount++;
    };

    $this->container->get('event_dispatcher')->addListener(
      ConversionPreCreateEvent::EVENT_NAME,
      $listener
    );

    // Create conversion - should fire event.
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $user->id(),
    ]);
    $conversion->save();
    $this->assertEquals(1, $eventCount);

    // Update conversion - should NOT fire event.
    $conversion->setNotes('Updated');
    $conversion->save();
    $this->assertEquals(1, $eventCount, 'Event should not fire for updates');
  }

  /**
   * Tests that affiliate can use their own non-global campaign.
   */
  public function testAffiliateCanUseOwnNonGlobalCampaign(): void {
    $affiliate = $this->createAffiliateUser();

    // Create a non-global campaign owned by the affiliate.
    $campaign = $this->entityTypeManager->getStorage('affiliate_campaign')->create([
      'user_id' => $affiliate->id(),
      'name' => 'My Campaign',
      'is_global' => FALSE,
      'is_default' => FALSE,
    ]);
    $campaign->save();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $affiliate->id(),
      'campaign' => $campaign->id(),
    ]);
    $conversion->save();

    // Campaign should be kept since affiliate owns it.
    $this->assertEquals($campaign->id(), $conversion->getCampaignId());
  }

  /**
   * Tests that affiliate cannot use another affiliate's non-global campaign.
   */
  public function testAffiliateCannotUseOthersNonGlobalCampaign(): void {
    $affiliate1 = $this->createAffiliateUser();
    $affiliate2 = $this->createAffiliateUser();

    // Create a default global campaign.
    $defaultCampaign = $this->entityTypeManager->getStorage('affiliate_campaign')->create([
      'user_id' => 0,
      'name' => 'Default Campaign',
      'is_global' => TRUE,
      'is_default' => TRUE,
    ]);
    $defaultCampaign->save();

    // Create a non-global campaign owned by affiliate1.
    $privateCampaign = $this->entityTypeManager->getStorage('affiliate_campaign')->create([
      'user_id' => $affiliate1->id(),
      'name' => 'Private Campaign',
      'is_global' => FALSE,
      'is_default' => FALSE,
    ]);
    $privateCampaign->save();

    // Affiliate2 tries to use affiliate1's campaign.
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $affiliate2->id(),
      'campaign' => $privateCampaign->id(),
    ]);
    $conversion->save();

    // Campaign should be replaced with default since affiliate2 doesn't own it.
    $this->assertEquals($defaultCampaign->id(), $conversion->getCampaignId());
  }

  /**
   * Tests that any affiliate can use a global campaign.
   */
  public function testAnyAffiliateCanUseGlobalCampaign(): void {
    $affiliate1 = $this->createAffiliateUser();
    $affiliate2 = $this->createAffiliateUser();

    // Create a global campaign owned by affiliate1.
    $globalCampaign = $this->entityTypeManager->getStorage('affiliate_campaign')->create([
      'user_id' => $affiliate1->id(),
      'name' => 'Global Campaign',
      'is_global' => TRUE,
      'is_default' => FALSE,
    ]);
    $globalCampaign->save();

    // Affiliate2 uses the global campaign.
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $affiliate2->id(),
      'campaign' => $globalCampaign->id(),
    ]);
    $conversion->save();

    // Campaign should be kept since it's global.
    $this->assertEquals($globalCampaign->id(), $conversion->getCampaignId());
  }

  /**
   * Tests campaign validation doesn't apply to syncing entities.
   */
  public function testCampaignValidationSkippedWhenSyncing(): void {
    $affiliate1 = $this->createAffiliateUser();
    $affiliate2 = $this->createAffiliateUser();

    // Create a non-global campaign owned by affiliate1.
    $privateCampaign = $this->entityTypeManager->getStorage('affiliate_campaign')->create([
      'user_id' => $affiliate1->id(),
      'name' => 'Private Campaign',
      'is_global' => FALSE,
      'is_default' => FALSE,
    ]);
    $privateCampaign->save();

    // Syncing entity should bypass validation (e.g., config import).
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $affiliate2->id(),
      'campaign' => $privateCampaign->id(),
    ]);
    $conversion->setSyncing(TRUE);
    $conversion->save();

    // Campaign should be kept since syncing bypasses validation.
    $this->assertEquals($privateCampaign->id(), $conversion->getCampaignId());
  }

}
