<?php

namespace Drupal\ahasend;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Drupal\file\Entity\File;
use GuzzleHttp\RequestOptions;
use Psr\Log\LoggerInterface;

/**
 * Mail handler to send out an email message array to the AhaSend API.
 */
class AhaSendHandler {


  /**
   * Entity Type Manager for getting attachemnt files.
   */
  protected $entityTypeManager;

  /**
   * Stream wrapper for handling file URIs
   */
  protected $streamWrapper;

  /**
   * HTTP Client for sending the requests to AhaSend.
   */
  protected $httpClient;

  /**
   * Config factory
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Configuration object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * Logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a new AhaSendHandler object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   * @param StreamWrapperManagerInterface $streamWrapper
   */
  public function __construct(ConfigFactoryInterface $configFactory, LoggerInterface $logger, StreamWrapperManagerInterface $streamWrapper, EntityTypeManagerInterface $entityTypeManager, \GuzzleHttp\Client $httpClient) {
    $this->config = $configFactory->get('ahasend.settings');
    $this->configFactory = $configFactory;
    $this->logger = $logger;
    $this->streamWrapper = $streamWrapper;
    $this->entityTypeManager = $entityTypeManager;
    $this->httpClient = $httpClient;
  }

  /**
   * Connects to AhaSend API and sends out the email.
   *
   * @param array $params
   *   A message array.
   *
   * @return bool
   *   TRUE if the mail was successfully accepted by the API, FALSE otherwise.
   */
  public function sendMail(array $params) {
    $api_key = $this->config->get('api_key');
    $from_name_default = $this->config->get('from_name');
    if (!$from_name_default) {
      $from_name_default = $this->configFactory->get('system.site')->get('name');
    }
    $debug_mode = $this->config->get('debug_mode');
    try {
      $html = !empty($params['html']) && empty($params['text']) ? $params['html'] : NULL;
      $text = !empty($params['text']) ? $params['text'] : NULL;

      $from_addr = !empty($params['from']) ? $params['from'] : NULL;
      if (!$from_addr) {
        // default to site mail.
        $from_addr = $this->configFactory->get('system.site')->get('mail');
      }
      $from = $this->getContactObjects($from_addr);
      if (count($from) > 0) {
        $from = $from[0];
      } else {
        throw new \Exception("No from address available for sending the email.");
      }
      if (!$from['name']) {
        $from['name'] = $from_name_default;
      }

      $recipients = $this->getContactObjects($params['to']);

      $email = [
        'from' => $from,
        'recipients' => $recipients,
        'content' => [
          'subject' => $params['subject'],
          'text_body' => $text,
          'html_body' => $html,
        ],
      ];

      $cc = !empty($params['cc']) ? $params['cc'] : NULL;
      if ($cc) {
        $email['content']['headers']['cc'] = $cc;
      }
      $bcc = !empty($params['bcc']) ? $params['bcc'] : NULL;
      if ($bcc) {
        $email['content']['headers']['bcc'] = $bcc;
      }

      $reply_to = !empty($params['reply-to']) ? $params['reply-to'] : NULL;
      if ($reply_to) {
        $structure = $this->getContactObjects($reply_to);
        if (!empty($structure)) {
          $email['content']['reply_to'] = $structure[0];
        } else {
          // somehow the parser failed to parse the reply-to address. Set it as is.
          $email['content']['reply_to'] = [
            'email' => $reply_to,
          ];
        }
      }

      if (!empty($params['attachments'])) {
        $attachments = [];
        foreach ($params['attachments'] as $attachment) {
          $file = [];
          if (!empty($attachment['filepath'])) {
            if ($this->streamWrapper->isValidUri($attachment['filepath'])) {
              $entities = $this->entityTypeManager->getStorage('file')->loadByProperties(['uri' => $attachment['filepath']]);
              /** @var File $f */
              $f = array_pop($entities);
              if (!empty($f)) {
                $content = file_get_contents($f->getFileUri());
                $file = [
                  'data' => base64_encode($content),
                  'base64' => true,
                  'file_name' => $f->getFilename(),
                  'content_type' => $f->getMimeType(),
                ];
              }
            } else {
              $content = file_get_contents($attachment['filepath']);
              $file = [
                'data' => base64_encode($content),
                'base64' => true,
                'file_name' => basename($attachment['filepath']),
                'content_type' => mime_content_type($attachment['filepath']),
              ];
            }
          }
          elseif (!empty($attachment['filecontent'])) {
            $file = [
              'data' => $attachment['filecontent'],
              'file_name' => $attachment['filename'],
              'content_type' => $attachment['filemime'],
            ];
            // Attachment data must be base64 encoded if it's not a plaintext file.
            $file['base64'] = $attachment['filemime'] != 'text/plain';
          }
          if (!empty($file)) {
            if (!empty($attachment['content_id'])) {
              $file['content_id'] = $attachment['content_id'];
            }
            $attachments[] = $file;
          }
        }
        $email['content']['attachments'] = $attachments;
      }

      $response = $this->httpClient->post(
        'https://api.ahasend.com/v1/email/send',
        [
          'json' => $email,
          'headers' => [
            'X-Api-Key' => $api_key,
          ],
        ]
      );

      if ($response->getStatusCode() == 400) {
        throw new \Exception("Received invalid request error from AhaSend: " . $response->getBody());
      }
      elseif ($response->getStatusCode() == 403) {
        throw new \Exception("Received access denied error from AhaSend: " . $response->getBody());
      }
      elseif ($response->getStatusCode() == 500) {
        throw new \Exception("Received internal server error from AhaSend: " . $response->getBody());
      }
      elseif ($response->getStatusCode() != 201) {
        throw new \Exception("Received " . $response->getStatusCode() . " status code from AhaSend: " . $response->getBody());
      }

      // We're OK!
      $response_parsed = json_decode($response->getBody(), true);

      // Debug mode: log all messages.
      if ($debug_mode) {
        $this->logger->notice('Sent message from %from to %to. Response data was %response.',
          [
            '%from' => $params['from'],
            '%to' => $params['to'],
            '%response' => '<pre>' . print_r($response_parsed, 1) . '</pre>',
          ]
        );
      }
    } catch (\Exception $e) {
      // A general exception is thrown if the API
      // was unreachable or times out.
      $this->logger->error('Exception occurred while trying to send email from %from to %to via AhaSend. @code: @message',
        [
          '%from' => $from,
          '%to' => $params['to'],
          '@code' => $e->getCode(),
          '@message' => $e->getMessage(),
        ]
      );
      return FALSE;
    }

    return TRUE;
  }

  private function getContactObjects($address_line) {
    $parser = new \Mail_RFC822();
    $structure = $parser->parseAddressList($address_line, null, false);
    $contacts = [];
    foreach ($structure as $address) {
      $contacts[] = [
        'name' => $address->personal,
        'email' => $address->mailbox . '@' . $address->host
      ];
    }
    return $contacts;
  }
}
