<?php

namespace Drupal\ahasend\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\ahasend\AhaSendHandler;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure AhaSend settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The core mail manager service.
   *
   * @var \Drupal\ahasend\AhaSendHandler
   */
  protected $ahasendHandler;

  /**
   * Constructs a AhaSend settings form.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\ahasend\AhaSendHandler $ahasend_handler
   *   The core mail manager service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, protected TypedConfigManagerInterface $typedConfigManager, AhaSendHandler $ahasend_handler) {
    parent::__construct($config_factory, $this->typedConfigManager);

    $this->ahasendHandler = $ahasend_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('ahasend.mail_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ahasend_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ahasend.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // The user's AhaSend API key
    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AhaSend API Token'),
      '#default_value' => $this->config('ahasend.settings')->get('api_key'),
      '#description' => $this->t('The API key similar to 2T8EETSMISq8ylbWIuQ9GteNyfM4ayvxj59VipKgzaFDFSERMi2zxlk1V0uw9Ido, generated on the AhaSend Credentials page.'),
      '#required' => TRUE,
    ];
    $form['from_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('From name'),
      '#default_value' => $this->config('ahasend.settings')->get('from_name'),
      '#description' => $this->t('Set the email sender/from name that will be used if the sender address does not include a display name (that is, it\'s not in the "John Doe <john@example.com>" format). If the email does not include a display name and this option is not set, site name will be used as the default value for the sender name.'),
    ];
    $form['debug'] = [
      '#type' => 'details',
      '#title' => $this->t('Debugging'),
      '#open' => FALSE,
    ];
    $form['debug']['debug_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable debug mode'),
      '#default_value' => $this->config('ahasend.settings')->get('debug_mode'),
      '#description' => $this->t('Log all ')
    ];

    $form['test'] = [
      '#type' => 'details',
      '#title' => $this->t('Test email'),
      '#open' => FALSE,
    ];
    $form['test']['from_address'] = [
      '#type' => 'textfield',
      '#title' => $this->t('From'),
      '#default_value' => $this->config('system.site')->get('mail'),
      '#description' => $this->t('Enter a valid email address to send a test email. The domain for this address must be added to the AhaSend Domains page and have valid DNS configuration within AhaSend.'),
    ];
    $form['test']['recipient_address'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Recipient'),
      '#default_value' => '',
      '#description' => $this->t('Enter a valid email address to send a test email.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('ahasend.settings');
    $config
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('from_name', $form_state->getValue('from_name'))
      ->set('debug_mode', $form_state->getValue('debug_mode'))
      ->save();

    // If an address is submitted, send a test email message.
    $from = $form_state->getValue('from_address');
    $recipient = $form_state->getValue('recipient_address');

    if ($from && $recipient) {
      $message = $this->t('A test e-mail has been sent from @from to @recipient.  You may want to check the logs for any error messages.', [
        '@from' => $from,
        '@recipient' => $recipient,
      ]);

      $this->messenger()->addWarning($message);

      $payload = [
        'from' => $from,
        'to' => $recipient,
        'subject' => $this->t('Test message from AhaSend Drupal Integration'),
        'text' => $this->t('Just testing.'),
      ];

      $this->ahasendHandler->sendMail($payload);
    }

    parent::submitForm($form, $form_state);
  }

}
