<?php

namespace Drupal\ai_agents\PluginBase;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Extension\ExtensionPathResolver;
use Drupal\Core\File\FileSystem;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\Service\PromptJsonDecoder\PromptJsonDecoderInterface;
use Drupal\ai_agents\Output\StructuredResultData;
use Drupal\ai_agents\PluginInterfaces\AiAgentInterface;
use Drupal\ai_agents\Service\AgentHelper;
use Drupal\ai_agents\Task\Task;
use Drupal\ai_agents\Task\TaskInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Helper for worker agents.
 */
abstract class AiAgentBase extends PluginBase implements AiAgentInterface, ContainerFactoryPluginInterface {

  // All should be translatable.
  use StringTranslationTrait;

  /**
   * The ai provider.
   *
   * @var \Drupal\ai\AiProviderInterface|\Drupal\ai\Plugin\ProviderProxy
   */
  protected $aiProvider;

  /**
   * The state of the agent.
   *
   * @var array
   *   The state.
   */
  protected $state;

  /**
   * The model name.
   *
   * @var string
   */
  protected $modelName;

  /**
   * The ai configuration.
   *
   * @var array
   *   The ai configuration.
   */
  protected $aiConfiguration = [];

  /**
   * The original configurations.
   *
   * @var array
   *   The original configurations.
   */
  protected $originalConfigurations;

  /**
   * The task.
   *
   * @var \Drupal\ai_agents\Task\Task
   */
  protected Task $task;

  /**
   * Create directly (or give a Blueprint)
   *
   * @var bool
   */
  protected $createDirectly = FALSE;

  /**
   * Unique runner id.
   *
   * @var string
   */
  protected $runnerId = '';

  /**
   * The structured result data.
   *
   * @var \Drupal\ai_agents\Output\StructuredResultDataInterface
   */
  protected $structuredResultData;

  /**
   * The user interface interacting with the agent.
   *
   * @var string
   */
  protected $userInterface;

  /**
   * The extra tags to send on prompts.
   *
   * @var array
   */
  protected $extraTags = [];

  /**
   * The information to give back.
   *
   * @var string
   */
  protected $information = 'No information available.';

  /**
   * Constructor.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected AgentHelper $agentHelper,
    protected FileSystem $fileSystem,
    protected ConfigFactoryInterface $config,
    protected AccountProxyInterface $currentUser,
    protected ExtensionPathResolver $extensionPathResolver,
    protected PromptJsonDecoderInterface $promptJsonDecoder,
    protected AiProviderPluginManager $aiProviderPluginManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    // Set to defaults if exist.
    $default = $this->aiProviderPluginManager->getDefaultProviderForOperationType('chat_with_complex_json');
    if (!empty($default['provider_id']) && !empty($default['model_id'])) {
      $this->aiProvider = $this->aiProviderPluginManager->createInstance($default['provider_id']);
      $this->modelName = $default['model_id'];
    }
  }

  /**
   * {@inheritDoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('ai_agents.agent_helper'),
      $container->get('file_system'),
      $container->get('config.factory'),
      $container->get('current_user'),
      $container->get('extension.path.resolver'),
      $container->get('ai.prompt_json_decode'),
      $container->get('ai.provider'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function getId() {
    return $this->pluginDefinition['id'];
  }

  /**
   * {@inheritDoc}
   */
  public function getModuleName() {
    return $this->pluginDefinition['provider'];
  }

  /**
   * {@inheritDoc}
   */
  public function hasAccess() {
    if ($this->currentUser->id() == 1) {
      return AccessResult::allowed();
    }
    $config = $this->config->get('ai_agents.settings')->get($this->getId());
    $roles = [];
    foreach ($config['permissions'] as $permission => $set) {
      if ($set) {
        $roles[] = $permission;
      }
    }
    if (empty($roles)) {
      return AccessResult::allowed();
    }
    foreach ($this->currentUser->getRoles() as $role) {
      if (in_array($role, $roles)) {
        return AccessResult::allowed();
      }
    }
    return AccessResult::forbidden();
  }

  /**
   * {@inheritDoc}
   */
  public function getAiProvider() {
    return $this->aiProvider;
  }

  /**
   * {@inheritDoc}
   */
  public function setAiProvider($aiProvider) {
    $this->aiProvider = $aiProvider;
  }

  /**
   * {@inheritDoc}
   */
  public function getModelName() {
    return $this->modelName;
  }

  /**
   * {@inheritDoc}
   */
  public function setModelName($modelName) {
    $this->modelName = $modelName;
  }

  /**
   * {@inheritDoc}
   */
  public function getAiConfiguration() {
    return $this->aiConfiguration;
  }

  /**
   * {@inheritDoc}
   */
  public function setAiConfiguration(array $aiConfiguration) {
    $this->aiConfiguration = $aiConfiguration;
  }

  /**
   * {@inheritDoc}
   */
  public function getTask() {
    return $this->task;
  }

  /**
   * {@inheritDoc}
   */
  public function setTask(TaskInterface $task) {
    $this->task = $task;
  }

  /**
   * {@inheritDoc}
   */
  public function determineSolvability() {
    // Set up a new output result.
    $this->structuredResultData = new StructuredResultData();
    // Setup the helper runner.
    $this->agentHelper->setupRunner($this);
  }

  /**
   * {@inheritDoc}
   */
  public function solve() {
    // Set up a new output result.
    $this->structuredResultData = new StructuredResultData();
    // Setup the helper runner.
    $this->agentHelper->setupRunner($this);
  }

  /**
   * {@inheritDoc}
   */
  public function agentsCapabilities() {
    return [];
  }

  /**
   * {@inheritDoc}
   */
  public function inform() {
    return $this->information;
  }

  /**
   * {@inheritDoc}
   */
  public function getCreateDirectly() {
    return $this->createDirectly;
  }

  /**
   * {@inheritDoc}
   */
  public function setCreateDirectly($createDirectly) {
    $this->createDirectly = $createDirectly;
  }

  /**
   * {@inheritDoc}
   */
  public function getStructuredOutput(): StructuredResultData {
    return $this->structuredResultData;
  }

  /**
   * {@inheritDoc}
   */
  public function setUserInterface($userInterface, array $extraTags = []) {
    $this->userInterface = $userInterface;
    $this->extraTags = $extraTags;
  }

  /**
   * {@inheritDoc}
   */
  public function getExtraTags() {
    return $this->extraTags;
  }

  /**
   * Sets an original configuration for diffing.
   *
   * @param \Drupal\Core\Config\Entity\ConfigEntityInterface $config
   *   The name of the configuration dependency.
   */
  public function setOriginalConfigurations(ConfigEntityInterface $config) {
    $configDependencyName = $config->getConfigDependencyName();
    // Only set once, otherwise its not the original.
    if (!isset($this->originalConfigurations[$configDependencyName])) {
      $originalConfigurations = $this->config->get($configDependencyName)->get();
      if ($originalConfigurations) {
        $this->originalConfigurations[$configDependencyName] = $originalConfigurations;
      }
    }
  }

  /**
   * Get diff of configurations.
   *
   * @param \Drupal\Core\Config\Entity\ConfigEntityInterface $config
   *   The name of the configuration dependency.
   *
   * @return array
   *   The diff.
   */
  public function getDiffOfConfigurations(ConfigEntityInterface $config) {
    $diff = [];
    $configDependencyName = $config->getConfigDependencyName();
    if (isset($this->originalConfigurations[$configDependencyName])) {
      $originalConfigurations = $this->originalConfigurations[$configDependencyName];
      $currentConfigurations = $this->config->get($configDependencyName)->getRawData();
      $diff = array_diff_assoc($currentConfigurations, $originalConfigurations);
      $minus_diff = array_diff_assoc($originalConfigurations, $currentConfigurations);
    }
    return [
      'new' => $diff,
      'original' => $minus_diff,
    ];
  }

  /**
   * Get full context of the task.
   *
   * @param \Drupal\ai_agents\Task\TaskInterface $task
   *   The task.
   * @param bool $stripTags
   *   Strip tags.
   *
   * @return string
   *   The context.
   */
  public function getFullContextOfTask(TaskInterface $task, $stripTags = TRUE) {
    // Get the description and the comments.
    $context = "Task Title: " . $task->getTitle() . "\n";
    $context .= "Task Author: " . $task->getAuthorsUsername() . "\n";
    $context .= "Task Description:\n" . $task->getDescription();
    $context .= "\n--------------------------\n";
    if (count($task->getComments())) {
      $context .= "These are the following comments:\n";
      $context .= "--------------------------\n";
      $comments = $task->getComments();

      $i = 1;
      foreach ($comments as $comment) {
        $context .= "Comment order $i: \n";
        $context .= "Comment Author: " . $comment['role'] . "\n";
        $context .= "Comment:\n" . str_replace(["<br>", "<br />", "<br/>"], "\n", $comment['message']) . "\n\n";
        $i++;
      }
      $context .= "--------------------------\n";
    }
    return $stripTags ? strip_tags($context) : $context;
  }

  /**
   * Helper function to get description.
   *
   * @return string
   *   The description.
   */
  public function getDescription() {
    $capabilities = $this->agentsCapabilities();
    return $capabilities[key($capabilities)]['description'] ?? '';
  }

  /**
   * Get inputs as string.
   *
   * @return string
   *   The string.
   */
  public function getInputsAsString() {
    $capabilities = $this->agentsCapabilities();
    $outputString = "";
    foreach ($capabilities as $data) {
      foreach ($data['inputs'] as $input) {
        $outputString .= "Field name: $input[name] ($input[type]):\n";
        $outputString .= "Required:";
        $outputString .= isset($input['required']) && $input['required'] ? "Yes\n" : "No\n";
        $outputString .= "Description: $input[description]\n\n";
      }
    }
    return $outputString;
  }

  /**
   * Get outputs as string.
   *
   * @return string
   *   The string.
   */
  public function getOutputsAsString() {
    $capabilities = $this->agentsCapabilities();
    $outputString = "";
    foreach ($capabilities as $data) {
      foreach ($data['outputs'] as $title => $output) {
        $outputString .= "Field name: $title ($output[type]):\n";
        $outputString .= "Description: $output[description]\n";
      }
    }
    return $outputString;
  }

  /**
   * Helper function to run the whole process of one file.
   *
   * @param string $file
   *   The filename of the prompt.
   * @param array $userContext
   *   The user context.
   * @param string $module
   *   The module to fetch for.
   * @param string $subDirectory
   *   The subdirectory to look for the prompt in.
   * @param string $promptType
   *   The prompt type.
   * @param string $outputType
   *   The output type.
   *
   * @return array
   *   The response.
   */
  public function runSubAgent($file, array $userContext, $module, $subDirectory, $promptType = 'yaml', $outputType = 'json') {
    if ($promptType !== 'yaml') {
      throw new \Exception("The prompt type '$promptType' is not supported.");
    }
    if ($outputType !== 'json') {
      throw new \Exception("The output type '$outputType' is not supported.");
    }

    $prompt = '';
    switch ($promptType) {
      case 'yaml':
        $prompt = $this->actionYamlPrompts($file, $userContext, $module, $subDirectory);
        break;
    }

    $response = $this->runAiProvider($prompt['prompt'], [], TRUE, $file);

    switch ($outputType) {
      case 'json':
        $response = $this->promptJsonDecoder->decode($response);
        break;
    }
    return $response;
  }

  /**
   * Helper function to run the AI Provider.
   *
   * @param string $prompt
   *   The prompt.
   * @param array $images
   *   The images.
   * @param bool $strip_tags
   *   If strip_tags HTML.
   * @param string $promptFile
   *   The prompt file.
   *
   * @return \Drupal\ai\OperationType\Chat\ChatMessage
   *   The response.
   */
  public function runAiProvider($prompt, array $images = [], $strip_tags = TRUE, $promptFile = '') {
    $this->aiProvider->setChatSystemRole($prompt);
    $context = $this->getFullContextOfTask($this->task, $strip_tags);
    $message = new ChatMessage("user", $context, $images);
    $input = new ChatInput([
      $message,
    ]);
    $this->aiProvider->setConfiguration($this->aiConfiguration);
    $tags = [
      'ai_agents',
      'ai_agents_' . $this->getId(),
    ];
    if ($this->runnerId) {
      $tags[] = 'ai_agents_runner_' . $this->getRunnerId();
    }
    if ($promptFile) {
      $tags[] = 'ai_agents_prompt_' . explode('.', $promptFile)[0];
    }
    $response = $this->aiProvider->chat($input, $this->modelName, $tags);
    return $response->getNormalized();
  }

  /**
   * Builds action prompts.
   *
   * @param string $type
   *   The type of prompt to fetch.
   * @param array $userPrompts
   *   The user prompts to add to the action prompt.
   * @param string $module
   *   The module name to fetch prompt yamls from.
   * @param string $subDirectory
   *   The subdirectory to look for the prompt in.
   *
   * @return array|null
   *   The action prompt and the model to use.
   */
  public function actionYamlPrompts($type, array $userPrompts, $module, $subDirectory = '') {
    // Developers makes mistakes.
    $subDirectory = $subDirectory ? trim($subDirectory, '/') . '/' : '';
    $file = $this->extensionPathResolver->getPath('module', $module) . '/prompts/' . $subDirectory . basename($type, '.yml') . '.yml';
    if (!file_exists($file)) {
      throw new \Exception("The action prompt file '$file' does not exist.");
    }
    $data = Yaml::parse(file_get_contents($file));
    // Set introduction.
    $prompt = $data['prompt']['introduction'] . "\n\n";
    // Set formats to use.
    $invariable = count($data['prompt']['formats']) == 1 ? 'this format' : 'these formats';
    $structure = "";
    foreach ($data['prompt']['formats'] as $format) {
      $structure .= json_encode($format) . "\n";
    }
    $prompt .= "Do not include any explanations, only provide a RFC8259 compliant JSON response following $invariable without deviation:\n[$structure]\n";
    if (!empty($data['prompt']['possible_actions'])) {
      $prompt .= "This is the list of actions:\n";
      foreach ($data['prompt']['possible_actions'] as $action => $description) {
        $prompt .= "$action - $description\n";
      }
    }
    $prompt .= "\n";
    if (!empty($data['one_shot_learning_examples'])) {
      $prompt .= "Example response given\n";
      $prompt .= json_encode($data['prompt']['one_shot_learning_examples']) . "\n";
    }
    foreach ($userPrompts as $header => $userPrompt) {
      $prompt .= "\n\n-----------------------------------\n$header:\n$userPrompt\n-----------------------------------";
    }

    return [
      'prompt' => $prompt,
      'preferred_llm' => $data['preferred_llm'] ?? 'openai',
      'preferred_model' => $data['preferred_model'] ?? 'gpt-3.5-turbo',
    ];
  }

  /**
   * Helper function to create the runner id.
   *
   * @return string
   *   The runner id.
   */
  public function createRunnerId() {
    return $this->runnerId = $this->getId() . '_' . microtime();
  }

  /**
   * {@inheritDoc}
   */
  public function getRunnerId() {
    if (!$this->runnerId) {
      $this->createRunnerId();
    }
    return $this->runnerId;
  }

  /**
   * Helper function to set the runner id.
   *
   * @param string $runnerId
   *   The runner id.
   */
  public function setRunnerId($runnerId) {
    $this->runnerId = $runnerId;
  }

  /**
   * Set information.
   *
   * @param string $information
   *   The information.
   */
  public function setInformation($information) {
    $this->information = $information;
  }

  /**
   * Get information.
   *
   * @return string
   *   The information.
   */
  public function getInformation() {
    return $this->information;
  }

}
