<?php

namespace Drupal\ai_agents_extra\Plugin\Mcp;

use Drupal\ai_agents\PluginInterfaces\AiAgentInterface;
use Drupal\ai_agents\Task\Task;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\mcp\Attribute\Mcp;
use Drupal\mcp\Plugin\McpPluginBase;
use Drupal\mcp\ServerFeatures\Tool;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the mcp.
 */
#[Mcp(
  id: 'ai-agent-calling',
  name: new TranslatableMarkup('AI Agent Calling'),
  description: new TranslatableMarkup(
    'Provides ai agent calling functionality.'
  ),
)]
class AiAgentCalling extends McpPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The pluginManagerAiAgents service.
   *
   * @var \Drupal\ai_agents\PluginManager\AiAgentManager
   */
  protected $agentManager;

  /**
   * The pluginManagerAiProvider service.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $providerPlugin;

  /**
   * {@inheritDoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    $instance = new AiAgentCalling(
      $configuration,
      $plugin_id,
      $plugin_definition,
    );

    $instance->agentManager = $container->get(
      'plugin.manager.ai_agents'
    );
    $instance->providerPlugin = $container->get(
      'ai.provider'
    );

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getTools(): array {
    $tools = [];

    foreach (
      $this->agentManager->getDefinitions() as $agent_id => $agent_definition
    ) {
      /** @var \Drupal\ai_agents\PluginInterfaces\AiAgentInterface $instance */
      $instance = $this->agentManager->createInstance($agent_id);

      if ($instance->hasAccess() !== AccessResult::allowed()
        || !$instance->isAvailable()
      ) {
        continue;
      }
      $capabilities = $instance->agentsCapabilities();
      foreach ($capabilities as $capability_name => $capability) {
        $tools[] = new Tool(
          name: "$agent_id-$capability_name",
          description: $capability['description'],
          inputSchema: [
            'type'       => 'object',
            'properties' => [
              'prompt' => [
                'type'        => 'string',
                'description' => 'The prompt to enable drupal modules or a question.',
              ],
            ],
            'required'   => ['prompt'],
          ],
        );
      }
    }

    return $tools;
  }

  /**
   * {@inheritdoc}
   */
  public function executeTool(string $toolId, mixed $arguments): array {
    $prompt = $arguments['prompt'];
    if (empty($prompt)) {
      throw new \InvalidArgumentException(
        'Prompt is required.'
      );
    }

    [$agent_id, $capability_name] = explode('-', $toolId);
    /** @var \Drupal\ai_agents\PluginInterfaces\AiAgentInterface $agent */
    $agent = $this->agentManager->createInstance($agent_id);

    // Verify access and availability.
    if ($agent->hasAccess() !== AccessResult::allowed()
      || !$agent->isAvailable()
    ) {
      throw new \InvalidArgumentException(
        'Agent is not available or access is denied.'
      );
    }
    $capabilities = $agent->agentsCapabilities();
    if (!isset($capabilities[$capability_name])) {
      throw new \InvalidArgumentException(
        'Capability not found in agent.'
      );
    }

    $defaults = $this->providerPlugin->getDefaultProviderForOperationType(
      'chat_with_complex_json'
    );
    $task = new Task($prompt);
    $task->setComments($this->messages ?? []);
    $agent->setTask($task);
    $agent->setAiProvider(
      $this->providerPlugin->createInstance($defaults['provider_id'])
    );
    $agent->setModelName($defaults['model_id']);
    $agent->setAiConfiguration([]);
    $agent->setCreateDirectly(TRUE);

    $solvability = $agent->determineSolvability();
    switch ($solvability) {
      case AiAgentInterface::JOB_NEEDS_ANSWERS:
        $questions = $agent->askQuestion();

        return [
          [
            "type" => "text",
            "text" => implode("\n", $questions),
          ],
        ];

      case AiAgentInterface::JOB_NOT_SOLVABLE:
        return [
          [
            "type" => "text",
            "text" => 'Task is not solvable by this agent. Please rephrase.',
          ],
        ];

      case AiAgentInterface::JOB_SHOULD_ANSWER_QUESTION:
        return [
          [
            "type" => "text",
            "text" => $agent->answerQuestion(),
          ],
        ];

      case AiAgentInterface::JOB_INFORMS:
        return [
          [
            "type" => "text",
            "text" => $agent->inform(),
          ],
        ];

      case AiAgentInterface::JOB_SOLVABLE:
        $response = $agent->solve();
        if ($response instanceof TranslatableMarkup) {
          try {
            $response = $response->render();
          }
          // For some cases the response can't render because of missing vars.
          // In this case we just return the json serialized response.
          catch (\Exception $e) {
            // Ideally this should not happen. But for some cases it does.
            // As there is missing arguments in the translatable markup.
            $response = $response->getUntranslatedString();
          }
        }

        return [
          [
            "type" => "text",
            "text" => $response,
          ],
        ];

      default:
        return [
          [
            "type" => "text",
            "text" => 'Unknown solvability status',
          ],
        ];
    }
  }

}
