<?php

namespace Drupal\ai_agents_extra_tools\Plugin\AiFunctionCall;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation to add an ai assistant agent action.
 */
#[FunctionCall(
  id: 'ai_agent:add_ai_assistant_agent_action',
  function_name: 'ai_agent_add_ai_assistant_agent_action',
  name: 'Add AI Assistant Action',
  description: 'This tool can add an AI assistant agent action.',
  group: 'modification_tools',
  module_dependencies: ['ai_assistant_api'],
  context_definitions: [
    'id' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Assistant ID"),
      description: new TranslatableMarkup("The required AI Assistant ID."),
      required: TRUE,
      constraints: ['Regex' => '/^[a-zA-Z0-9_]+$/'],
    ),
    'action' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Action"),
      description: new TranslatableMarkup("The action that should be added to the assistant."),
      required: TRUE,
    ),
  ],
)]
class AddAiAssistantAgentAction extends FunctionCallBase implements ExecutableFunctionCallInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->currentUser = $container->get('current_user');
    return $instance;
  }

  /**
   * The information if something was saved or not.
   *
   * @var string
   */
  protected string $information = "";

  /**
   * {@inheritdoc}
   */
  public function execute() {
    // Collect the context values.
    $id = $this->getContextValue('id');
    $action = $this->getContextValue('action');

    // Make sure that the user has the right permissions.
    if (!$this->currentUser->hasPermission('administer ai assistant')) {
      throw new \Exception('The current user does not have the right permissions to administer the assistants.');
    }

    $storage = $this->entityTypeManager->getStorage('ai_assistant');

    $entity = NULL;
    try {
      /** @var \Drupal\ai_agents\Entity\AiAssistant $entity */
      $entity = $storage->load($id);
    }
    catch (\Exception $e) {
      $this->information = $this->t('The assistant (%id) could not be loaded.', [
        '%id' => $id,
      ]);
      return;
    }
    // Add the assistant action.
    $actions = $entity->get('actions_enabled');
    $actions['agent_action']['agent_ids'][$action] = $action;
    $entity->set('actions_enabled', $actions);
    try {
      $entity->save();
      $this->information = $this->t('The action (%action) was added to the assistant (%id).', [
        '%action' => $action,
        '%id' => $id,
      ]);
    }
    catch (\Exception $e) {
      $this->information = $this->t('The action (%action) could not be added to the assistant (%id).', [
        '%action' => $action,
        '%id' => $id,
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getReadableOutput(): string {
    return $this->information;
  }

}
