<?php

namespace Drupal\ai_agents_extra_tools\Plugin\AiFunctionCall;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Drupal\ai_agents\PluginInterfaces\AiAgentContextInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the adding of user roles.
 */
#[FunctionCall(
  id: 'ai_agent:add_user_role',
  function_name: 'ai_agent_add_user_role',
  name: 'Add user role to user',
  description: 'This sets a new user role to a user.',
  group: 'information_tools',
  context_definitions: [
    'user_id' => new ContextDefinition(
      data_type: 'string',
      label: 'User ID',
      description: 'The user id to add some user roles for.',
      required: TRUE,
    ),
    'user_role' => new ContextDefinition(
      data_type: 'string',
      label: 'User Role',
      description: 'The user role to add to the user.',
      required: TRUE,
    ),
  ],
)]
class AddUserRoleToUser extends FunctionCallBase implements ExecutableFunctionCallInterface, AiAgentContextInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->currentUser = $container->get('current_user');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    // Collect the context values.
    $user_id = $this->getContextValue('user_id');
    $user_role = $this->getContextValue('user_role');

    // Check so the user has permissions to add roles.
    if (!$this->currentUser->hasPermission('administer users')) {
      throw new \Exception('You do not have permission to add user roles.');
    }

    // Load the user entity.
    try {
      /** @var \Drupal\user\Entity\User $user */
      $user = $this->entityTypeManager->getStorage('user')->load($user_id);
    }
    catch (\Exception $e) {
      throw new \Exception('User not found.');
    }
    if (!$user) {
      throw new \Exception('User not found.');
    }
    // Check if the user role exists.
    $roles = $this->entityTypeManager->getStorage('user_role')->loadByProperties(['id' => $user_role]);
    if (empty($roles)) {
      throw new \Exception('User role not found.');
    }
    // Add the user role to the user.
    $user->addRole($user_role);
    // Save the user entity.
    try {
      $user->save();
    }
    catch (\Exception $e) {
      throw new \Exception('User role could not be added.');
    }
    $this->setOutput("User role $user_role added to user $user_id.");
  }

}
