<?php

namespace Drupal\ai_agents_extra_tools\Plugin\AiFunctionCall;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Drupal\ai_agents\Plugin\AiFunctionCall\Children\CreateAgentConfigToolsEnabled;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the create agents function.
 */
#[FunctionCall(
  id: 'ai_agent:create_agent_config',
  function_name: 'ai_agent_create_agent_config',
  name: 'Create Agent Config',
  description: 'This tool can save an agent config.',
  group: 'modification_tools',
  context_definitions: [
    'id' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("ID"),
      description: new TranslatableMarkup("The required agent config ID."),
      required: TRUE,
      constraints: ['Regex' => '/^[a-zA-Z0-9_]+$/'],
    ),
    'label' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Label"),
      description: new TranslatableMarkup("The required agent config label."),
      required: TRUE,
    ),
    'description' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Description"),
      description: new TranslatableMarkup("The required agent config description."),
      required: TRUE,
    ),
    'system_prompt' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Agent Instructions"),
      description: new TranslatableMarkup("The required agent instructions."),
      required: TRUE,
    ),
    'secured_system_prompt' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Secured System Prompt"),
      description: new TranslatableMarkup("The secured system prompt with tokens."),
      required: FALSE,
    ),
    'tools' => new ContextDefinition(
      data_type: 'map',
      label: new TranslatableMarkup("Entity Array"),
      description: new TranslatableMarkup("The entity array to seed the entity with."),
      required: TRUE,
      multiple: TRUE,
      constraints: [
        'ComplexToolItems' => CreateAgentConfigToolsEnabled::class,
      ],
    ),
    'orchestration_agent' => new ContextDefinition(
      data_type: 'boolean',
      label: new TranslatableMarkup("Orchestration Agent"),
      description: new TranslatableMarkup("If this is an orchestration agent."),
      required: TRUE,
    ),
    'triage_agent' => new ContextDefinition(
      data_type: 'boolean',
      label: new TranslatableMarkup("Triage Agent"),
      description: new TranslatableMarkup("If this is a triage agent."),
      required: TRUE,
    ),
    'max_loops' => new ContextDefinition(
      data_type: 'integer',
      label: new TranslatableMarkup("Max Loops"),
      description: new TranslatableMarkup("The maximum number of loops."),
      required: TRUE,
      default_value: 3,
    ),
  ],
)]
class CreateAgentConfig extends FunctionCallBase implements ExecutableFunctionCallInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->currentUser = $container->get('current_user');
    return $instance;
  }

  /**
   * The information if something was saved or not.
   *
   * @var string
   */
  protected string $information = "";

  /**
   * {@inheritdoc}
   */
  public function execute() {
    // Collect the context values.
    $id = $this->getContextValue('id');
    $label = $this->getContextValue('label');
    $description = $this->getContextValue('description');
    $system_prompt = $this->getContextValue('system_prompt');
    $secured_system_prompt = $this->getContextValue('secured_system_prompt');
    $tools = $this->getContextValue('tools');
    $orchestration_agent = $this->getContextValue('orchestration_agent');
    $triage_agent = $this->getContextValue('triage_agent');
    $max_loops = $this->getContextValue('max_loops');

    // Set default secured system prompt if empty.
    if (empty($secured_system_prompt)) {
      $secured_system_prompt = '[agent_instructions]';
    }

    // Make sure that the user has the right permissions.
    if (!$this->currentUser->hasPermission('administer ai agent')) {
      throw new \Exception('The current user does not have the right permissions to create agents.');
    }

    $storage = $this->entityTypeManager->getStorage('ai_agent');

    $listed_tools = [];
    foreach ($tools as $tool) {
      $listed_tools[$tool['key']] = $tool['value'];
    }
    try {
      $entity = $storage->create([
        'id' => $id,
        'label' => $label,
        'description' => $description,
        'system_prompt' => $system_prompt,
        'secured_system_prompt' => $secured_system_prompt,
        'tools' => $listed_tools,
        'orchestration_agent' => $orchestration_agent,
        'triage_agent' => $triage_agent,
        'max_loops' => $max_loops,
        'default_information_tools' => '',
        'tool_usage_limits' => [],
      ]);
      $entity->save();
      $this->information = $this->t('The agent config %label (%id) has been saved.', [
        '%label' => $label,
        '%id' => $id,
      ]);
    }
    catch (\Exception $e) {
      $this->information = $this->t('The agent config %label (%id) could not be saved.', [
        '%label' => $label,
        '%id' => $id,
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getReadableOutput(): string {
    return $this->information;
  }

}
