<?php

namespace Drupal\ai_agents_extra_tools\Plugin\AiFunctionCall;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the delete content type function.
 */
#[FunctionCall(
  id: 'ai_agent:delete_content_type',
  function_name: 'delete_content_type',
  name: 'Delete Content Type',
  description: 'This function is used to delete a content type.',
  group: 'modification_tools',
  module_dependencies: ['node'],
  context_definitions: [
    'data_name' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Data Name"),
      description: new TranslatableMarkup("The required node type data name to delete. Allows for underscore and alphanumeric characters"),
      required: TRUE
    ),
  ],
)]
class DeleteContentType extends FunctionCallBase implements ExecutableFunctionCallInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->currentUser = $container->get('current_user');
    return $instance;
  }

  /**
   * The information of what was created.
   *
   * @var array
   */
  protected array $information = [];

  /**
   * {@inheritdoc}
   */
  public function execute() {
    $data_name = $this->getContextValue('data_name');
    if (empty($data_name)) {
      throw new \Exception('The data name is required.');
    }
    // Make sure that the user has the right permissions.
    if (!$this->currentUser->hasPermission('administer content types')) {
      throw new \Exception('The current user does not have the right permissions to run this tool.');
    }
    $node_type_storage = $this->entityTypeManager->getStorage('node_type');
    // Check so the node type exists.
    /** @var \Drupal\node\NodeTypeInterface $node_type */
    $node_type = $node_type_storage->load($data_name);
    if (!$node_type) {
      throw new \Exception('The node type does not exist and can not be deleted.');
    }
    $node_type->delete();

    $this->information[] = $this->t('The node type %node_type (%data_name) has been deleted.');
  }

  /**
   * {@inheritdoc}
   */
  public function getReadableOutput(): string {
    return implode("\n", $this->information);
  }

}
