<?php

namespace Drupal\ai_agents_extra_tools\Plugin\AiFunctionCall;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Drupal\ai_agents\PluginInterfaces\AiAgentContextInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the Download file from url function.
 */
#[FunctionCall(
  id: 'ai_agent:download_file_from_url',
  function_name: 'ai_agents_download_file_from_url',
  name: 'Download file from url',
  description: 'This method will download a file url and create a file entity if wanted from it.',
  group: 'modification_tools',
  module_dependencies: ['file'],
  context_definitions: [
    'file_url' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("File url"),
      description: new TranslatableMarkup("The full url to the file to scrape."),
      required: TRUE
    ),
    'create_file_entity' => new ContextDefinition(
      data_type: 'boolean',
      label: new TranslatableMarkup("Create file entity"),
      description: new TranslatableMarkup("If a file entity should be created."),
      required: FALSE,
      default_value: TRUE
    ),
    'path' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Path"),
      description: new TranslatableMarkup("The path to put the files."),
      required: FALSE,
      default_value: 'public://agent_downloads'
    ),
    'file_name' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("File name"),
      description: new TranslatableMarkup("The name of the file to save. Takes the file name from the url if not set."),
      required: FALSE,
      default_value: ''
    ),
    'replace_or_rename' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Replace or rename"),
      description: new TranslatableMarkup("Replace the file if it exists or rename it."),
      required: FALSE,
      default_value: 'rename',
      constraints: [
        'AllowedValues' => ['replace', 'rename'],
      ],
    ),
  ],
)]
class DownloadFileFromUrl extends FunctionCallBase implements ExecutableFunctionCallInterface, AiAgentContextInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->fileSystem = $container->get('file_system');
    return $instance;
  }

  /**
   * The answer.
   *
   * @var string
   */
  public string $answer = '';

  /**
   * {@inheritdoc}
   */
  public function execute() {
    $file_url = $this->getContextValue('file_url');
    $create_file_entity = $this->getContextValue('create_file_entity');
    $path = $this->getContextValue('path');
    $file_name = $this->getContextValue('file_name');
    $replace_or_rename = $this->getContextValue('replace_or_rename');
    if (!$file_url) {
      $this->answer = "You need to provide a file url.";
      return;
    }
    // First download to a temp file.
    try {
      $temp_file = $this->download($file_url);
    }
    catch (\Exception $e) {
      $this->answer = "Could not download the file.";
      return;
    }
    // Get the file name.
    if (!$file_name) {
      $name = basename($file_url);
      // Remove query string.
      $name = explode('?', $name)[0];
      $file_name = $name;
    }
    // Prepare the path and make sure it exists and is readable.
    try {
      $this->fileSystem->prepareDirectory($path, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);
    }
    catch (\Exception $e) {
      $this->answer = "Could not create the directory.";
      return;
    }
    // Copy the file.
    $destination = $path . '/' . $file_name;
    $file_mode = $replace_or_rename == 'replace' ? FileExists::Replace : FileExists::Rename;
    $destination = $this->fileSystem->copy($temp_file, $destination, $file_mode);

    // Create a file entity if wanted.
    if ($create_file_entity) {
      $file = $this->entityTypeManager->getStorage('file')->create([
        'uri' => $destination,
      ]);
      $file->save();
      $this->answer = "File entity created at $destination with file id " . $file->id();
    }
    else {
      $this->answer = "File copied to $destination.";
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getReadableOutput(): string {
    return $this->answer;
  }

  /**
   * Download a temp file.
   *
   * @param string $file_url
   *   The file url.
   *
   * @return string
   *   The temp file.
   */
  public function download($file_url) {
    $temp_file = tempnam(sys_get_temp_dir(), 'ai_agents');
    $fp = fopen($temp_file, 'w');
    $ch = curl_init($file_url);
    curl_setopt($ch, CURLOPT_FILE, $fp);
    curl_setopt($ch, CURLOPT_HEADER, 0);
    curl_exec($ch);
    curl_close($ch);
    fclose($fp);
    return $temp_file;
  }

}
