<?php

namespace Drupal\ai_agents_extra_tools\Plugin\AiFunctionCall;

use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Exception\AiToolsValidationException;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Drupal\ai_agents\PluginInterfaces\AiAgentContextInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the describe tool code.
 */
#[FunctionCall(
  id: 'ai_agent:get_plugin_attribute_data',
  function_name: 'ai_agent_get_plugin_attribute_data',
  name: 'Get Plugin Attribute Data',
  description: 'You can use this to load any plugin manager and get the meta data for it.',
  group: 'information_tools',
  context_definitions: [
    'permission' => new ContextDefinition(
      data_type: 'string',
      label: 'Permission',
      description: 'The permission that is needed to list this.',
      required: TRUE,
      default_value: 'administer site configuration',
      // We set a fixed value, that has to be overridden by the agent.
      constraints: [
        'FixedValue' => 'administer site configuration',
      ],
    ),
    'plugin_service_name' => new ContextDefinition(
      data_type: 'string',
      label: 'Plugin Service Name',
      description: 'The service data name of the plugin manager to load.',
      required: TRUE,
      // We set a fixed value, that has to be overridden by the agent.
      constraints: [
        'FixedValue' => 'ai.provider',
      ],
    ),
    'plugin_item_id' => new ContextDefinition(
      data_type: 'string',
      label: 'Plugin Item ID',
      description: 'If you only need information on a specific plugin item, you can use this.',
      required: FALSE,
    ),
    'plugin_attributes' => new ContextDefinition(
      data_type: 'list',
      label: 'Plugin Attributes',
      description: 'The attributes of the plugin manager to load.',
      required: FALSE,
      multiple: TRUE,
    ),
  ],
)]
class GetPluginAttributeData extends FunctionCallBase implements ExecutableFunctionCallInterface, AiAgentContextInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The Drupal container.
   *
   * @var \Drupal\Core\DependencyInjection\ContainerInterface
   */
  protected ContainerInterface $container;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->currentUser = $container->get('current_user');
    $instance->container = $container;
    return $instance;
  }

  /**
   * The listing of data.
   *
   * @var string
   */
  protected string $listing = '';

  /**
   * {@inheritdoc}
   */
  public function execute() {
    $plugin_service_name = $this->getContextValue('plugin_service_name');
    $plugin_item_id = $this->getContextValue('plugin_item_id');
    $plugin_attributes = $this->getContextValue('plugin_attributes');
    $permission = $this->getContextValue('permission');
    // Check so the user has the permission to list this.
    if (!$this->currentUser->hasPermission($permission)) {
      throw new AiToolsValidationException('The user does not have the permission needed to use this tool.');
    }
    // Check so the service really exists.
    if (!$this->container->has($plugin_service_name)) {
      throw new AiToolsValidationException(
        sprintf('The service "%s" does not exist.', $plugin_service_name),
      );
    }

    // We load the service.
    // Since there are services with magic methods where just loading the
    // service might cause issues on construct or destruct, this can be a
    // problem with prompt injections, that's why we use FixedValue to make
    // sure this is set by the administrator.
    $plugin_manager = NULL;
    try {
      $plugin_manager = $this->container->get($plugin_service_name);
      if (!($plugin_manager instanceof DefaultPluginManager)) {
        throw new AiToolsValidationException(
          sprintf('The service "%s" is not a valid plugin manager.', $plugin_service_name),
        );
      }
    }
    catch (\ReflectionException $e) {
      throw new AiToolsValidationException(
        sprintf('The service "%s" is not a valid plugin manager.', $plugin_service_name),
      );
    }

    // Get all definitions.
    $list = [];
    foreach ($plugin_manager->getDefinitions() as $id => $definition) {
      if ($plugin_item_id && $id !== $plugin_item_id) {
        continue;
      }
      if ($plugin_attributes) {
        $list[$id] = [];
        foreach ($plugin_attributes as $attribute) {
          if (isset($definition[$attribute])) {
            $list[$id][$attribute] = $definition[$attribute];
          }
          else {
            $list[$id][$attribute] = NULL;
          }
        }
      }
      else {
        $list[$id] = $definition;
      }
    }
    if (!empty($list)) {
      // Make sure that translatable strings, becomes strings.
      foreach ($list as $id => $definition) {
        foreach ($definition as $key => $value) {
          if ($value instanceof TranslatableMarkup) {
            $list[$id][$key] = (string) $value;
          }
        }
      }
      $this->listing = Yaml::dump($list, 4, 10);
    }
    else {
      if ($plugin_item_id) {
        $this->listing = sprintf('No plugin item found with ID "%s".', $plugin_item_id);
      }
      else {
        $this->listing = 'No plugin items found.';
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getReadableOutput(): string {
    return $this->listing;
  }

}
