<?php

namespace Drupal\ai_agents_extra_tools\Plugin\AiFunctionCall;

use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Drupal\ai_agents\PluginInterfaces\AiAgentContextInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the describe tool code.
 */
#[FunctionCall(
  id: 'ai_agent:get_tool_execute_code',
  function_name: 'ai_agent_get_tool_execute_code',
  name: 'Get Tool Execute Code',
  description: 'This tool can get the execute code of a tool.',
  group: 'information_tools',
  context_definitions: [
    'tool_id' => new ContextDefinition(
      data_type: 'string',
      label: 'Tool ID',
      description: 'The required tool ID to read the execute code.',
      required: TRUE,
    ),
  ],
)]
class GetToolExecuteCode extends FunctionCallBase implements ExecutableFunctionCallInterface, AiAgentContextInterface {

  /**
   * The Function Call Plugin Manager.
   *
   * @var \Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager
   */
  protected FunctionCallPluginManager $functionCallPluginManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->functionCallPluginManager = $container->get('plugin.manager.ai.function_calls');
    $instance->currentUser = $container->get('current_user');
    return $instance;
  }

  /**
   * The code.
   *
   * @var string
   */
  protected string $code = '';

  /**
   * {@inheritdoc}
   */
  public function execute() {
    // Collect the context values.
    $tool_id = $this->getContextValue('tool_id');
    // Just set the highest permission possible here.
    if (!$this->currentUser->hasPermission('administer permissions')) {
      $this->code = 'You do not have the right permissions to view this code.';
      return;
    }
    $tool = $this->functionCallPluginManager->getDefinition($tool_id);
    $class = $tool['class'];
    // Create reflection class.
    $reflector = new \ReflectionMethod($class, 'execute');
    // Output the execute code.
    $file = new \SplFileObject($reflector->getFileName());
    $file->seek($reflector->getStartLine() - 1);

    while ($file->key() < $reflector->getEndLine()) {
      $this->code .= $file->current();
      $file->next();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getReadableOutput(): string {
    return $this->code;
  }

}
