<?php

namespace Drupal\ai_agents_extra_tools\Plugin\AiFunctionCall;

use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Drupal\ai_agents\PluginInterfaces\AiAgentContextInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the list tools function.
 */
#[FunctionCall(
  id: 'ai_agent:list_ai_tools',
  function_name: 'ai_agent_list_ai_tools',
  name: 'List Tools',
  description: 'This method can list all the tools for function calling.',
  group: 'information_tools',
  context_definitions: [
    'function_group' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Function Group"),
      description: new TranslatableMarkup("Options filter for a specific function group."),
      required: FALSE,
    ),
  ],
)]
class ListAiTools extends FunctionCallBase implements ExecutableFunctionCallInterface, AiAgentContextInterface {

  /**
   * The Function Call Plugin Manager.
   *
   * @var \Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager
   */
  protected FunctionCallPluginManager $functionCallPluginManager;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->functionCallPluginManager = $container->get('plugin.manager.ai.function_calls');
    return $instance;
  }

  /**
   * The list.
   *
   * @var array
   */
  protected array $list = [];

  /**
   * {@inheritdoc}
   */
  public function execute() {
    // Collect the context values.
    $function_group = $this->getContextValue('function_group');
    $this->list = [];
    foreach ($this->functionCallPluginManager->getDefinitions() as $definition) {
      if ($function_group && $definition['group'] !== $function_group) {
        continue;
      }
      $this->list[$definition['id']] = [
        'id' => $definition['id'],
        'name' => $definition['name'],
        'description' => $definition['description'],
        'group' => $definition['group'],
        'module_dependencies' => $definition['module_dependencies'] ?? [],
      ];

      foreach ($definition['context_definitions'] as $context_name => $context_definition) {
        $this->list[$definition['id']]['context_definitions'][$context_name] = [
          'data_type' => $context_definition->getDataType(),
          'label' => (string) $context_definition->getLabel(),
          'description' => (string) $context_definition->getDescription(),
          'required' => $context_definition->isRequired(),
        ];
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getReadableOutput(): string {
    return Yaml::dump($this->list, 10, 2);
  }

}
