<?php

namespace Drupal\ai_agents_extra_tools\Plugin\AiFunctionCall;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Drupal\ai_agents\PluginInterfaces\AiAgentContextInterface;
use League\HTMLToMarkdown\HtmlConverter;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the field validation results.
 */
#[FunctionCall(
  id: 'ai_agents:wikipedia',
  function_name: 'ai_agents_wikipedia',
  name: 'Wikipedia',
  description: 'This function searches and returns the first result from Wikipedia for a search query.',
  group: 'information_tools',
  context_definitions: [
    'search_query' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Search Query"),
      description: new TranslatableMarkup("The search query to search Wikipedia for."),
      required: TRUE,
    ),
  ],
)]
class Wikipedia extends FunctionCallBase implements ExecutableFunctionCallInterface, AiAgentContextInterface {

  /**
   * The HTTP client.
   *
   * @var \Drupal\Core\Http\ClientInterface
   */
  protected $httpClient;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->httpClient = $container->get('http_client');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    // Collect the context values.
    $search_query = $this->getContextValue('search_query');

    // Search Wikipedia for the query.
    $title = $this->searchWikipedia($search_query);
    if (empty($title)) {
      $output = "No results found for the search query: $search_query";
    }
    else {
      // Do some waiting to not overwhelm the Wikipedia API.
      sleep(1);
      // Get the article from Wikipedia.
      $article = $this->getArticle($title);
      // The link to the article.
      $link = "https://en.wikipedia.org/wiki/" . urlencode($title);
      // Format the output.
      $output = "Here is the article from Wikipedia for $title:\n\n";
      $output .= $article;
      $output .= "\n\n---\n\n";
      $output .= "You can read the full article [here]($link).";
    }

    $this->setOutput($output);
  }

  /**
   * Get the article.
   *
   * @param string $title
   *   The title of the article to get.
   *
   * @return string
   *   The article.
   */
  public function getArticle(string $title): string {
    $html_url = "https://en.wikipedia.org/w/rest.php/v1/page/$title/html";
    $html = $this->httpClient->get($html_url)->getBody()->getContents();
    // Remove everything in style tags.
    $html = preg_replace('/<style\b[^>]*>(.*?)<\/style>/is', '', $html);
    $converter = new HtmlConverter();
    $converter->getConfig()->setOption('strip_tags', TRUE);

    $markdown = strip_tags($converter->convert($html));

    return $markdown;
  }

  /**
   * Search Wikipedia.
   *
   * @param string $search_query
   *   The search query to search Wikipedia for.
   *
   * @return string
   *   The search result.
   */
  public function searchWikipedia(string $search_query): string {
    // Url encode the search query.
    $encoded_search_query = urlencode($search_query);
    $url = "https://en.wikipedia.org/w/api.php?action=query&list=search&srsearch=$encoded_search_query&format=json";
    $content = $this->httpClient->get($url)->getBody()->getContents();

    $data = Json::decode($content);

    // Return the search result.
    return $data['query']['search'][0]['title'] ?? "";
  }

}
