<?php

namespace Drupal\ai_agents\Plugin\AiFunctionCall;

use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Drupal\ai_agents\PluginInterfaces\AiAgentContextInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the describe config function.
 */
#[FunctionCall(
  id: 'ai_agent:get_config_schema',
  function_name: 'ai_agent_get_config_schema',
  name: 'Get Config Schema',
  description: 'This gets the Drupal configuration schema for a single schema.',
  group: 'information_tools',
  context_definitions: [
    'schema_id' => new ContextDefinition(
      data_type: 'string',
      label: 'Entity Type',
      description: 'The entity type to get the configuration schema for.',
      required: TRUE,
    ),
  ],
)]
class GetConfigSchema extends FunctionCallBase implements ExecutableFunctionCallInterface, AiAgentContextInterface {

  /**
   * The config typed data manager.
   *
   * @var \Drupal\Core\Config\TypedConfigManagerInterface
   */
  protected TypedConfigManagerInterface $typedConfigManager;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->typedConfigManager = $container->get('config.typed');
    return $instance;
  }

  /**
   * The schema.
   *
   * @var array
   */
  protected array $schema = [];

  /**
   * {@inheritdoc}
   */
  public function execute() {
    // Collect the context values.
    $schema_id = $this->getContextValue('schema_id');
    // @todo Permissions?
    $this->schema = $this->typedConfigManager->getDefinition($schema_id);
  }

  /**
   * {@inheritdoc}
   */
  public function getReadableOutput(): string {
    return Yaml::dump($this->schema, 10, 2);
  }

}
