<?php

namespace Drupal\ai_agents\Plugin\AiFunctionCall;

use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Drupal\ai_agents\Exception\AgentPermissionsException;
use Drupal\ai_agents\PluginInterfaces\AiAgentContextInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the list bundles function.
 */
#[FunctionCall(
  id: 'ai_agent:list_field_types',
  function_name: 'ai_agents_list_field_types',
  name: 'List Field Types',
  description: 'This method list all field types available on the website.',
  group: 'information_tools',
  context_definitions: [
    'field_type' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Field Type"),
      description: new TranslatableMarkup("If you just want information about one field type."),
      required: FALSE,
    ),
    'simple_representation' => new ContextDefinition(
      data_type: 'boolean',
      label: new TranslatableMarkup("Simple Representation"),
      description: new TranslatableMarkup("If you want a simple representation of the field type."),
      required: FALSE,
    ),
  ],
)]
class ListFieldTypes extends FunctionCallBase implements ExecutableFunctionCallInterface, AiAgentContextInterface {

  /**
   * Get the current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The field type plugin manager.
   *
   * @var \Drupal\Core\Field\FieldTypePluginManagerInterface
   */
  protected $fieldTypeManager;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->currentUser = $container->get('current_user');
    $instance->fieldTypeManager = $container->get('plugin.manager.field.field_type');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    // Make sure the user can create or edit the field config.
    if (!$this->currentUser->hasPermission('administer field config')) {
      throw new AgentPermissionsException('You do not have permission to create or edit field configs.');
    }

    // Get the field type from the context.
    $field_type_id = $this->getContextValue('field_type');
    $simple_representation = $this->getContextValue('simple_representation');

    // Get all field types.
    $field_types = $this->fieldTypeManager->getDefinitions();
    $field_types_list = [];
    foreach ($field_types as $id => $field_type) {
      if ($field_type_id && $field_type_id !== $id) {
        continue;
      }
      if ($simple_representation) {
        $field_types_list[$id]['id'] = (string) $id;
        $field_types_list[$id]['label'] = (string) $field_type['label'];
        $field_types_list[$id]['description'] = (string) $field_type['description'];
        continue;
      }
      foreach ($field_type as $key => $value) {
        $field_types_list[$id][$key] = (string) $value;
      }
    }
    $this->setOutput(Yaml::dump($field_types_list, 10, 2, Yaml::DUMP_MULTI_LINE_LITERAL_BLOCK));
  }

}
