<?php

namespace Drupal\ai_agents\Plugin\AiFunctionCall;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the create bundle type function.
 */
#[FunctionCall(
  id: 'ai_agent:modify_bundle_type',
  function_name: 'modify_bundle_type',
  name: 'Modify Bundle Type',
  description: 'This function is used to modify a bundle type.',
  group: 'modification_tools',
  context_definitions: [
    'entity_type' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Entity Type"),
      description: new TranslatableMarkup("The data name of the entity type you are trying to to modify a bundle for."),
      required: TRUE,
    ),
    'id' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("ID"),
      description: new TranslatableMarkup("The data name of the bundle you are trying to create or edit. So vid for vocabularies, type for node types, etc."),
      required: TRUE,
      constraints: ['Regex' => '/^[a-zA-Z0-9_]+$/'],
    ),
    'label' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Label/Title/Name"),
      description: new TranslatableMarkup("The required label for the bundle type, can be omitted during editing if you do not want to change it."),
      required: FALSE,
    ),
  ],
)]
class ModifyBundleType extends FunctionCallBase implements ExecutableFunctionCallInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->currentUser = $container->get('current_user');
    return $instance;
  }

  /**
   * The information of what was created.
   *
   * @var string
   */
  protected string $information = '';

  /**
   * {@inheritdoc}
   */
  public function execute() {
    // Collect the context values.
    $entity_type = $this->getContextValue('entity_type');
    $id = $this->getContextValue('id');
    $label = $this->getContextValue('label');

    // Get the definition.
    $definition = $this->entityTypeManager->getDefinition($entity_type);
    // Check if the bundle key exists.
    $bundle_key = $definition->getKey('bundle');
    if (!$bundle_key) {
      throw new \Exception('The entity type does not have a bundle key.');
    }

    // Get the bundle entity type.
    $bundle_entity_type = $definition->getBundleEntityType();

    $storage = $this->entityTypeManager->getStorage($bundle_entity_type);
    /** @var \Drupal\Core\Entity\ContentEntityInterface $bundle */
    $bundle = $storage->load($id);
    // Get the access control handler to check access.
    $access_control_handler = $this->entityTypeManager->getAccessControlHandler($bundle_entity_type);
    $method = 'update';
    if ($bundle) {
      // Check so the user has access to edit.
      if (!$access_control_handler->access($bundle, 'update', $this->currentUser)) {
        throw new \Exception('The current user does not have the right permissions to edit this bundle.');
      }
      // Set the label if it is not empty.
      if (!empty($label)) {
        $bundle->set('label', $label);
      }
    }
    // This means creation.
    else {
      if (!$access_control_handler->createAccess($id, $this->currentUser)) {
        throw new \Exception('The current user does not have the right permissions to create this bundle.');
      }
      $method = 'create';
      // Check so the user has create access.
      $bundle = $storage->create([
        $bundle_key => $id,
        'label' => $label,
      ]);
    }
    // @todo More specific logic needed.
    if ($bundle->save()) {
      $this->information = 'The bundle ' . $id . ' was ' . $method . ' on the entity type ' . $entity_type . '.';
    }
    else {
      throw new \Exception('Could not save the bundle.');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getReadableOutput(): string {
    return $this->information;
  }

}
