<?php

namespace Drupal\ai_agents\PluginBase;

use Drupal\Component\Plugin\Exception\ContextException;
use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Utility\Token;
use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\Tools\ToolsInput;
use Drupal\ai\OperationType\GenericType\ImageFile;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager;
use Drupal\ai_agents\AiAgentInterface;
use Drupal\ai_agents\Event\AgentResponseEvent;
use Drupal\ai_agents\Event\BuildSystemPromptEvent;
use Drupal\ai_agents\Output\StructuredResultData;
use Drupal\ai_agents\Output\StructuredResultDataInterface;
use Drupal\ai_agents\PluginInterfaces\AiAgentContextInterface;
use Drupal\ai_agents\PluginInterfaces\AiAgentFunctionInterface;
use Drupal\ai_agents\PluginInterfaces\AiAgentInterface as PluginInterfacesAiAgentInterface;
use Drupal\ai_agents\PluginInterfaces\ConfigAiAgentInterface;
use Drupal\ai_agents\Service\AgentHelper;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\Validator\ConstraintViolationInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * AI Agent Entity Wrapper.
 */
class AiAgentEntityWrapper implements PluginInterfacesAiAgentInterface, ConfigAiAgentInterface {

  /**
   * The AI Provider.
   *
   * @var \Drupal\ai\AiProviderInterface|\Drupal\ai\Plugin\ProviderProxy
   */
  protected $aiProvider;

  /**
   * The model.
   *
   * @var string
   */
  protected $modelName;

  /**
   * The AI configuration.
   *
   * @var array
   */
  protected $aiConfiguration;

  /**
   * The Task.
   *
   * @var \Drupal\ai_agents\Task\TaskInterface
   */
  protected $task;

  /**
   * Create directly.
   *
   * @var bool
   */
  protected $createDirectly;

  /**
   * The runner ID.
   *
   * @var string
   */
  protected $runnerId;

  /**
   * The tools used for action.
   *
   * @var array
   */
  protected $actionTools = [];

  /**
   * The tools used for context.
   *
   * @var array
   */
  protected $contextTools = [];

  /**
   * The question answered.
   *
   * @var string
   */
  protected $question;

  /**
   * Chat history.
   *
   * @var array
   */
  protected $chatHistory = [];

  /**
   * Tool results.
   *
   * @var array
   */
  protected $toolResults = [];

  /**
   * Amount of times looped.
   *
   * @var int
   */
  protected $looped = 0;

  /**
   * Looped enabled.
   *
   * @var bool
   */
  protected $loopedEnabled = TRUE;

  /**
   * The tokens.
   *
   * @var array
   */
  protected $tokens = [];

  /**
   * An overridden set of function definitions.
   *
   * @var array|null
   */
  private ?array $functionsOverride = NULL;

  /**
   * The constructor.
   *
   * @param \Drupal\ai_agents\AiAgentInterface $aiAgent
   *   The AI agent interface.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager $functionCallPluginManager
   *   The function call plugin manager.
   * @param \Drupal\ai_agents\Service\AgentHelper $agentHelper
   *   The agent helper.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   * @param \Drupal\ai\AiProviderPluginManager $aiProviderPluginManager
   *   The AI provider plugin manager.
   */
  public function __construct(
    protected AiAgentInterface $aiAgent,
    protected AccountInterface $currentUser,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected FunctionCallPluginManager $functionCallPluginManager,
    protected AgentHelper $agentHelper,
    protected Token $token,
    protected EventDispatcherInterface $eventDispatcher,
    protected AiProviderPluginManager $aiProviderPluginManager,
  ) {
  }

  /**
   * Get the AI agent entity.
   *
   * @return \Drupal\ai_agents\AiAgentInterface
   *   The AI agent interface.
   */
  public function getAiAgentEntity() {
    return $this->aiAgent;
  }

  /**
   * Set the AI agent interface.
   *
   * @param \Drupal\ai_agents\AiAgentInterface $aiAgent
   *   The AI agent interface.
   */
  public function setAiAgentEntity(AiAgentInterface $aiAgent) {
    $this->aiAgent = $aiAgent;
  }

  /**
   * {@inheritDoc}
   */
  public function setAiProvider($provider) {
    $this->aiProvider = $provider;
  }

  /**
   * {@inheritDoc}
   */
  public function getAiProvider() {
    return $this->aiProvider;
  }

  /**
   * {@inheritDoc}
   */
  public function getModelName() {
    return $this->modelName;
  }

  /**
   * {@inheritDoc}
   */
  public function setModelName($modelName) {
    $this->modelName = $modelName;
  }

  /**
   * {@inheritDoc}
   */
  public function getAiConfiguration() {
    return $this->aiConfiguration;
  }

  /**
   * {@inheritDoc}
   */
  public function setAiConfiguration($configuration) {
    $this->aiConfiguration = $configuration;
  }

  /**
   * {@inheritDoc}
   */
  public function getData() {
    return $this->contextTools;
  }

  /**
   * {@inheritDoc}
   */
  public function setData($data) {
    $this->contextTools = $data;
  }

  /**
   * {@inheritDoc}
   */
  public function getTask() {
    return $this->task;
  }

  /**
   * {@inheritDoc}
   */
  public function setTask($task) {
    $this->task = $task;
  }

  /**
   * {@inheritDoc}
   */
  public function getCreateDirectly() {
    return $this->createDirectly;
  }

  /**
   * {@inheritDoc}
   */
  public function setCreateDirectly($createDirectly) {
    $this->createDirectly = $createDirectly;
  }

  /**
   * Set if you want to loop.
   *
   * @param bool $enabled
   *   If you want to loop.
   */
  public function setLooped($enabled) {
    $this->loopedEnabled = $enabled;
  }

  /**
   * {@inheritDoc}
   */
  public function agentsCapabilities() {
    return [
      $this->aiAgent->id() => [
        'name' => $this->aiAgent->get('label'),
        'description' => $this->aiAgent->get('description'),
        'usage_instructions' => "",
        'inputs' => [
          'free_text' => [
            'name' => 'Prompt',
            'type' => 'string',
            'description' => 'The prompt with the instructions.',
            'default_value' => '',
          ],
        ],
        'outputs' => [
          'answers' => [
            'description' => 'The answers to the questions asked about.',
            'type' => 'string',
          ],
        ],
      ],
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function isAvailable() {
    return TRUE;
  }

  /**
   * {@inheritDoc}
   */
  public function getRunnerId() {
    return $this->runnerId;
  }

  /**
   * {@inheritDoc}
   */
  public function setRunnerId($runnerId) {
    $this->runnerId = $runnerId;
  }

  /**
   * {@inheritDoc}
   */
  public function getExtraTags() {
    return [];
  }

  /**
   * {@inheritDoc}
   */
  public function inform() {
    return '';
  }

  /**
   * {@inheritDoc}
   */
  public function determineSolvability() {
    // We need to set the default AI Provider if not set.
    if (!$this->aiProvider) {
      $defaults = $this->aiProviderPluginManager->getDefaultProviderForOperationType('chat_with_tools');
      $this->aiProvider = $this->aiProviderPluginManager->createInstance($defaults['provider_id']);
      $this->modelName = $defaults['model_id'];
    }
    $this->looped++;
    if ($this->looped > $this->aiAgent->get('max_loops')) {
      return PluginInterfacesAiAgentInterface::JOB_NOT_SOLVABLE;
    }
    // Get the system prompt.
    $system_prompt = $this->getSystemPrompt();
    // Check if someone wants to change something.
    $event = new BuildSystemPromptEvent($system_prompt, $this->aiAgent->id(), $this->tokens);
    $this->eventDispatcher->dispatch($event, BuildSystemPromptEvent::EVENT_NAME);
    // Set possible new values.
    $system_prompt = $event->getSystemPrompt();
    $this->tokens = $event->getTokens();
    // Run tokens to replace.
    $system_prompt = $this->applyTokens($system_prompt);
    $user_prompt = $this->agentHelper->getFullContextOfTask($this->task);

    $this->aiProvider->setChatSystemRole($system_prompt);

    $functions = $this->getFunctions();

    // We need to append the tool results if any.
    if (count($this->contextTools)) {
      foreach ($this->contextTools as $tool) {
        try {
          $this->executeTool($tool);
          $output = $tool->getReadableOutput();
        }
        catch (ContextException $exception) {
          $output = strip_tags($exception->getMessage());
        }
        $this->toolResults[] = $tool;
        // We need to check so its should not be returned.
        if ($this->toolShouldReturnDirectly($tool)) {
          $this->chatHistory[] = new ChatMessage('tool', $output);
          $this->question = $output;
          return PluginInterfacesAiAgentInterface::JOB_SOLVABLE;
        }
        $message = new ChatMessage('tool', $output);
        $message->setToolsId($tool->getToolsId());
        $this->chatHistory[] = $message;
      }
    }

    // Reset all tools between runs.
    $this->actionTools = [];
    $this->contextTools = [];

    $tags = [
      'ai_agents',
      'ai_agents_' . $this->aiAgent->id(),
      'ai_agents_prompt_' . $this->aiAgent->id(),
    ];
    if ($this->runnerId) {
      $tags[] = 'ai_agents_runner_' . $this->runnerId;
    }

    // Add the final message.
    if ($this->looped == 1) {
      // Get possible images.
      $images = [];
      foreach ($this->task->getFiles() as $file) {
        // Check if image.
        if (strpos($file->filemime->value, 'image') !== FALSE) {
          $image = new ImageFile();
          $image->setFileFromFile($file);
          $images[] = $image;
        }
        else {
          $user_prompt .= "\nThe uploaded file is: " . $file->getFilename() . " with file id: " . $file->id() . "\n";
        }
      }
      $this->chatHistory[] = new ChatMessage('user', $user_prompt, $images);
    }
    $input = new ChatInput($this->chatHistory);
    if (count($functions) && count($functions['normalized'])) {
      $input->setChatTools(new ToolsInput($functions['normalized']));
    }

    $return = $this->aiProvider->chat($input, $this->modelName, $tags);
    $response = $return->getNormalized();
    // Trigger the response event.
    $event = new AgentResponseEvent(
      $this,
      $system_prompt,
      $this->aiAgent->id(),
      $user_prompt,
      $this->chatHistory,
      $return,
      $this->looped,
    );

    $this->eventDispatcher->dispatch($event, AgentResponseEvent::EVENT_NAME);

    $this->chatHistory[] = $response;

    $tools = $response->getTools();

    if (!empty($tools)) {
      foreach ($tools as $tool) {
        $function = $this->functionCallPluginManager->convertToolResponseToObject($tool);
        if ($function instanceof AiAgentContextInterface) {
          $this->contextTools[] = $function;
        }
        elseif ($function instanceof ExecutableFunctionCallInterface) {
          // Check if its an agent tool and if it is a context tool.
          if ($function instanceof AiAgentFunctionInterface) {
            $agent = $function->getAgent();
            if ($agent instanceof AiAgentEntityWrapper && $agent->getAiAgentEntity()->get('context_agent')) {
              $this->contextTools[] = $function;
            }
            else {
              $this->contextTools[] = $function;
            }
          }
          else {
            $this->contextTools[] = $function;
          }
        }
      }
      // If tools are available, we should run this again filled out.
      if ($this->loopedEnabled) {
        return $this->determineSolvability();
      }
    }
    $this->question = $response->getText();
    return PluginInterfacesAiAgentInterface::JOB_SOLVABLE;
  }

  /**
   * {@inheritDoc}
   */
  public function hasAccess() {
    return TRUE;
  }

  /**
   * {@inheritDoc}
   */
  public function askQuestion() {
    return '';
  }

  /**
   * {@inheritDoc}
   */
  public function solve() {
    return $this->question;
  }

  /**
   * {@inheritDoc}
   */
  public function approveSolution() {
    $this->solve();
  }

  /**
   * {@inheritDoc}
   */
  public function getTokenContexts(): array {
    return $this->tokens;
  }

  /**
   * {@inheritDoc}
   */
  public function setTokenContexts(array $tokens): void {
    $this->tokens = $tokens;
  }

  /**
   * {@inheritDoc}
   */
  public function getStructuredOutput(): StructuredResultDataInterface {
    return new StructuredResultData();
  }

  /**
   * {@inheritDoc}
   */
  public function setUserInterface($userInterface, array $extraTags = []) {
  }

  /**
   * {@inheritDoc}
   */
  public function rollback() {
  }

  /**
   * {@inheritDoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritDoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritDoc}
   */
  public function getConfiguration() {
    return [];
  }

  /**
   * {@inheritDoc}
   */
  public function setConfiguration(array $configuration) {
  }

  /**
   * {@inheritDoc}
   */
  public function defaultConfiguration() {
    return [];
  }

  /**
   * {@inheritDoc}
   */
  public function getId() {
    return $this->aiAgent->id();
  }

  /**
   * {@inheritDoc}
   */
  public function getModuleName() {
    return 'ai_agent';
  }

  /**
   * {@inheritDoc}
   */
  public function agentsNames() {
    return [
      $this->aiAgent->get('label'),
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function answerQuestion() {
    return $this->question;
  }

  /**
   * Get all tool results.
   *
   * @return array
   *   The tool results.
   */
  public function getToolResults(): array {
    return $this->toolResults;
  }

  /**
   * Helper function to render the system prompt.
   *
   * @return string
   *   The system prompt.
   */
  public function getSystemPrompt() {
    $dynamic = $this->getDefaultInformationTools();
    $secured_system_prompt = $this->aiAgent->get('secured_system_prompt');
    // If its empty, we need to set the token.
    if (empty($secured_system_prompt)) {
      $secured_system_prompt = "[agent_instructions]";
    }
    // Apply the agent instructions token.
    $prompt = $this->applyTokens($secured_system_prompt);
    return $prompt . "\n\n" . $dynamic;
  }

  /**
   * Helper function for getting the default information.
   *
   * @return string
   *   The default information.
   */
  public function getDefaultInformationTools() {
    $tools_yaml = $this->applyTokens($this->aiAgent->get('default_information_tools') ?? '[]');
    $data = Yaml::parse($tools_yaml);
    $dynamic = "This is the ";
    if ($this->looped == 1) {
      $dynamic .= "first ";
    }
    elseif ($this->looped == 2) {
      $dynamic .= "second ";
    }
    elseif ($this->looped == 3) {
      $dynamic .= "third ";
    }
    else {
      $dynamic .= $this->looped . "th ";
    }
    $dynamic .= "time that this agent has been run. \n";

    if (isset($data)) {
      $dynamic .= "The following is information that is important as context: \n";
      foreach ($data as $values) {
        if (isset($values['available_on_loop']) && is_array($values['available_on_loop'])) {
          if (!in_array($this->looped, $values['available_on_loop'])) {
            continue;
          }
        }
        /** @var \Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface $tool */
        $tool = $this->functionCallPluginManager->createInstance($values['tool']);
        foreach ($values['parameters'] as $parameter_key => $parameter_value) {
          if ($parameter_value) {
            $tool->setContextValue($parameter_key, $parameter_value);
          }
        }
        $dynamic .= "-----------------------------------------------\n";
        $dynamic .= "Values: " . $values['label'] . "\n";
        if (!empty($values['description'])) {
          $dynamic .= "Description of values: " . $values['description'] . "\n";
        }
        $this->executeTool($tool);
        $dynamic .= $tool->getReadableOutput();
        $dynamic .= "\n";
      }
    }

    return $dynamic;
  }

  /**
   * Helper function for getting functions.
   *
   * @return array
   *   The functions.
   */
  public function getFunctions() {
    // Use overridden functions, if set.
    $function_definitions = $this->functionsOverride['tools'] ?? $this->aiAgent->get('tools');
    $usage_limits = $this->functionsOverride['tool_usage_limits'] ?? $this->aiAgent->get('tool_usage_limits');

    $functions = [];
    foreach ($function_definitions as $function_call_name => $value) {
      if ($value) {
        /** @var \Drupal\ai\Service\FunctionCalling\FunctionCallInterface $function_call */
        $function_call = $this->functionCallPluginManager->createInstance($function_call_name);
        $this->applyToolUsageLimitsToContext($function_call);
        $functions['normalized'][$function_call->getFunctionName()] = $function_call->normalize();
        // Check if we need to hide some property from the LLM.
        if ($usage_limits[$function_call->getPluginId()] ?? NULL) {
          foreach ($usage_limits[$function_call->getPluginId()] as $property_name => $limit) {
            if ($limit['action'] == 'force_value' && !empty($limit['hide_property'])) {
              // Unset the property if it is set to be hidden.
              $functions['normalized'][$function_call->getFunctionName()]->unsetProperty($property_name);
            }
          }
        }
        $functions['object'][$function_call->getFunctionName()] = $function_call;
      }
    }
    return $functions;
  }

  /**
   * Set function overrides.
   *
   * @param array{tools: array<string, bool>, tool_usage_limits: array<string, array<string, array{action: string, hide_property: bool, values: scalar[]}>>, tool_settings: array<string, array{return_directly: bool}>} $functions
   *   An array of function overrides.
   */
  public function overrideFunctions(array $functions): void {
    $this->functionsOverride = $functions;
  }

  /**
   * Reset function overrides.
   */
  public function resetFunctions(): void {
    $this->functionsOverride = NULL;
  }

  /**
   * Helper function for checking if a tool returns early.
   *
   * @return bool
   *   True if the tool should return early.
   */
  public function toolShouldReturnDirectly(ExecutableFunctionCallInterface $tool): bool {
    // Use overridden functions, if set.
    $settings = $this->functionsOverride['tool_settings'] ?? $this->aiAgent->get('tool_settings');

    if (isset($settings[$tool->getPluginId()]['return_directly'])) {
      return $settings[$tool->getPluginId()]['return_directly'];
    }
    return FALSE;
  }

  /**
   * Applies tool usage limits to the function schema.
   *
   * @param \Drupal\ai\Service\FunctionCalling\FunctionCallInterface $function_call
   *   The function call plugin.
   */
  protected function applyToolUsageLimitsToContext(FunctionCallInterface $function_call) {
    // Use overridden functions, if set.
    $tool_limits = $this->functionsOverride['tool_usage_limits'] ?? $this->aiAgent->get('tool_usage_limits');

    // Process each property with limits.
    foreach ($tool_limits[$function_call->getPluginId()] ?? [] as $property_name => $limit) {
      $context_definition = $function_call->getContextDefinition($property_name);

      // Apply token in values if an action is set.
      if ($limit['action']) {
        $values = array_map(
          fn ($value) => $this->applyTokens($value),
          array_filter(
            $limit['values'] ?? [],
            fn ($value) => $value !== NULL && $value !== '',
          ),
        );

        // Apply restrictions based on the action.
        switch ($limit['action']) {
          // Set constant value (forced value).
          case 'force_value':
            if (isset($values[0])) {
              $context_value = $context_definition->getDataType() === 'list' ? $values : $values[0];
              $context_definition->addConstraint('FixedValue', $context_value);
              $context_definition->setDefaultValue($context_value);
            }
            $context_definition->setRequired(FALSE);
            break;

          case 'only_allow':
            $context_definition->addConstraint('Choice', $values);
            break;
        }
      }
    }
  }

  /**
   * Apply the tokens to the system prompt.
   *
   * @param string $prompt
   *   The prompt to apply the tokens to.
   *
   * @return string
   *   The prompt with the tokens applied.
   */
  public function applyTokens(string $prompt): string {
    $tokens = [
      'user' => $this->currentUser,
      'ai_agent' => $this->aiAgent,
      'agent_instructions' => $this->aiAgent->get('system_prompt'),
    ];
    // Add dynamical tokens.
    $tokens = array_merge($tokens, $this->tokens);
    return $this->token->replace($prompt, $tokens);
  }

  /**
   * Helper function to execute a tool.
   *
   * @param \Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface $tool
   *   The tool to execute.
   */
  public function executeTool(ExecutableFunctionCallInterface $tool) {
    $this->validateTool($tool);
    $tool->execute();
  }

  /**
   * Validate the tool against any possible restrictions before running.
   *
   * @param \Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface $tool
   *   The tool to validate values from.
   *
   * @throws \Drupal\Component\Plugin\Exception\ContextException
   *   Thrown when context constraints are violated.
   */
  public function validateTool(ExecutableFunctionCallInterface $tool): void {
    $violations = $tool->validateContexts();
    if (count($violations)) {
      throw new ContextException(implode("\n", array_map(
        fn (ConstraintViolationInterface $violation) => new FormattableMarkup('Invalid value for @property in @function: @violation', [
          // @todo Consider using a property name when the context validator is
          //   fixed.
          // @see https://www.drupal.org/project/drupal/issues/3153847
          '@property' => $violation->getRoot()->getDataDefinition()->getLabel(),
          '@function' => $tool->getPluginId(),
          '@violation' => $violation->getMessage(),
        ]),
        (array) $violations->getIterator(),
      )));
    }
  }

  /**
   * Change user permissions temporarily if needed.
   */
  public function changeUserPermissions() {
    // Always reload the user to make sure no one has intercepted the user.
    /** @var \Drupal\user\Entity\User $user */
    $user = $this->entityTypeManager->getStorage('user')->load($this->currentUser->id());
    $exclude_users_role = $this->aiAgent->get('exclude_users_role');
    // If we should remove all the roles.
    if ($exclude_users_role) {
      /** @var \Drupal\user\Entity\User $user */
      $user = $this->currentUser;
      foreach ($user->getRoles() as $role) {
        if ($role != 'anonymous') {
          $user->removeRole($role);
        }
      }
    }
    // If we should add masquerade roles.
    $masquerade_roles = $this->aiAgent->get('masquerade_roles');
    if ($masquerade_roles) {
      /** @var \Drupal\user\Entity\User $user */
      $user = $this->currentUser;
      foreach ($masquerade_roles as $role) {
        $user->addRole($role);
      }
    }
    // Set the user as the current user.
    $this->currentUser->setAccount($user);
  }

  /**
   * Reset the user permissions after solving a tool.
   */
  public function resetUserPermissions() {
    // Reload the user from entity.
    /** @var \Drupal\user\Entity\User $user */
    $user = $this->entityTypeManager->getStorage('user')->load($this->currentUser->id());
    // Set the user as the current user.
    $this->currentUser->setAccount($user);
  }

  /**
   * Returns the chat history of the agent.
   *
   * @return array
   *   An array of chat messages and tool results.
   */
  public function getChatHistory(): array {
    return $this->chatHistory;
  }

  /**
   * Sets the chat history of the agent.
   *
   * @param array $history
   *   An array of chat messages and tool results to restore.
   */
  public function setChatHistory(array $history): void {
    $this->chatHistory = $history;
  }

}
